/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2004
 *
 *  Last modified:
 *     $Date: 2009-09-08 21:10:29 +0200 (Tue, 08 Sep 2009) $ by $Author: schulte $
 *     $Revision: 9692 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

namespace Gecode { namespace Iter { namespace Ranges {

  /**
   * \brief %Range iterator cache
   *
   * Allows to iterate the ranges as defined by the input iterator
   * several times provided the Cache is %reset by the reset member
   * function.
   *
   * \ingroup FuncIterRanges
   */
  template<class I>
  class Cache {
  private:
    /// Check that \a I is a range iterator;
    IsRangeIter<I> _checkI;
  protected:
    /// %Ranges stored in cache
    class Range {
    public:
      int min; int max;
    };
    /// Array for ranges
    SharedArray<Range> r;
    /// Current range
    int c;
  public:
    /// \name Constructors and initialization
    //@{
    /// Default constructor
    Cache(void);
    /// Initialize with ranges from \a i
    Cache(I& i);
    /// Initialize with ranges from \a i
    void init(I& i);
    //@}

    /// \name Iteration control
    //@{
    /// Test whether iterator is still at a range or done
    bool operator ()(void) const;
    /// Move iterator to next range (if possible)
    void operator ++(void);
    /// Reset iterator to start from beginning
    void reset(void);
    //@}

    /// \name %Range access
    //@{
    /// Return smallest value of range
    int min(void) const;
    /// Return largest value of range
    int max(void) const;
    /// Return width of range (distance between minimum and maximum)
    unsigned int width(void) const;
    //@}
  };


  template<class I>
  forceinline
  Cache<I>::Cache(void) {}

  template<class I>
  inline void
  Cache<I>::init(I& i) {
    Support::DynamicArray<Range,Heap> d(heap);
    int n=0;
    while (i()) {
      d[n].min = i.min(); d[n].max = i.max();
      ++n; ++i;
    }
    r.init(n);
    for (int j=n; j--; )
      r[j]=d[j];
    c = 0;
  }

  template<class I>
  inline
  Cache<I>::Cache(I& i) {
    init(i);
  }

  template<class I>
  forceinline void
  Cache<I>::operator ++(void) {
    c++;
  }
  template<class I>
  forceinline bool
  Cache<I>::operator ()(void) const {
    return c < r.size();
  }

  template<class I>
  forceinline void
  Cache<I>::reset(void) {
    c = 0;
  }

  template<class I>
  forceinline int
  Cache<I>::min(void) const {
    return r[c].min;
  }
  template<class I>
  forceinline int
  Cache<I>::max(void) const {
    return r[c].max;
  }
  template<class I>
  forceinline unsigned int
  Cache<I>::width(void) const {
    return static_cast<unsigned int>(r[c].max-r[c].min+1);
  }

}}}

// STATISTICS: iter-any

