// ---------------------------------------------------------------------
// $Id: CmdLineParser.cc,v 1.8 2007/03/28 09:47:10 daaugusto Exp $
//
//   CmdLineParser.cc (created on Tue Aug  8 11:01:58 BRT 2006)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2007 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// ---------------------------------------------------------------------

#include <cstring>
#include <iostream>

#include "CmdLineParser.hh"

// ---------------------------------------------------------------------
bool 
OptionsINT::Match(int& i, char** argv, int argc)
{
   // try to find
   Option<INT>* p = Find(argv[i]);

   if (p) // found!
   {
      // try to convert the option arg to 'int'
      if (i+1<argc)
      {
         istringstream s(argv[i+1]); INT tmp;

         if (!(s >> std::dec >> tmp).fail()) // ok, successful conversion 
         {
            // do not accept if the input type is 'float'
            if (string(argv[i+1]).find('.') != string::npos) return false;

            p->m_found = true;

            if (tmp < p->m_min || tmp > p->m_max)
            {
               cerr << "> [Ignoring] Option '" << argv[i] 
                    << "': value out of range." << endl;

            } else p->m_value = tmp;

            // if used then "eat the argument"
            ++i;

            return true;
         }
      }
   }
   return false;
}

// ---------------------------------------------------------------------
bool 
OptionsFLOAT::Match(int& i, char** argv, int argc)
{
   // try to find
   Option<FLOAT>* p = Find(argv[i]);

   if (p) // found!
   {
      // try to convert the option arg to 'FLOAT'
      if (i+1<argc)
      {
         istringstream s(argv[i+1]); FLOAT tmp;

         if (!(s >> std::dec >> tmp).fail()) // ok, successful conversion
         {
            p->m_found = true;

            if (tmp < p->m_min || tmp > p->m_max)
            {
               cerr << "> [Ignoring] Option '" << argv[i] 
                    << "': value out of range." << endl;

            } else p->m_value = tmp;

            // if used then "eat the argument"
            ++i;

            return true;
         }
      }
   }
   return false;
}

// ---------------------------------------------------------------------
bool 
OptionsBOOL::Match(int& i, char** argv, int argc)
{
   /* A minus sign ("-") after the option name disable the previous one,
      i.e.: -d- disables -d if -d was already declared. */
   string tmp(argv[i]); bool disable = false;

   if (tmp.size()>1 && tmp[tmp.size()-1]=='-') 
   { 
      disable = true; tmp.erase(tmp.end()-1);
   }

   Option<bool>* p = Find(tmp);

   if (p) // found!
   {
      p->m_found = true;
      p->m_value = disable ? false : true;

      return true;
   }
   return false;
}

// ---------------------------------------------------------------------
bool 
OptionsCHAR::Match(int& i, char** argv, int argc)
{
   Option<char>* p = Find(argv[i]);

   if (p) // found!
   {
      if (i+1<argc)
      {
         // accept only if has one character or two (if the first is
         // a backslash)
         if (argv[i+1][0] == '\0' || argv[i+1][1] == '\0')
         {
            p->m_value = argv[i+1][0];
         }
         else if (argv[i+1][0] == '\\' && argv[i+1][2] == '\0')
         {
            switch (argv[i+1][1])
            {
               case '0' : p->m_value = '\0'; break;
               case 'a' : p->m_value = '\a'; break;
               case 'b' : p->m_value = '\b'; break;
               case 'f' : p->m_value = '\f'; break;
               case 'n' : p->m_value = '\n'; break;
               case 'r' : p->m_value = '\r'; break;
               case 't' : p->m_value = '\t'; break;
               case 'v' : p->m_value = '\v'; break;
               case '\\': p->m_value = '\\'; break;
               default  : p->m_value = argv[i+1][1]; break;
            }
         } else return false;

         p->m_found = true;

         // if used then "eat the argument"
         ++i;

         return true;
      }
   }
   return false;
}

// ---------------------------------------------------------------------
bool 
OptionsSTRING::Match(int& i, char** argv, int argc)
{
   // try to find
   Option<string>* p = Find(argv[i]);

   if (p) // found!
   {
      if (i+1<argc)
      {
         p->m_found = true;
         p->m_value = string(argv[i+1]);

         // if used then "eat the argument"
         ++i;

         return true;
      }
   }
   return false;
}
// ---------------------------------------------------------------------
void 
CmdLineParser::Process()
{
   int i=1;
   while (i<m_argc)
   {
      bool match = false;

      if (m_argv[i][0] != '-') match = false; // not an option, skip
      else
      {
         // "--" stops the option search/processing
         if (!strcmp(m_argv[i],"--")) break;

         match = Int.Match(i, m_argv, m_argc) || Float.Match(i, m_argv, m_argc) 
             || Bool.Match(i, m_argv, m_argc) || Char.Match(i, m_argv, m_argc)
             || String.Match(i, m_argv, m_argc);

      }

      if (!match) m_remains.push_back(m_argv[i]); // not a valid option
      ++i;
   }

   // only if encountered "--": do not process the remaining
   while (++i<m_argc) { m_remains.push_back(m_argv[i]); }
}

// ---------------------------------------------------------------------
