/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <unistd.h>
#include "gettext.h"
#include "widgets.h"
#include "allocate.h"
#include "show_info.h"
#include "functions.h"
#include "commented.h"

extern int MAX_READ_POPEN;
extern int MAX_CONF_LINE;
extern char DHCPD_CONF_BUF[1024];


void run_command_show_err(gchar *command)
{
    FILE *fp;
    char *line, *info;
    int once=0;
    
    if((fp=popen(command, "r"))==NULL)
    {
	perror("popen");
	return;
    }
    else
      {
	  line = allocate(MAX_READ_POPEN+2);
	  info = allocate(MAX_READ_POPEN+2);
	  while(fgets(line, MAX_READ_POPEN, fp)!=NULL)
	  {
	     strcat(info, line);
	     /* Show the error line a bit more clearly */
	     if( strstr(line, "visit") && ! once )
	     {
	        once = 1;
		strcat(info, "\n");
	     }
	  }
	  pclose(fp);
	  show_info(info);
	  free(info);
	  free(line);
      }
}

int run_command(gchar *command)
{
    FILE *fp;
    int status=0, exit_status=0;
    
    if((fp=popen(command, "w"))==NULL)
    {
	perror("popen");
	return 0;
    }
    status = pclose(fp);

    exit_status = WEXITSTATUS(status);
    
    if( exit_status == 1 )
      exit_status = 0;
    else
      exit_status = 1;

    return exit_status;
}

void init_start(struct w *widgets)
{
    gchar *cmd;
    cmd = g_strdup_printf("%s", SYSINIT_START_CMD);

    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	}
    }
    if( cmd!=NULL )
      g_free(cmd);
}

void init_stop(struct w *widgets)
{
    gchar *cmd;
    cmd = g_strdup_printf("%s", SYSINIT_STOP_CMD);

    if( strlen(cmd) > 4 )
    {
	if( ! run_command(cmd) )
	{
	    run_command_show_err(cmd);
	}
    }
    if( cmd!=NULL )
      g_free(cmd);
}

int file_exists(char *infile)
{
    FILE *fp;
    if((fp=fopen(infile, "r"))==NULL)
      return 0;

    fclose(fp);
    return 1;
}

/* Get all comments for a scope. */
char * get_comments_for_scope(char *subnet, char *netmask, char *nic)
{
    FILE *fp;
    long saved_pos=0, conf_size = 0;
    char *tmp, *line;
    int i=0, found_scope = 0;
    int scope_begin = 0, scope_end = 0;
    char *comments = NULL;

    /* 3 different scope styles ...\n{ or ...{ or ... { */
    gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", subnet, netmask);
    gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", subnet, netmask);
    gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", subnet, netmask);
    gchar *nic_line = g_strdup_printf("interface %s;\n", nic);

    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        return NULL;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    tmp  = allocate(MAX_CONF_LINE+1);
    comments = allocate(16384); /* Max size for comments. */

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( commented(line) || strlen(line) < 3 || strlen(line) > MAX_CONF_LINE )
	  continue;

	if( strstr(line, scope_line1) || strstr(line, scope_line2) || strstr(line, scope_line3) )
	{
    	    saved_pos = ftell(fp);
	    	        
	    /* We expect to find the interface decl on the second line */
	    while(fgets(line, conf_size, fp)!=NULL)
	    {
		if( commented(line) )
	          continue;

		if( strstr(line, nic_line) )
		{
	    	    found_scope = 1;
	    	    break;
		}
	    }
	}
	if( found_scope )
          break;
    }
    g_free(scope_line1);
    g_free(scope_line2);
    g_free(scope_line3);
    g_free(nic_line);

    /* Scroll back to the line before the scope line. */
    fseek(fp, saved_pos-strlen(line), SEEK_SET);

    /* Get all comments */
    if( found_scope && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( strlen(line) > MAX_CONF_LINE )
	  continue;

	/* Break at scope end */
	if( ! commented(line) && strstr(line, "{") )
	  scope_begin++;
	if( ! commented(line) && strstr(line, "}") )
	  scope_end++;
	if( scope_begin == scope_end )
	  break;

	/* This line contains a comment. */
	if( strstr(line, "#") )
	{
	    remove_semicolon(line);

	    /* Scroll to the beginning of the comment. */
	    for(i=0; i<strlen(line); i++)
	    {
		if( line[i]=='#' )
		  break;
	    }
	    /* Scroll past some stuff in the beginning of the comment. */
	    for(i=i+1; i<strlen(line); i++)
	    {
		if( line[i]!='#' && line[i]!=' ' && line[i]!='\t' )
		  break;
	    }
	    
	    snprintf(tmp, MAX_CONF_LINE-1, "%s", &line[i]);

	    /* Max comment size to gather. */
	    if( strlen(comments) + strlen(line) > 16000 )
	      break;	    	    
	
	    strcat(comments, tmp);

	    /* Add a newline char if missing. */
	    if( tmp[strlen(tmp)-1]!='\n' )
	      strcat(comments, "\n");
	}
    }
    fclose(fp);
    free(line);
    free(tmp);

    return comments;
}


/* Get all comments for a host. */
char * get_comments_for_host(char *host_name)
{
    FILE *fp;
    long saved_pos=0, conf_size = 0;
    char *tmp, *line;
    int i=0, found_host = 0;
    int host_begin = 0, host_end = 0;
    char *comments = NULL;

    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        return NULL;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    tmp  = allocate(MAX_CONF_LINE+1);
    comments = allocate(16384); /* Max size for comments. */

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( commented(line) || strlen(line) < 3 || strlen(line) > MAX_CONF_LINE )
	  continue;

	if( is_selected_host(host_name, line) )
	{
	    found_host = 1;
	    saved_pos = ftell(fp);
	    break;
	}
    }

    /* Scroll back to the line before the host line. */
    fseek(fp, saved_pos-strlen(line), SEEK_SET);

    /* Get all comments */
    if( found_host && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( strlen(line) > MAX_CONF_LINE )
	  continue;

	/* Break at host end */
	if( ! commented(line) && strstr(line, "{") )
	  host_begin++;
	if( ! commented(line) && strstr(line, "}") )
	  host_end++;
	if( host_begin == host_end )
	  break;

	/* This line contains a comment. */
	if( strstr(line, "#") )
	{
	    remove_semicolon(line);

	    /* Scroll to the beginning of the comment. */
	    for(i=0; i<strlen(line); i++)
	    {
		if( line[i]=='#' )
		  break;
	    }
	    /* Scroll past some stuff in the beginning of the comment. */
	    for(i=i+1; i<strlen(line); i++)
	    {
		if( line[i]!='#' && line[i]!=' ' && line[i]!='\t' )
		  break;
	    }
	    
	    snprintf(tmp, MAX_CONF_LINE-1, "%s", &line[i]);

	    /* Max comment size to gather. */
	    if( strlen(comments) + strlen(line) > 16000 )
	      break;	    	    
	
	    strcat(comments, tmp);

	    /* Add a newline char if missing. */
	    if( tmp[strlen(tmp)-1]!='\n' )
	      strcat(comments, "\n");
	}
    }
    fclose(fp);
    free(line);
    free(tmp);

    return comments;
}


/* Checks if this line contains the host we are looking for. */
int is_selected_host(char *host_name, char *conf_line)
{
    int i = 0, retval = 0;
    char *tmp;

    tmp = allocate(MAX_CONF_LINE+1);

    /* Locate "host " as the first thing written on a line. */
    for(i=0; i<strlen(conf_line); i++)
    {
        /* Break at the first char found. */
        if( conf_line[i]!=' ' && conf_line[i]!='\t' )
          break; 
    }
    snprintf(tmp, MAX_CONF_LINE, "%s", &conf_line[i]);

    for(i=0; i<strlen(tmp); i++)
    {
        /* Break at the first space found. */
        if( tmp[i]==' ' || tmp[i]=='\t' )
          break; 
    }
    tmp[i]=' ';
    tmp[i+1]='\0';

    if( ! strcmp(tmp, "host ") == 0 )
    {
    	free(tmp);
      	return 0;
    }

    /* Locate "HostName" matching the host_name. */
    for(i=0; i<strlen(conf_line); i++)
    {
        /* Break at the first char found. */
        if( conf_line[i]!=' ' && conf_line[i]!='\t' )
          break; 
    }
    for(i=i; i<strlen(conf_line); i++)
    {
        /* Break at the first space found. */
        if( conf_line[i]==' ' || conf_line[i]=='\t' )
          break; 
    }
    for(i=i; i<strlen(conf_line); i++)
    {
        /* Break at the first char found. */
        if( conf_line[i]!=' ' && conf_line[i]!='\t' )
          break; 
    }
    snprintf(tmp, MAX_CONF_LINE, "%s", &conf_line[i]);

    for(i=0; i<strlen(tmp); i++)
    {
        /* Cut anything after the hostname. */
        if( tmp[i]==' ' || tmp[i]=='\t' || tmp[i]=='{' )
          break; 
    }
    tmp[i]='\0';

    if( strcmp(tmp, host_name) == 0 )	
    {
	 retval = 1;
    }

    free(tmp);

    return retval;
}


char * remove_semicolon(char *input)
{
    int y;
    gchar *info;
    if( strlen(input)>1000 )
    {
        info = g_strdup_printf(_("This settings value is too long:\n%s\n"), input);
        show_info(info);
        g_free(info);
	return input; /* Itll never'tm get here */
    }
    /* Delete any ; (semicolon) and everything behind it */
    for(y=0; y<strlen(input)-1; y++)
    if( input[y]==';' )
        input[y]='\0';
    
    return input;
}


char * isolate_setting(char *input)
{
    int y, z, begin=0, end=0, isolated=0;
    char *isol;
    gchar *info;
    
    if( strlen(input) > MAX_CONF_LINE )
    {
        info = g_strdup_printf(_("This settings value is too long:\n%s\n"), input);
        show_info(info);
        g_free(info);
	return input;
    }

    isol = allocate(strlen(input)+3);
    
    /* If it has a " char get everything in between */    
    for(y=0; y<strlen(input)-1; y++)
    if( input[y]=='"' )
    {
        begin=y+1;

        for(z=strlen(input)-1; z>y; z--)
	if( input[z]=='"' )
	{
	    end=z-1; isolated=1;
	    break;
	}
    
	if( isolated )
	  break;
    }

    if( isolated )
    {
	strncat(isol, &input[begin], end-begin+1);
	/* This is a shorter string */
	sprintf(input, "%s", isol);
    }

    free(isol);

    return input;
}


void add_missing_values(gchar *conf)
{
    /* Adds missing values to dhcpd.conf.
       If dhcpd.conf is missing itll create it and add the values. */

    FILE *fp;
    long conf_size;
    char *line, *orig_conf;
    int ddns_update_style = 0;
    int ddns_updates   = 0;
    int opt_t150       = 0;
    int client_updates = 0;
    int one_lease_per_client = 0;
    int bootp   = 0;
    int booting = 0;
    int unknown_clients = 0;
    int authoritative   = 0;
    int ip_forwarding   = 0;
    int mask_supplier   = 0;
    int default_lease_time = 0;
    
    if( strlen(conf) < 10 )
    {
	printf("Add missing values: conf path too short.\n");
	return;
    }

    if( file_exists(conf) )
    {
	if((fp=fopen(conf, "r"))==NULL)
	{
    	    printf("Error opening: %s\n", conf);  
    	    return;
	}
    }
    else
      {
	  if((fp=fopen(conf, "w+"))==NULL)
	  {
    	      printf("Error opening: %s\n", conf);  
    	      return;
	  }
      }

    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);
    
    line = allocate(conf_size+1);
    orig_conf = allocate(conf_size+1);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Collect all current global values */
	strcat(orig_conf, line);

	/* Break after the global declaration section */
	if( ! commented(line) && strstr(line, "{") )
	{
	    break;
	}
	
	if( strstr(line, "ddns-update-style") )
	    ddns_update_style = 1;

	if( strstr(line, "ddns-updates") )
	    ddns_updates = 1;

	if( strstr(line, "option T150 code 150 = string;") )
	    opt_t150 = 1;

	if( strstr(line, "client-updates;") )
	    client_updates = 1;

	if( strstr(line, "one-lease-per-client") )
	    one_lease_per_client = 1;

	if( strstr(line, "bootp;") )
	    bootp = 1;

	if( strstr(line, "booting;") )
	    booting = 1;
	
	if( strstr(line, "unknown-clients;") )
	    unknown_clients = 1;

	if( strstr(line, "authoritative;") )
	    authoritative = 1;

	if( strstr(line, "option ip-forwarding") )
	    ip_forwarding = 1;

	if( strstr(line, "option mask-supplier") )
	    mask_supplier = 1;

	if( strstr(line, "default-lease-time") )
	    default_lease_time = 1;
    }

    /* Collect the rest of the subnet and host declarations */
    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	strcat(orig_conf, line);
    }
    fclose(fp);
    free(line);

    /* We have everything required, return */
    if( ddns_update_style && ddns_updates && opt_t150
    &&  client_updates && one_lease_per_client
    &&  bootp && booting && unknown_clients
    &&  authoritative && ip_forwarding
    &&  mask_supplier &&  default_lease_time )
    {
	free(orig_conf);
        return;
    }

    /* Add the missing global options, subnets and hosts */      
    if((fp=fopen(conf, "w+"))==NULL)
    {
        printf("Error opening: %s\n", conf);  
	free(orig_conf);
        return;
    }

    if( ! ddns_update_style )
      fputs("ddns-update-style none;\n", fp);

    if( ! ddns_updates )
      fputs("ddns-updates off;\n", fp);

    if( ! opt_t150 )
      fputs("option T150 code 150 = string;\n", fp);

    if( ! client_updates )
      fputs("deny client-updates;\n", fp);

    if( ! one_lease_per_client )
      fputs("one-lease-per-client false;\n", fp);

    if( ! bootp )
      fputs("allow bootp;\n", fp);

    if( ! booting )
      fputs("allow booting;\n", fp);

    if( ! unknown_clients )
      fputs("allow unknown-clients;\n", fp);

    if( ! authoritative )
      fputs("not authoritative;\n", fp);

    if( ! ip_forwarding )
      fputs("option ip-forwarding false;\n", fp);

    if( ! mask_supplier )
      fputs("option mask-supplier false;\n", fp);
	
    if( ! default_lease_time )
      fputs("default-lease-time 604800;\n", fp);

    fputs(orig_conf, fp);

    fclose(fp);
    free(orig_conf);
}
