/*******************************************************************************
*	fische - standalone sound visualisation for linux		       *
*	Copyright (C) 2006 Marcel Ebmer					       *
*									       *
*	This program is free software; you can redistribute it and/or	       *
*	modify it under the terms of the GNU General Public License	       *
*	as published by the Free Software Foundation; either version 2	       *
*	of the License, or (at your option) any later version.		       *
*									       *
*	This program is distributed in the hope that it will be useful,	       *
*	but WITHOUT ANY WARRANTY; without even the implied warranty of	       *
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       *
*	GNU General Public License for more details.			       *
*									       *
*	You should have received a copy of the GNU General Public License      *
*	along with this program; if not, write to the Free Software	       *
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 	       *
*	MA  02110-1301, USA.						       *
*******************************************************************************/
#include "fische.h"

/* main ( ) caught SIGINT. We want to exit cleanly */
void sigint_caught_MAIN ( ) {
	kill ( ALSAChildPID, SIGINT );
	printf ( "*** main: Caught SIGINT, exiting. ***\n" );
	exit ( EXIT_SUCCESS );
}

/* the audio sampling thread died for some reason. Exit cleanly */
void child_died_MAIN ( ) {
 	printf ( "*** main: Child died, exiting. ***\n" );
	exit ( EXIT_FAILURE );
}

int main ( int argc, char **argv ) {

	int pipe_fd[2];		// the pipe for the sound samples
	int bytes_read;		// how many bytes were read?
	int bogus_count;	// how many are left?
	char *bogus_bytes;	// buffer for garbage sound samples
	char *data_in;		// buffer for sound samples
	char paused = 0;	// is the display paused?
	int events = 0;		// holds one bit for every possible event ( keys, etc. )
	int overruns = 0;	// counts buffer overruns
	int underruns = 0;	// counts buffer underruns
	int analysed;		// sound analysis data
	char do_blur = 1;	// is blurring switched on?
	char mark_beats = 1;	// do we mark beats additionally?

	#ifdef BENCHMARK
		long int ticks_1;
	#endif

	srand ( time ( NULL ) );	// rand ( ) needs initialisation

	XRes = 800;
	YRes = 400;

	if ( ! parse_options ( argc - 1, argv ) ) exit ( EXIT_FAILURE );

	samplelength = XRes * 4;	// length of the sound samples ( 16 bit stereo = 4 bytes per frame )
	surfacepitch = XRes * 4;	// width of the screen in memory ( 4 bytes per pixel )

	vector_fieldsize = XRes * YRes * 2;

	signal ( SIGCHLD, child_died_MAIN );
	signal ( SIGINT, sigint_caught_MAIN );

	if ( argc < 2 ) usage ( );	// FIXME: this needs to be a more precise test for correct syntax

	pipe ( pipe_fd );		// open a pipe for the sound data
	
	ALSAChildPID = fork ( );

	if ( ! ALSAChildPID ) {
		/* I am the child */
		close ( pipe_fd[0] );	// only one pipe end (for writing) is needed
		sample_alsa_pcm ( argv[argc-1], pipe_fd[1], samplelength, 44100 );
	}

	if ( ALSAChildPID ) {
		/* I am the parent */

		if ( ! ( vectors = malloc ( vector_fieldsize ) ) ) {
			printf ( "EE main: out of memory\n" );
			exit ( EXIT_FAILURE );
		}
		if ( ! ( data_in = malloc ( samplelength ) ) ) {
			printf ( "EE main: out of memory\n" );
			exit ( EXIT_FAILURE );
		}
		if ( ! ( bogus_bytes = malloc ( samplelength * 4 ) ) ) {
			printf ( "EE main: out of memory\n" );
			exit ( EXIT_FAILURE );
		}
		if ( ! ( blur_dest = malloc ( YRes * XRes * 4 ) ) ) {
			printf ( "EE main: out of memory\n" );
			exit ( EXIT_FAILURE );
		}
		fprintf ( stdout, "II main: allocated memory\n" );

		create_vectors ( );		// vectors are created once and stored in memory

		close ( pipe_fd[1] );		// only the reading end is needed
		fcntl ( pipe_fd[0], F_SETFL, O_NONBLOCK );	// required for "realtime" functionality
		
		color1 = rand ( ) % 0xFFFFFF * 0x100 + 0xff;

		current.vectors = rand ( ) % 7;
		current.wave = rand ( ) % 5;
		current.beatmark = rand ( ) % 6;

		freeze.vectors = 0;
		freeze.color = 0;
		freeze.waveform = 0;

		init_SDL ( XRes, YRes, 0 );
		while ( ! ( events & EXIT ) ) {				// EXIT bit is not set
			events = check_events ( );
			/* check for all bits in events */
			if ( events & PAUSE ) toggle ( &paused );
			if ( events & TOGGLE_BLUR ) toggle ( &do_blur );
			if ( events & TOGGLE_VECTORFREEZE ) toggle ( &freeze.vectors );
			if ( events & TOGGLE_WAVEFREEZE ) toggle ( &freeze.waveform );
			if ( events & TOGGLE_COLORFREEZE ) toggle ( &freeze.color );
			if ( events & TOGGLE_BEATMARK ) toggle ( &mark_beats );
			
			/* now we get the sound data from the pipe */
			bytes_read = read ( pipe_fd[0], data_in, samplelength );
			
			/* test for empty result and wait for data if so */
			if ( bytes_read <= 0 ) { 
				usleep ( 1000 );
				underruns++;
			} else {
				/* read remaining sound data - drain the pipe */
				bogus_count = read ( pipe_fd[0], bogus_bytes, samplelength * 4 );
				if ( bogus_count > 0 ) {
					overruns++;
					/* FIXME - broken
					if ( overruns * 1000 / SDL_GetTicks ( ) > 1 ) fprintf ( stderr, "WW main: too many audio buffer overruns!\n   ( reducing resolution might help! )\n" );
					*/
				}
				if ( paused ) {
					usleep ( 10000 );
				} else {

#ifdef BENCHMARK			/* BENCHMARKING */
					ticks_1 = SDL_GetTicks ( );
#endif

					/* analyse sound data, look for beats and other significant changes */
					analysed = analyse ( data_in, samplelength );
					/* PHASE bit changes vectors OR waveform */
					if ( analysed & PHASE ) {
						if ( do_blur ) {
							if ( ( rand ( ) % 2 ) && ( ! freeze.vectors ) ) change ( &current.vectors, 6 );
							else if ( ! freeze.waveform ) change ( &current.wave, 4 );
						} else change ( &current.wave ,4 );
					}
					/* LINE bit changes color AND type of beatmark */
					if ( analysed & LINE ) {
						if ( ! freeze.color ) color1 = rand ( ) % 0xFFFFFF * 0x100 + 0xff;
						change ( &current.beatmark, 5 );
					}
					
#ifdef BENCHMARK			/* BENCHMARKING */
					fprintf ( stdout, "a%ld - ", SDL_GetTicks ( ) - ticks_1 );
					ticks_1 = SDL_GetTicks ( );
#endif

					/* blur the screen along the vector field */
					if ( do_blur ) blur ( current.vectors );
					/* unless blur is switched off */
					else bzero ( sdl_screen->pixels, XRes * YRes * 4 );

#ifdef BENCHMARK			/* BENCHMARKING */
					fprintf ( stdout, "b%ld - ", SDL_GetTicks ( ) - ticks_1 );
					ticks_1 = SDL_GetTicks ( );
#endif

					/* draw lines, rings or similar to mark the beat */
					if ( analysed & BEAT ) {
						if ( mark_beats ) beatmark ( current.beatmark );
					}
					/* draw the sound data */
					wave ( current.wave, data_in );

#ifdef BENCHMARK			/* BENCHMARKING */
					fprintf ( stdout, "d%ld - ", SDL_GetTicks ( ) - ticks_1 );
					ticks_1 = SDL_GetTicks ( );
#endif

					/* update the graphics buffer */
					SDL_UpdateRect ( sdl_screen, 0, 0, 0, 0 );

#ifdef BENCHMARK			/* BENCHMARKING */
					fprintf ( stdout, "u%ld\n", SDL_GetTicks ( ) - ticks_1 );
					ticks_1 = SDL_GetTicks ( );
#endif
				}
			}
		}
		fprintf ( stdout, "II main: freeing memory\n" );
		fprintf ( stdout, "II main: audio buffer: underruns: %u (%0.2f/s), overruns: %u (%0.2f/s)\n", underruns, (double)underruns / (double)SDL_GetTicks ( ) * 1000.0, overruns, (double)overruns / (double)SDL_GetTicks ( ) * 1000.0 );
		SDL_Quit ( );
		free ( data_in );
		free ( bogus_bytes );
		free ( vectors );
		free ( blur_dest );
		signal ( SIGCHLD, SIG_IGN );	// from now on dieing childs will be ignored
		kill ( ALSAChildPID, SIGINT );	// here's why.
	}
	exit ( EXIT_SUCCESS );
}
