<?
	/**
	 * Image functions for use with the Human Theme for Firefox
	 * creation script.
	 */

	require_once "filesystemfunctions.php";

	// custom image definitions
	DEFINE(NOIMAGE, 'empty');

	/**
	 * Get a filename pointing to a PNG version of the requested icon.
	 *
	 * This method will always return a PNG file which exists, and will
	 * try to render scalable SVG versions of the icon if no pre-rendered
	 * one is available.
	 */
	function getPngFilenameFor($category, $icon, $size) {
		// get the preferred source file
		$preferred = getIconFilenameFor($category, $icon, $size);
		
		// render a PNG version if necessary
		if(isSvg($preferred)) {
			return renderPng($preferred, $size);
		} else {
			// check for correct size
			$fileinfo = system('file -L '.$preferred);
			if(!strpos($fileinfo, $size.' x '.$size)) {
				// need to create larger, new image
				return resizePng($preferred, $size);
			} else {
				// size is already correct
				return $preferred;
			}
		}
	}

	/**
	 * Resize an image by centering it on a larger canvas and return the
	 * resulting filename.
	 */
	function resizePng($infile, $size) {
		// temporary storage name for the PNG
		$outfile = getTempFile(PNG);
		
		// build and execute the Inkscape command
		$border = '-bordercolor none -border '.$size.'x'.$size;
		$gravity = '-gravity center';
		$crop = '-crop '.$size.'x'.$size.'+0+0';
		$command = "convert $infile $border $gravity $crop $outfile";
		system($command);

		return $outfile;
	}

	/**
	 * Render a SVG input file to a PNG version and return the PNG filename.
	 *
	 * This method will use Inkscape to create a rasterized version of the
	 * input image. The filename of the resulting file is returned for
	 * further processing. 
	 */
	function renderPng($svgfile, $size) {
		// check if input file exists
		if(!file_exists($svgfile)) {
			echo "Fatal error: SVG $svgfile does not exist\n";
			die(2);
		}		
		
		// temporary storage name for the PNG
		$pngfile = getTempFile(PNG);
		
		// build and execute the Inkscape command
		$command = "inkscape -z -e $pngfile -w $size -h $size $svgfile";
		system($command);
		
		return $pngfile;
	}

	/**
	 * Render an array containing category and icon name, custom name or
	 * null, to a PNG image and return the file name.
	 */
	function renderRow($iconarray, $size) {
		// render the icons, store PNG filenames
		$pngfiles = array();
		foreach($iconarray as $icon) {
			// check if a category and icon are given
			if(is_array($icon) && (2 == count($icon))) {
				// add the preferred icon image
				$pngfile =
					getPngFilenameFor(
						$icon[0], $icon[1], $size);
				$pngfiles[] = $pngfile;
			} else if (is_string($icon)) {
				// add our own image, check if SVG
				if(isSvg($icon)) {
					$pngfiles[] = renderPng($icon, $size);
				} else {
					$pngfiles[] = getCustomImage($icon);
				}
			} else {
				// add empty custom image
				$pngfiles[] = getCustomImage(NOIMAGE);
			}
		}
	
		// delegate to the real rendering method
		return renderRowFromPngs($pngfiles, $size);
	}

	/**
	 * Render a set of PNG images to a single image row.
	 *
	 * This method will use Imagemagick to compose a single image from a
	 * given array of PNGs and returns the resulting filename. 
	 */
	function renderRowFromPngs($pngfiles, $size) {
		// build the command to execute
		$tiles = '-tile '.count($pngfiles).'x1';
		$geometry = '-geometry '.$size.'x'.$size;
		$background = '-background none';
		
		$command = "montage $geometry $tiles $background";
	
		// loop through the supplied images
		foreach($pngfiles as $pngfile) {
			$command = $command.' '.$pngfile;
		}
		
		// get output file name and execute command
		$outfile = getTempFile(PNG);
		$command = $command.' '.$outfile;
		
		system($command);
		
		return $outfile;
	}
	
	/**
	 * Render a set of PNG images to a single image column.
	 *
	 * This method will use Imagemagick to create a single image containing
	 * all input images in a column and returns the resulting file name.
	 */
	function renderColumnFromPngs($pngfiles, $width, $rowheight) {
		// build the command to execute
		$tiles = '-tile 1x'.count($pngfiles);
		$geometry = '-geometry '.$width.'x'.$rowheight;
		$background = '-background none';
		
		$command = "montage $geometry $tiles $background";
		
		// loop through the supplied images
		foreach($pngfiles as $pngfile) {
			$command = $command.' '.$pngfile;
		}
		
		// create output file and execute command
		$outfile = getTempFile(PNG);
		$command = $command.' '.$outfile;
		
		system($command);
		
		return $outfile;
	}

	/**
	 * Render a highlighted version of an image file and return the
	 * resulting filename.
	 */
	function renderHighlight($infile) {
		$outfile = getTempFile(PNG);

		$command = "convert -modulate 110 $infile $outfile";
		system($command);
		
		return $outfile; 
	}
	
	/**
	 * Render a darkened version of an image file and return the
	 * resulting filename.
	 */
	function renderDarkened($infile) {
		$outfile = getTempFile(PNG);

		$command = "convert -modulate 70 $infile $outfile";
		system($command);
		
		return $outfile; 
	}
	
	/**
	 * Render a desatured version of an image file and return the resulting
	 * filename.
	 */
	function renderDesaturated($infile) {
		$outfile = getTempFile(PNG);
		
		$command = "convert -fx G $infile $outfile";
		system($command);
	
		return $outfile;
	}
	
	/**
	 * Render a semi transparent version of an image file and return the
	 * resulting filename.
	 */
	function renderSemiTransparent($infile) {
		$outfile = getTempFile(PNG);
		
		$command =
			"convert -channel alpha -fx u-0.5 $infile $outfile";
		system($command);
	
		return $outfile;
	}
	
	/**
	 * Render a disabled version of an image file and return the
	 * resulting filename.
	 */
	function renderDisabled($infile) {
		// first step is to convert to grayscale
		$desaturated = renderDesaturated($infile);
		$disabled = renderSemiTransparent($desaturated);
		
		return $disabled;
	}

	/**
	 * Render a transparent border around an image and return the resulting
	 * image file.
	 */
	function renderBorder($infile, $size, $border) {
		// build the command to execute
		$outfile = getTempFile(PNG);
		$tiles = '-tile 1x1';
		$geometry =
			'-geometry '.$size.'x'.$size.'+'.$border.'+'.$border;
		$background = '-background none';
		
		$command =
			"montage $geometry $tiles $background $infile $outfile";
			
		system($command);
		
		return $outfile;
	}

	/**
	 * Flip an image horizontally and return the resulting image filename.
	 */
	function flipHorizontal($infile) {
		$outfile = getTempFile(PNG);
		
		$command = "convert -flop $infile $outfile";
		system($command);
	
		return $outfile;
	}

	/**
	 * Resize an image and return the resulting filename.
	 */
	function resizeImage($infile, $width, $height) {
		$outfile = getTempFile(PNG);
		
		$resize = '-resize '.$width.'x'.$height.'\\!';
		$command = "convert $resize $infile $outfile";
		system($command);
	
		return $outfile;
	}

	/**
	 * Make an image opaque by replacing alpha with white.
	 */
	function makeOpaque($infile, $width, $height) {
		$whitefile = getTempFile(PNG);
		$size = '-size '.$width.'x'.$height;
		$white = 'xc:white';
		
		$command = "convert $size $white $whitefile";
		system($command);
		
		$outfile = getTempFile(PNG);
		$mode = '-compose atop';
		$command =
			"composite $mode $infile $whitefile $outfile";
		system($command);
		
		return $outfile;
	}

	/**
	 * Add an image overlay to an existing background
	 * image.
	 */
	function overlay($infile, $overlay, $x = 0, $y = 0) {
		$outfile = getTempFile(PNG);
		
		$mode = '-compose atop';
		$geometry = '-geometry +'.$x.'+'.$y;
		
		$command =
			"composite $mode $geometry $overlay $infile $outfile";
		system($command);
		
		return $outfile;
	}

	/**
	 * Write a text string to an existing image.
	 */
	function writeText($infile, $text, $size, $x, $y) {
		$outfile = getTempFile(PNG);
		
		$pointsize = '-pointsize '.$size;
		$draw = "-draw \"text $x,$y '$text'\"";
		
		$command =
			"convert $pointsize $draw $infile $outfile";
		system($command);
		
		return $outfile;
	}

?>
