/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file speedport.c
 * \brief This file contains Speedport router related functions
 */

#include <ffgtk.h>
#include <router/common.h>

static int speedportDetectFirmware( struct sProfile *psProfile );

/**
 * \brief Get router protocol
 * \param psProfile profile structure
 * \return used protocol "https://" or "http://"
 */
static gchar *getRouterProtocol( struct sProfile *psProfile ) {
	if ( psProfile -> sRouterInfo.bUseSsl == TRUE ) {
		return "https://";
	}

	return "http://";
}

/**
 * \brief Get router port
 * \param psProfile profile structure
 * \return 443 or 80
 */
static int getRouterPort( struct sProfile *psProfile ) {
	if ( psProfile -> sRouterInfo.bUseSsl == TRUE ) {
		return 443;
	}

	return 80;
}

/**
 * \brief Detect speedport
 * \param psProfile profile structure
 * \return error code of speedportDetectFirmware()
 */
static int speedportDetect( struct sProfile *psProfile ) {
	int nError;

	Debug( KERN_INFO, "Starting detection\n" );

	nError = speedportDetectFirmware( psProfile );
	if ( nError == 0 ) {
		Debug( KERN_DEBUG, "Speedport found!\n" );
	} else {
		Debug( KERN_DEBUG, "No Speedport\n" );
	}

	return nError;
}

/**
 * \brief Init speedport info structure
 * \param psProfile profile structure
 * \return 0
 */
static int speedportInit( struct sProfile *psProfile ) {
	memset( &psProfile -> sRouterInfo, 0, sizeof( psProfile -> sRouterInfo ) );
	psProfile -> sRouterInfo.pnFirmware = ( char * ) routerGetFirmware( psProfile );
	commonParseFirmware( psProfile );

	return 0;
}

/**
 * \brief Login to speedport
 * \param psProfile profile structure
 * \return error code, 0=success else error
 */
static int speedportLogin( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gint nError;
	gchar anUrl[ BUF_SIZE ];
	const gchar *pnPassword = routerGetPassword( psProfile );

	if ( pnPassword == NULL ) {
		return -1;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	if ( psProfile -> sRouterInfo.nSubType == 1 ) {
		setPostData( psHandler, "&getpage=../html/hcti_startseite.htm&login:command/password=%s", pnPassword );
	} else {
		setPostData( psHandler, "&login:command/password=%s", pnPassword );
	}
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		return nError;
	}
	saveDebugData( "speedport_login.html", psHandler -> pnData, psHandler -> nSize );

	if ( strstr( psHandler -> pnData, "<div id=c_errtitel>" ) ) {
		nError = -1;
		Debug( KERN_WARNING, "Login failed, wrong password!\n" );
		ShowError( TRUE, _( "Login failed" ), _( "Wrong password or already logged in!" ) );
	} else {
		nError = 0;
		Debug( KERN_DEBUG, "Login successful!\n" );
	}
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Logout from speedport
 * \param psProfile profile structure
 * \return error code, 0=success else error
 */
static int speedportLogout( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gint nError = 0;
	gchar anUrl[ BUF_SIZE ];

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	if ( psProfile -> sRouterInfo.nSubType == 1 ) {
		setPostData( psHandler, "&security:command/logout=1" );
	} else {
		setPostData( psHandler, "&logout" );
	}
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		return -3;
	}
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Detect new speedport firmware
 * \param psProfile profile structure
 * \param pnData data pointer
 * \error code 0=success, else error
 */
static int speedportDetectNewFirmware( struct sProfile *psProfile, char *pnData ) {
	int nError = 1, nStart0, nStart, nEnd;

	/* Get firmware */
	nStart0 = findString( pnData, 0, "Firmware Version:" );
	if ( nStart0 == -1 ) {
		goto end;
	}
	Debug( KERN_DEBUG, "Found 'Firmware Version:' string\n" );
	nStart = findString( pnData, nStart0, "<DIV class=colIndiv>" );
	if ( nStart == -1 ) {
		nStart = findString( pnData, nStart0, "<td>" );
		if ( nStart == -1 ) {
			goto end;
		}
		psProfile -> sRouterInfo.nSubType = 1;
		nStart += 4;
	} else {
		nStart += 20;
	}
	nEnd = findString( pnData, nStart, "<" );
	if ( nEnd == -1 ) {
		goto end;
	}
	Debug( KERN_DEBUG, "nStart %d, nEnd %d\n", nStart, nEnd );
	Debug( KERN_DEBUG, "Firmware: '%s'\n", ( char * ) getSubString( pnData, nStart, nEnd - nStart ) );
		
	psProfile -> sRouterInfo.pnFirmware = getSubString( pnData, nStart, nEnd - nStart );
	routerSetFirmware( psProfile, psProfile -> sRouterInfo.pnFirmware );

	commonParseFirmware( psProfile );
	nError = 0;

end:

	return nError;
}

/**
 * \brief Detect speedport firmware
 * \param psProfile profile structure
 * \error code 0=success, else error
 */
static int speedportDetectFirmware( struct sProfile *psProfile ) {
	gint nError = 1;
	gint nStart, nEnd;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];

	/* Check for https */
	Debug( KERN_DEBUG, "Check for https:443\n" );
	snprintf( anUrl, sizeof( anUrl ), "https://%s/cgi-bin/webcm?getpage=../html/top_status.htm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 443 );
	nError = readUrl( psHandler, psProfile );
	Debug( KERN_DEBUG, "nError %d\n", nError );
	if ( nError == 0 ) {
		saveDebugData( "speedport_newfirmware.html", psHandler -> pnData, psHandler -> nSize );

		nError = speedportDetectNewFirmware( psProfile, psHandler -> pnData );
		if ( nError == 0 ) {
			psProfile -> sRouterInfo.bUseSsl = TRUE;
		}
		freeHandler( psHandler );
		return nError;
	}
	freeHandler( psHandler );

	Debug( KERN_DEBUG, "Check for https:80\n" );
	snprintf( anUrl, sizeof( anUrl ), "https://%s/cgi-bin/webcm?getpage=../html/top_status.htm", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Check for http:443\n" );
		snprintf( anUrl, sizeof( anUrl ), "http://%s/cgi-bin/webcm?getpage=../html/top_status.htm", routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, 80 );
		nError = readUrl( psHandler, psProfile );
		psProfile -> sRouterInfo.bUseSsl = FALSE;
	} else {
		psProfile -> sRouterInfo.bUseSsl = TRUE;
	}
	Debug( KERN_DEBUG, "Use SSL %d\n", psProfile -> sRouterInfo.bUseSsl );
	Debug( KERN_DEBUG, "nError %d\n", nError );

	if ( nError == 0 ) {
		nError = 1;

		/* Get firmware */
		saveDebugData( "speedport_firmware.html", psHandler -> pnData, psHandler -> nSize );

		nStart = findString( psHandler -> pnData, 0, "Firmware Version:" );
		if ( nStart == -1 ) {
			goto end;
		}
		nStart = findString( psHandler -> pnData, nStart, "<td>" );
		if ( nStart == -1 ) {
			goto end;
		}
		nStart += 4;
		nEnd = findString( psHandler -> pnData, nStart, "</td>" );
		if ( nEnd == -1 ) {
			goto end;
		}
		//printf( "nStart %d, nEnd %d\n", nStart, nEnd );
		//printf( "Firmware: '%s'\n", ( char * ) getSubString( psHandler -> pnData, nStart, nEnd - nStart ) );
		
		psProfile -> sRouterInfo.pnFirmware = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
		//Debug( KERN_DEBUG, "firmware: '%s'\n", psProfile -> sRouterInfo.pnFirmware );
		routerSetFirmware( psProfile, psProfile -> sRouterInfo.pnFirmware );

		commonParseFirmware( psProfile );
		nError = 0;
	}

end:
	freeHandler( psHandler );
	return nError;
}

/**
 * \brief Read call list from speedport
 * \param psProfile profile structure
 * \param pnCallerList callerlist filename
 * \return error code
 */
static int speedportGetCallList( struct sProfile *psProfile, const gchar *pnCallerList ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nSkip = 0;
	gint nError = -1;

	if ( routerLogin( psProfile ) ) {
		Debug( KERN_WARNING, "Wrong password\n" );
		return -1;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	setPostData( psHandler, "getpage=../html/hcti_status_telanrl.htm" );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}
	saveDebugData( "speedport_getcalllist.html", psHandler -> pnData, psHandler -> nSize );
	freeHandler( psHandler );

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/anrliste.csv", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}

	saveData( ( char * ) pnCallerList, psHandler -> pnData + nSkip, psHandler -> nSize - nSkip );
	freeHandler( psHandler );

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/gsprdatn.csv", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		routerLogout( psProfile );
		return nError;
	}

	nSkip = findString( psHandler -> pnData, 0, "\n" );
	addData( ( char * ) pnCallerList, psHandler -> pnData + nSkip );
	freeHandler( psHandler );

	routerLogout( psProfile );

	return 0;
}

/**
 * \brief Get phone settings (country/city prefix) of speedport (UNSUPPORTED)
 * \param psProfile profile structure
 * \return error code, -1
 */
static int speedportGetPhoneSettings( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gint nError = 0;

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_telefon_voip_einst.htm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		saveDebugData( "speedport_getphonesettings.html", psHandler -> pnData, psHandler -> nSize );
		gint nCityPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/OKZ" );
		if ( nCityPrefixStart == -1 ) {
			Debug( KERN_DEBUG, "error parsing city prefix start\n" );
			goto end;
		}
		nCityPrefixStart += 37;

		gint nCityPrefixStop = findString( psHandler -> pnData, nCityPrefixStart, "\"" );
		if ( nCityPrefixStop == -1 ) {
			Debug( KERN_DEBUG, "error parsing city prefix stop\n" );
			goto end;
		}

		gint nNationalPrefixStart = findString( psHandler -> pnData, 0, "telcfg:settings/Location/OKZPrefix" );
		if ( nNationalPrefixStart == -1 ) {
			Debug( KERN_DEBUG, "error parsing national prefix start\n" );
			goto end;
		}
		nNationalPrefixStart += 43;

		gint nNationalPrefixStop = findString( psHandler -> pnData, nNationalPrefixStart, "\"" );
		if ( nNationalPrefixStop == -1 ) {
			Debug( KERN_DEBUG, "error parsing national prefix stop\n" );
			goto end;
		}

		psProfile -> sRouterInfo.pnNationalPrefix = getSubString( psHandler -> pnData, nNationalPrefixStart, nNationalPrefixStop - nNationalPrefixStart );
		psProfile -> sRouterInfo.pnAreaCode = getSubString( psHandler -> pnData, nCityPrefixStart, nCityPrefixStop - nCityPrefixStart );

		if ( psProfile -> sRouterInfo.pnCountryCode == NULL ) {
			Debug( KERN_DEBUG, "Country code set to germany (49)\n" );
			psProfile -> sRouterInfo.pnCountryCode = g_strdup( "49" );
		}

		Debug( KERN_DEBUG, "CountryCode: %s\n", psProfile -> sRouterInfo.pnCountryCode );
		Debug( KERN_DEBUG, "AreaCode: %s\n", psProfile -> sRouterInfo.pnAreaCode );
		Debug( KERN_DEBUG, "National prefix: %s\n", psProfile -> sRouterInfo.pnNationalPrefix );
		Debug( KERN_DEBUG, "International prefix: %s\n", psProfile -> sRouterInfo.pnInternationalPrefix );
	}

end:
	freeHandler( psHandler );

	routerLogout( psProfile );

	return 0;
}

/**
 * \brief Get fon msn
 * \param pnData data buffer
 * \param nType what kind of fon do we want
 * \return msn number
 */
static char *getFon( char *pnData, int nType ) {
	int nSkip = 0, nEnd;

	switch ( nType ) {
		case 0 ... 9: {
			gchar *pnTmp = g_strdup_printf( "telcfg:settings/MSN/MSN%d", nType );
			nSkip = findString( pnData, 0, pnTmp );
			g_free( pnTmp );
			break;
		}
		case 10:
			nSkip = findString( pnData, 0, "telcfg:settings/MSN/POTS" );
			break;
	}

	if ( nSkip == -1 ) {
		return NULL;
	}

	nSkip = findString( pnData, nSkip, "value=\"" );
	if ( nSkip == -1 ) {
		return NULL;
	}

	nSkip += 7;
	nEnd = findString( pnData, nSkip, "\"" );
	if ( nEnd - nSkip <= 0 ) {
		return NULL;
	}

	return getSubString( pnData, nSkip, nEnd - nSkip );
}

/**
 * \brief Get phone name
 * \param pnData html data
 * \return phone name
 */
static gchar *speedportGetPhoneName( gchar *pnData ) {
	gchar *pnRet = NULL;
	gint nStart, nEnd;
	gchar *pnName = NULL;

	nStart = findString( pnData, 0, "Anschlussbezeichnung" );
	if ( nStart != -1 ) {
		nStart = findString( pnData, nStart, "document.write" );
		if ( nStart != -1 ) {
			nStart = findString( pnData, nStart, "\"" );
			nEnd = findString( pnData, nStart + 1, "\"" );

			pnName = getSubString( pnData, nStart + 1, nEnd - nStart - 1 );
			if ( pnName != NULL ) {
				pnRet = convertEntities( pnName );
				g_free( pnName );
			}
		}
	}

	return pnRet;
}


/**
 * \brief Get supported ports of speedport
 * \param psProfile profile structure
 * \return error code, -1
 */
static int speedportGetActivePorts( struct sProfile *psProfile ) {
	gint nStart, nEnd, nIndex;
	gchar *pnMsn = NULL;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gint nError = -1;

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_telefon_kombi.htm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		saveDebugData( "speedport_active_ports_new.html", psHandler -> pnData, psHandler -> nSize );

		routerClearNumbers( psProfile );

		for ( nIndex = 0; nIndex < 11; nIndex++ ) {
			gchar *pnFon = NULL;

			pnFon = getFon( psHandler -> pnData, nIndex );
			if ( pnFon != NULL ) {
				routerSetFonNumber( psProfile, nIndex, pnFon );
				Debug( KERN_DEBUG, "MSN%d %s\n", nIndex, pnFon );
			}
		}
		freeHandler( psHandler );

		/* Try to read telephone name */
		snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_telefon_as.htm&var:as=1", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
		nError = readUrl( psHandler, psProfile );
		if ( nError == 0 ) {
			gchar *pnName = NULL;
			saveDebugData( "speedport_active_ports_new_as1.html", psHandler -> pnData, psHandler -> nSize );

			pnName = speedportGetPhoneName( psHandler -> pnData );
			if ( pnName != NULL ) {
				routerSetPortName( psProfile, PORT_ANALOG1, pnName );
				g_free( pnName );
			}
		}

		freeHandler( psHandler );

		snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_telefon_as.htm&var:as=2", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
		psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
		nError = readUrl( psHandler, psProfile );
		if ( nError == 0 ) {
			gchar *pnName = NULL;
			saveDebugData( "speedport_active_ports_new_as2.html", psHandler -> pnData, psHandler -> nSize );

			pnName = speedportGetPhoneName( psHandler -> pnData );
			if ( pnName != NULL ) {
				routerSetPortName( psProfile, PORT_ANALOG2, pnName );
				g_free( pnName );
			}
		}
	}
	freeHandler( psHandler );

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_telefon_s0_kommend.htm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		/* Get msns */
		saveDebugData( "speedport_active_ports.html", psHandler -> pnData, psHandler -> nSize );
		nStart = findString( psHandler -> pnData, 0, "new Array(" );
		nStart += 10;
		nEnd = findString( psHandler -> pnData, nStart, ");" );
		nIndex = 0;
		while ( nStart < nEnd ) {
			int nPos1 = findString( psHandler -> pnData, nStart, "'" );
			if ( nPos1 >= nEnd ) {
				break;
			}
			int nPos2 = findString( psHandler -> pnData, nPos1 + 1, "'" );
			if ( nPos2 >= nEnd ) {
				break;
			}

			//Debug( KERN_DEBUG, "nStart %d, nEnd %d, nPos1 %d, nPos2 %d\n", nStart, nEnd, nPos1, nPos2 );

			pnMsn = getSubString( psHandler -> pnData, nPos1 + 1, nPos2 - nPos1 - 1 );
			Debug( KERN_DEBUG, "MSN: '%s'\n", pnMsn );
			routerSetFonNumber( psProfile, nIndex, pnMsn );
			Debug( KERN_DEBUG, "FON%d %s\n", nIndex, pnMsn );
			nIndex++;
			nStart = nPos2 + 1;
			g_free( pnMsn );
		}
	}

	freeHandler( psHandler );

	routerLogout( psProfile );
	return 0;
}

/**
 * \brief Clear speedport call list
 * \param psProfile profile structure
 * \return error code
 */
static int speedportClearCallList( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	if ( routerLogin( psProfile ) != 0 ) {
		return -1;
	}

	Debug( KERN_DEBUG, "Delete call list on speedport...\n" );

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	setPostData( psHandler,	"getpage=../html/hcti_status.htm&errorpage=../html/hcti_status.htm&telcfg:settings/ClearJournal=1" );
	nError = readUrl( psHandler, psProfile );
	freeHandler( psHandler );

	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Reconnect internet connection
 * \param psProfile profile structure
 * \return error code
 */
static int speedportReconnect( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	if( routerLogin( psProfile ) != 0 ) {
		return nError;
	}

	/* Disconnect */
	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	setPostData( psHandler,	"connection0:settings/cmd_disconnect=1" );
	nError = readUrl( psHandler, psProfile );
	freeHandler( psHandler );

	/* Connect */
	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	setPostData( psHandler,	"connection0:settings/cmd_connect=1" );
	nError = readUrl( psHandler, psProfile );
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Get external ip
 * \param psProfile profile structure
 * \return ip address or NULL
 */
static gchar *speedportGetIp( struct sProfile *psProfile ) {
	gchar *pnIp = NULL;
	gint nStart, nEnd;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gint nError = -1;

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/top_status.htm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		saveDebugData( "speedport_getip.html", psHandler -> pnData, psHandler -> nSize );
		/* Get external ip */
		nStart = findString( psHandler -> pnData, 0, "Aktiv: " );
		if ( nStart == -1 ) {
			goto end;
		}

		nStart += 7;
		nEnd = findString( psHandler -> pnData, nStart, "</td>" );
		if ( nEnd == -1 ) {
			goto end;
		}

		//Debug( KERN_DEBUG, "nStart %d, nEnd %d\n", nStart, nEnd );

		pnIp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
		//Debug( KERN_DEBUG, "IP: '%s'\n", pnIp );
	}

end:
	freeHandler( psHandler );

	return pnIp;
}

/**
 * \brief Get maximal speed
 * \param psProfile profile structure
 * \param pnMaxUp maximal up speed
 * \param pnMaxDown maximal down speed
 */
static void speedportGetSpeed( struct sProfile *psProfile, gint *pnMaxUp, gint *pnMaxDown ) {
	gint nInfo, nStart, nEnd;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gchar *pnTmp;
	gint nError = -1;

	if( routerLogin( psProfile ) != 0 ) {
		return;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s%s/cgi-bin/webcm?getpage=../html/hcti_status_dsl.htm", getRouterProtocol( psProfile ), routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, getRouterPort( psProfile ) );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		saveDebugData( "speedport_getspeed.html", psHandler -> pnData, psHandler -> nSize );
		nInfo = findString( psHandler -> pnData, 0, "DSL Down-/Upstream:" );
		if ( nInfo == -1 ) {
			Debug( KERN_DEBUG, "Error 0\n" );
			goto end;
		}
			 
		nInfo += 18;
		nStart = findString( psHandler -> pnData, nInfo, "colText>" );
		if ( nStart == -1 ) {
			nStart = findString( psHandler -> pnData, nInfo, "<td width" );
			if ( nStart == -1 ) {
				Debug( KERN_DEBUG, "Error 1.1\n" );
				goto end;
			}

			nStart += 7;
			nStart = findString( psHandler -> pnData, nStart, ">" );
			if ( nStart == -1 ) {
				Debug( KERN_DEBUG, "Error 1.2\n" );
				goto end;
			}
			nStart += 1;

			nEnd = findString( psHandler -> pnData, nStart, " kBit/s" );
			if ( nEnd == -1 ) {
				Debug( KERN_DEBUG, "Error 1.3\n" );
				goto end;
			}
		} else {
			nStart += 8;

			nEnd = findString( psHandler -> pnData, nStart, " kbit/s" );
			if ( nEnd == -1 ) {
				Debug( KERN_DEBUG, "Error 2.1\n" );
				goto end;
			}
		}

		//Debug( KERN_DEBUG, "nStart %d, nEnd %d\n", nStart, nEnd );

		pnTmp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
		//Debug( KERN_DEBUG, "Down: '%s'\n", pnTmp );
		*pnMaxDown = atoi( pnTmp ) * 1024;
		g_free( pnTmp );

		nStart = findString( psHandler -> pnData, nEnd, "colLast>" );
		if ( nStart == -1 ) {
			nStart = findString( psHandler -> pnData, nEnd, "<td>" );
			if ( nStart == -1 ) {
				Debug( KERN_DEBUG, "Error 3.1\n" );
				goto end;
			}
			nStart += 4;
			nEnd = findString( psHandler -> pnData, nStart + 1, " kBit/s" );
			if ( nEnd == -1 ) {
				Debug( KERN_DEBUG, "Error 3.2\n" );
				goto end;
			}
		} else {
			nStart += 8;
			nEnd = findString( psHandler -> pnData, nStart + 1, " kbit/s" );
			if ( nEnd == -1 ) {
				Debug( KERN_DEBUG, "Error 4.1\n" );
				goto end;
			}
		}


		//Debug( KERN_DEBUG, "nStart %d, nEnd %d\n", nStart, nEnd );

		pnTmp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
		//Debug( KERN_DEBUG, "Up: '%s'\n", pnTmp );

		*pnMaxUp = atoi( pnTmp ) * 1024;
		g_free( pnTmp );
	}

end:
	freeHandler( psHandler );

	routerLogout( psProfile );
}

/**
 * \brief Get speedport name
 * \param psProfile profile structure
 * \return speedport name or NULL
 */
static const gchar *speedportGetName( struct sProfile *psProfile ) {
	switch ( psProfile -> sRouterInfo.nType ) {
		case FRITZBOX_SPEEDPORT_W501V:
			return "T-Com Speedport W 501V";
		case FRITZBOX_SPEEDPORT_W701V:
			return "T-Com Speedport W 701V";
		case FRITZBOX_SPEEDPORT_W900V:
			return "T-Com Speedport W 900V";
		case FRITZBOX_SPEEDPORT_W920V:
			return "T-Com Speedport W 920V";
		default:
			break;
	}

	return NULL;
}

/**
 * \brief Get speedport version
 * \param psProfile profile structure
 * \return speedport version or "Unknown"
 */
static const gchar *speedportGetVersion( struct sProfile *psProfile ) {
	if ( psProfile -> sRouterInfo.pnFirmware != NULL ) {
		return psProfile -> sRouterInfo.pnFirmware;
	}

	return "Unknown";
}

/** Speedport Router structure */
struct sRouter sSpeedport = {
	"T-Com Speedport",
	speedportDetect,
	speedportInit,
	speedportLogin,
	speedportLogout,
	speedportDetectFirmware,
	speedportGetPhoneSettings,
	speedportGetActivePorts,
	speedportGetCallList,
	speedportClearCallList,
	NULL,
	NULL,
	speedportReconnect,
	speedportGetIp,
	speedportGetSpeed,
	NULL,
	NULL,
	speedportGetName,
	speedportGetVersion,
	NULL,
	NULL,
};
