/* -*- mode: c++; coding: utf-8; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; show-trailing-whitespace: t -*- vim:fenc=utf-8:ft=tcl:et:sw=4:ts=4:sts=4

  This file is part of the Feel library

  Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
       Date: 2007-06-11

  Copyright (C) 2007-2011 Université Joseph Fourier (Grenoble I)

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3.0 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file heatsink.cpp
   \author Baptiste Morin <baptistemorin@gmail.com>
   \date 2011-06-28
 */
#include <feel/options.hpp>
#include <feel/feelcore/application.hpp>

#include <feel/feelalg/backendgmm.hpp>
#include <feel/feelalg/backendpetsc.hpp>

#include <feel/feeldiscr/functionspace.hpp>
#include <feel/feeldiscr/region.hpp>
#include <feel/feelpoly/im.hpp>

#include <feel/feelfilters/gmsh.hpp>
#include <feel/feelfilters/exporter.hpp>
#include <feel/feelfilters/gmshhypercubedomain.hpp>
#include <feel/feelpoly/polynomialset.hpp>

#include <feel/feelmesh/filters.hpp>
#include <feel/feelvf/vf.hpp>
#include <feel/feeldiscr/bdf2.hpp>

Feel::gmsh_ptrtype makefin( double hsize, double width, double deep , double L );

//# marker1 #
inline
Feel::po::options_description
makeOptions()
{
    Feel::po::options_description heatsinkoptions("heatsink options");
    heatsinkoptions.add_options()
    // mesh parameters
    ("hsize", Feel::po::value<double>()->default_value( 0.0001 ),
     "first h value to start convergence")
    ("L", Feel::po::value<double>()->default_value( 0.02 ),
     "dimensional length of the sink (in meters)")
    ("width", Feel::po::value<double>()->default_value( 0.0005 ),
     "dimensional width of the fin (in meters)")

	// 3D parameter
	("deep", Feel::po::value<double>()->default_value( 0 ),
	 "depth of the mesh (in meters) only in 3D simulation")

    // thermal conductivities parameters
    ("kappa_s", Feel::po::value<double>()->default_value( 386 ),
     "thermal conductivity of the base spreader in SI unit W.m^{-1}.K^{-1}")
    ("kappa_f", Feel::po::value<double>()->default_value( 386 ),
     "thermal conductivity of the fin in SI unit W.m^{-1}.K^{-1}")

	// density parameter
	("rho_s", Feel::po::value<int>()->default_value( 8940 ),
	 "density of the spreader's material in SI unit kg.m^{-3}")
	("rho_f", Feel::po::value<int>()->default_value( 8940 ),
	 "density of the fin's material in SI unit kg.m^{-3}")

	// heat capacities parameter
	("c_s", Feel::po::value<double>()->default_value( 385 ),
	 "heat capacity of the spreader's material in SI unit J.kg^{-1}.K^{-1}")
	("c_f", Feel::po::value<double>()->default_value( 385 ),
	 "heat capacity of the fin's material in SI unit J.kg^{-1}.K^{-1}")

	// physical coeff
    ("therm_coeff", Feel::po::value<double>()->default_value(1e3),
     "thermal coefficient")
    ("Tamb", Feel::po::value<double>()->default_value(300),
     "ambiant temperature")
    ("heat_flux", Feel::po::value<double>()->default_value(1e6),
     "heat flux generated by CPU")

    ("steady", Feel::po::value<bool>()->default_value(false),
     "if true : steady else unsteady")

    // export
    ("export-matlab", "export matrix and vectors in matlab" );

    return heatsinkoptions.add( Feel::feel_options() );
}
//# endmarker1 #

inline
Feel::AboutData
makeAbout()
{
    Feel::AboutData about( "heatsink" ,
                            "heatsink" ,
                            "0.1",
                            "nD(n=1,2,3) Heat sink thermal fin on simplices or simplex products",
                            Feel::AboutData::License_GPL,
                            "Copyright (c) 2006-2011 Universite Joseph Fourier");

    about.addAuthor("Christophe Prud'homme", "developer", "christophe.prudhomme@ujf-grenoble.fr", "");
    about.addAuthor("Baptiste Morin", "junior developer", "baptistemorin@gmail.com","");
    return about;
}


namespace Feel
{
/**
 * Heat sink application
 */
template<int Dim, int Order>
class HeatSink
    :
    public Application
{
    typedef Application super;
public:

#define Entity Simplex

    // -- TYPEDEFS --
    typedef double value_type;

    typedef Backend<value_type> backend_type;
    typedef boost::shared_ptr<backend_type> backend_ptrtype;

    /*matrix*/
    typedef backend_type::sparse_matrix_type sparse_matrix_type;
    typedef backend_type::sparse_matrix_ptrtype sparse_matrix_ptrtype;
    typedef backend_type::vector_type vector_type;
    typedef backend_type::vector_ptrtype vector_ptrtype;

    /*mesh*/
    typedef Entity<Dim> entity_type;
    typedef Mesh<entity_type> mesh_type;
    typedef boost::shared_ptr<mesh_type> mesh_ptrtype;

    typedef FunctionSpace<mesh_type, fusion::vector<Lagrange<0, Scalar> >, Discontinuous > p0_space_type;
    typedef typename p0_space_type::element_type p0_element_type;

    /*basis*/
    typedef fusion::vector<Lagrange<Order, Scalar> > basis_type;

    /*space*/
    typedef FunctionSpace<mesh_type, basis_type, value_type> space_type;
    typedef boost::shared_ptr<space_type> space_ptrtype;
    typedef typename space_type::element_type element_type;

	/* BDF discretization */
	typedef Bdf<space_type>  bdf_type;
	typedef boost::shared_ptr<bdf_type> bdf_ptrtype;

    /* export */
    typedef Exporter<mesh_type> export_type;

	/* constructor */
	HeatSink( int argc, char** argv, AboutData const& ad, po::options_description const& od );

    /* run the simulation */
    void run();

private:

    /**
     * export results to ensight format
     */
    void exportResults( double time, element_type& u );

private:

    backend_ptrtype M_backend;

	/* mesh parameters */
    double meshSize;
    double depth;
    double L;
    double width;

	/* thermal conductivities */
	double kappa_s;
	double kappa_f;

	/* density of the materials */
	int rho_s;
	int rho_f;

	/* heat capacity of the materials*/
	double c_s;
	double c_f;

	/* thermal coeff */
    double therm_coeff;

	/* ambien temperature, and heat flux (Q) */
    double Tamb;
    double heat_flux;

    /* surfaces*/
    double surface_base, surface_fin;

    /* mesh, pointers and spaces */
    mesh_ptrtype mesh;
    space_ptrtype Xh;

    sparse_matrix_ptrtype D;
    vector_ptrtype F;

	/* time management */
	bdf_ptrtype M_bdf;
    bool steady;

    // average file
    std::ofstream out;
    double Tavg, Tgamma1;
    boost::shared_ptr<export_type> exporter;

}; // HeatSink class

/* Constructor */
template<int Dim, int Order>
HeatSink<Dim,Order>::HeatSink( int argc, char** argv, AboutData const& ad, po::options_description const& od )
		:
		super( argc, argv, ad, od ),
		M_backend( backend_type::build( this->vm() ) ),
		meshSize( this->vm()["hsize"].template as<double>() ),
		depth( this->vm()["deep"].template as<double>() ),
		L( this->vm()["L"].template as<double>() ),
        width( this->vm()["width"].template as <double>() ),
		kappa_s( this-> vm()["kappa_s"].template as<double>() ),
		kappa_f( this-> vm()["kappa_f"].template as<double>() ),
		rho_s( this-> vm()["rho_s"].template as<int>() ),
		rho_f( this-> vm()["rho_f"].template as<int>() ),
		c_s( this-> vm()["c_s"].template as<double>() ),
		c_f( this-> vm()["c_f"].template as<double>() ),
        therm_coeff( this-> vm()["therm_coeff"].template as <double>() ),
        Tamb( this-> vm()["Tamb"].template as <double>() ),
        heat_flux( this-> vm()["heat_flux"].template as <double>() ),
        steady( this->vm()["steady"].template as<bool>() ),
		exporter( export_type::New( this->vm(), this->about().appName() ) )
		{
			this->changeRepository( boost::format( "%1%/%2%/%3%/" )
								   % this->about().appName()
								   % entity_type::name()
								   % this->vm()["hsize"].template as<double>()
								  );
		using namespace Feel::vf;

	    /*
		 * First we create the mesh
		 */
        mesh = createGMSHMesh ( _mesh = new mesh_type,
                                _desc = makefin( meshSize, width, depth , L),
                                _update=MESH_UPDATE_FACES | MESH_UPDATE_EDGES );

		/*
		 * Calculate the two surfaces used for averages calculation
		 */
		surface_base = integrate( _range= markedfaces(mesh, "gamma4"), _expr= cst(1.) ).evaluate()(0,0);
        surface_fin = integrate( _range= markedfaces(mesh,"gamma1"), _expr=cst(1.) ).evaluate()(0,0);

        /*
         * The function space associated to the mesh
         */
        Xh = space_type::New( mesh );
        M_bdf = bdf( _space=Xh, _vm=this->vm(), _name="Temperature" );

        /*
         * Right hand side
         */
		F = M_backend->newVector( Xh );

		/*
		 * Left hand side
		 */
		D = M_backend->newMatrix( Xh, Xh );

    }



template<int Dim, int Order>
void
HeatSink<Dim, Order>::run()
{
    if ( this->vm().count( "help" ) )
    {
        std::cout << this->optionsDescription() << "\n";
        return;
    }

    using namespace Feel::vf;

    Log() << "meshSize = " << meshSize << "\n"
          << "L = "<< L <<"\n"
          << "width = " << width << "\n"
          << "depth = " << depth << "\n"
          << "kappa_spreader = " << kappa_s << "\n"
          << "kappa_fin = " << kappa_f << "\n"
          << "rho_spreader = " << rho_s << "\n"
          << "rho_fin = " << rho_f << "\n"
          << "heat capacity of the base = " << c_s << "\n"
          << "heat capacity of the fin = " << c_f << "\n"
          << "ambient temperature = " << Tamb << "\n"
          << "thermal coefficient = " << therm_coeff << "\n"
          << "heat flux Q = " << heat_flux << "\n"
          << "steady state = " << steady << "\n";

    /*
     * T is the unknown, v the test function
     */
    element_type T( Xh, "T" );
    element_type v( Xh, "v" );

    /*
     * Right hand side construction (steady state)
     */
    form1( _test=Xh, _vector=F, _init=true ) = integrate( _range= markedfaces(mesh, "gamma1"), _expr= therm_coeff*Tamb*id(v));

    /*
     * Left hand side construction (steady state)
     */
    form2( Xh, Xh, D, _init=true ) = integrate( _range= markedelements(mesh,"spreader_mesh"), _expr= kappa_s*gradt(T)*trans(grad(v)) );
    form2( Xh, Xh, D) += integrate( _range= markedelements(mesh,"fin_mesh"), _expr= kappa_f*gradt(T)*trans(grad(v)) );
    form2 (Xh, Xh, D) += integrate( _range= markedfaces(mesh, "gamma1"), _expr= therm_coeff*idt(T)*id(v));

    M_bdf->start();

    //from now if the option "steady" is set to true then M_bdf->setSteady will set time-step=time-final=1e30
    if (steady)
    {
        M_bdf->setSteady();
    }

    form2(Xh, Xh, D) +=
        integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=rho_s*c_s*idt(T)*id(v)*M_bdf->polyDerivCoefficient(0) )
        + integrate( _range=markedelements(mesh, "fin_mesh"), _expr=rho_f*c_f*idt(T)*id(v)*M_bdf->polyDerivCoefficient(0) );

    D->close();

    if ( this->vm().count( "export-matlab" ) )
    {
        D->printMatlab( "D.m" );
    }

    /*
     * Left and right hand sides construction (non-steady state) with BDF
     */
    T = vf::project( _space=Xh, _expr=cst(Tamb) );

    M_bdf->initialize(T);

    std::cout << "The step is : " << M_bdf->timeStep() << "\n"
              << "The initial time is : " << M_bdf->timeInitial() << "\n"
              << "The final time is  : " << M_bdf->timeFinal() << "\n";

    // average file which contains: time Tavg_base Tavg_gamma1
    out.open("averages", std::ios::out);
    auto Ft = M_backend->newVector( Xh );
    for ( M_bdf->start(); M_bdf->isFinished()==false; M_bdf->next() )
    {
        // update right hand side with time dependent terms
        auto bdf_poly = M_bdf->polyDeriv();
        form1( _test=Xh, _vector=Ft ) =
            integrate( _range=markedelements(mesh, "spreader_mesh"), _expr=rho_s*c_s*idv(bdf_poly)*id(v)) +
            integrate( _range=markedelements(mesh, "fin_mesh"), _expr=rho_f*c_f*idv(bdf_poly)*id(v) );
        form1( _test=Xh, _vector=Ft ) +=
            integrate( _range= markedfaces(mesh,"gamma4"), _expr= heat_flux*(1-exp(-M_bdf->time()))*id(v) );

        // add contrib from time independent terms
        Ft->add( 1., F );

		M_backend->solve( _matrix=D, _solution=T, _rhs=Ft );

        Tavg = integrate( _range=markedfaces(mesh,"gamma4"), _expr=(1/surface_base)*idv(T) ).evaluate()(0,0);
        Tgamma1 = integrate( _range=markedfaces(mesh,"gamma1"), _expr=(1/surface_fin)*idv(T) ).evaluate()(0,0);

        // export results
        out << M_bdf->time() << " " << Tavg << " " << Tgamma1 << "\n";
		this->exportResults( M_bdf->time(), T );

     }

    std::cout << "Resolution ended, export done \n";

} // HeatSink::run


template<int Dim, int Order>
void
HeatSink<Dim, Order>::exportResults( double time, element_type& U )
{
    if ( this->vm().count( "export" ) )
        {
            exporter->step(time)->setMesh( U.functionSpace()->mesh() );
            exporter->step(time)->add( "Temperature", U );
            exporter->save();
        }
} // HeatSink::exportResults

} // Feel


int
main( int argc, char** argv )
{
    using namespace Feel;

	/* Parameters to be changed */
	const int nDim = 2;
	const int nOrder = 2;

	/* define application */
	typedef Feel::HeatSink<nDim, nOrder> heat_sink_type;

	/* instanciate */
	heat_sink_type heatsink( argc, argv, makeAbout(), makeOptions() );

	/* run */
    heatsink.run();
}





