/* $Id: glue-vhdl.c 4801 2009-10-02 13:38:29Z potyra $
 *
 *  Glue layer between fauhdli and FAUmachine to access FAUmachine signals 
 *  from VHDL.
 *  This file is only used for the standalone fauhdli interpreter, and
 *  does nothing but log a few error messages.
 *  Since the standalone interpreter uses free/malloc from glibc, it's
 *  safe to use these here as well.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "glue-vhdl.h"
#include "glue-log.h"
#include <assert.h>
#include <stdlib.h>

struct glue_vhdl {
	unsigned int max_sig_id;
	unsigned int max_comp_id;
	unsigned int max_arch_id;
};


void
glue_vhdl_proc_set(
	void *_cpssp,
	const char *proc,
	const char *param,
	union fauhdli_value value
)
{
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Set foreign paramter %s of procedure %s. Skipping.\n",
		param, proc);
}

void
glue_vhdl_proc_call(void *_cpssp, const char *proc)
{
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Foreign procedure call of %s requested. Skipping.\n",
		proc);
}

void
glue_vhdl_set(
	void *_cpssp,
	unsigned int sig_id, 
	union fauhdli_value data,
	void *drv
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(sig_id <= cpssp->max_sig_id);
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Setting foreign signal %d requested. Skipping.\n", sig_id);
}

void
glue_vhdl_connect_out(
	void *_cpssp, 
	unsigned int sig_id,
	union fauhdli_value init,
	void *drv
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(sig_id <= cpssp->max_sig_id);
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Connecting a driver to a foreign signal %d requested. "
		"Skipping.\n", sig_id);
}

void
glue_vhdl_connect_in(
	void *_cpssp,
	unsigned int sig_id,
	void *drv
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(sig_id <= cpssp->max_sig_id);
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Connecting a signal to a foreign signal %d requested. "
		"Skipping.\n", sig_id);
}

unsigned int
glue_vhdl_create_signal(
	void *_cpssp, 
	const char *type,
	const char *name
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl", 
		"Foreign signal %s of type %s requested. Skipping.\n",
		name, type);
	cpssp->max_sig_id++;
	return cpssp->max_sig_id;
}

unsigned int
glue_vhdl_comp_create(
	void *_cpssp, 
	const char *type, 
	const char *name
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl*)_cpssp;

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Foreign component %s of type %s requested. Skipping\n",
		name, type);
	cpssp->max_comp_id++;
	return cpssp->max_comp_id;
}

unsigned int
glue_vhdl_arch_create(
	void *_cpssp,
	const char *type, 
	const char *name
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Foreign architecture %s of type %s requested. Skipping\n",
		name, type);
	cpssp->max_arch_id++;
	return cpssp->max_arch_id;
}


void
glue_vhdl_comp_init(void *_cpssp, unsigned int comp)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(comp <= cpssp->max_comp_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Initialization of foreign component %d requested."
		"Skipping.\n", comp);
}

void
glue_vhdl_arch_init(void *_cpssp, unsigned int arch)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl*)_cpssp;
	assert(arch <= cpssp->max_comp_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Initialization of foreign architecture %d requested."
		"Skipping.\n", arch);
}


void
glue_vhdl_comp_port_connect(
	void *_cpssp,
	unsigned int comp,
	const char *port,
	unsigned int sig
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(comp <= cpssp->max_comp_id);
	assert(sig <= cpssp->max_sig_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Foreign port %s of component %d request to connect to "
		"signal %d. Skipping\n", port, comp, sig);
}

void
glue_vhdl_arch_port_connect(
	void *_cpssp,
	unsigned int arch,
	const char *port,
	unsigned int sig
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(arch <= cpssp->max_arch_id);
	assert(sig <= cpssp->max_sig_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Foreign port %s of architecture %d request to connect to "
		"signal %d. Skipping\n", port, arch, sig);
}

void
glue_vhdl_comp_generic_nonarray_set(
	void *_cpssp,
	unsigned int comp,
	const char *generic,
	union fauhdli_value val,
	const char *type
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(comp <= cpssp->max_comp_id);
	
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Setting foreign generic %s of component %d requested. "
		"Skipping.\n", generic, comp);
}

void
glue_vhdl_arch_generic_nonarray_set(
	void *_cpssp,
	unsigned int arch,
	const char *generic,
	union fauhdli_value val,
	const char *type
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(arch <= cpssp->max_arch_id);
	
	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Setting foreign generic %s of architecture %d requested. "
		"Skipping.\n", generic, arch);
}


void
glue_vhdl_comp_generic_array_set(
	void *_cpssp,
	unsigned int comp,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(comp <= cpssp->max_comp_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Setting foreign generic %s of component %d requested. "
		"Skipping.\n", generic, comp);
}

void
glue_vhdl_arch_generic_array_set(
	void *_cpssp,
	unsigned int arch,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
)
{
	struct glue_vhdl *cpssp = (struct glue_vhdl *)_cpssp;
	assert(cpssp != NULL);
	assert(arch <= cpssp->max_arch_id);

	fauhdli_log(FAUHDLI_LOG_WARNING, "fauhdli", "glue-vhdl",
		"Setting foreign generic %s of architecture %d requested. "
		"Skipping.\n", generic, arch);
}


void *
glue_vhdl_create(void)
{
	struct glue_vhdl *ret = malloc(sizeof(struct glue_vhdl));
	ret->max_sig_id = 0;
	ret->max_comp_id = 0;
	ret->max_arch_id = 0;
	return ret;
}

void
glue_vhdl_destroy(void *_cpssp)
{
	assert(_cpssp != NULL);
	free(_cpssp);
}
