/*
 * FALCON - The Falcon Programming Language
 * FILE: zlib.cpp
 *
 * zlib module main file
 * -------------------------------------------------------------------
 * Author: Jeremy Cowgar
 * Begin: Thu Jan 3 2007
 * -------------------------------------------------------------------
 * (C) Copyright 2008: the FALCON developers (see list in AUTHORS file)
 *
 * See LICENSE file for licensing details.
 * In order to use this file in it's compiled form, this source or
 * part of it you have to read, understand and accept the conditions
 * that are stated in the LICENSE file that comes bundled with this
 * package.
 */

/**
 * \file
 * This module exports zlib and module loader facility to falcon
 * scripts.
 */

#include <falcon/module.h>
#include "zlib_ext.h"
#include "zlib.h"
#include "zlib_st.h"

#include "version.h"
/*#
   @module feather_zlib ZLib
   @brief Compress/uncompress functions (zlib binding).

   This module provides an essential interface to the Zlib compression routines.

   The greatest part of the functionalites of the module are encapsulated in the
   @a ZLib class, which provided some class-wide methods to compress and uncompress
   data.

   The following example can be considered a minimal usage pattern:
   @code
   load zlib

   original = "Mary had a little lamb, it's fleece was white as snow."
   > "Uncompressed: ", original

   comped = ZLib.compressText( original )
   > "Compressed then uncompressed:"
   > "   ", ZLib.uncompressText( comped )

   @endcode

   The module declares also a @b ZLibError that is raised in case of
   failure in compression/decompression operations.

   @beginmodule feather_zlib
*/


FALCON_MODULE_DECL
{
   #define FALCON_DECLARE_MODULE self

   // setup DLL engine common data

   Falcon::Module *self = new Falcon::Module();
   self->name( "zlib" );
   self->language( "en_US" );
   self->engineVersion( FALCON_VERSION_NUM );
   self->version( VERSION_MAJOR, VERSION_MINOR, VERSION_REVISION );

   //====================================
   // Message setting
   #include "zlib_st.h"

   //====================================
   // Error enumeration

   /*#
      @enum ZLibErrorCode
      @brief Error codes generated by ZLib methods.


      In case of ZLibError raised by the ZLib module
      function and methods, the @a ZLibError.code property
      assumes one of this values:
      - @b errno: Generic/unknown error.
      - @b stream: Error in reading the ZLIB data.
      - @b data: Error in the compression buffer.
      - @b memory: Not enough memory to complete the operation.
      - @b version: Version of this library is not able to handle
         the given compressed buffer.
   */
   Falcon::Symbol *c_errcode = self->addClass( "ZLibErrorCode" );
   self->addClassProperty( c_errcode, "errno")
      .setInteger( FALCON_ZLIB_ERROR_BASE - Z_ERRNO );
   self->addClassProperty( c_errcode, "stream")
      .setInteger( FALCON_ZLIB_ERROR_BASE - Z_STREAM_ERROR );
   self->addClassProperty( c_errcode, "data")
      .setInteger( FALCON_ZLIB_ERROR_BASE - Z_DATA_ERROR );
   self->addClassProperty( c_errcode, "memory")
      .setInteger( FALCON_ZLIB_ERROR_BASE - Z_BUF_ERROR );
   self->addClassProperty( c_errcode, "version")
      .setInteger( FALCON_ZLIB_ERROR_BASE - Z_VERSION_ERROR );


   //====================================
   // ZLib class

   Falcon::Symbol *c_zlib = self->addClass( "ZLib" );
   self->addClassMethod( c_zlib, "compress", Falcon::Ext::ZLib_compress ).asSymbol()->
      addParam("buffer");
   self->addClassMethod( c_zlib, "uncompress", Falcon::Ext::ZLib_uncompress ).asSymbol()->
      addParam("buffer");
   self->addClassMethod( c_zlib, "compressText", Falcon::Ext::ZLib_compressText ).asSymbol()->
      addParam("text");
   self->addClassMethod( c_zlib, "uncompressText", Falcon::Ext::ZLib_uncompressText ).asSymbol()->
      addParam("buffer");
   self->addClassMethod( c_zlib, "getVersion", Falcon::Ext::ZLib_getVersion );

   //============================================================
   // ZlibError class
   Falcon::Symbol *error_class = self->addExternalRef( "Error" ); // it's external
   Falcon::Symbol *procerr_cls = self->addClass( "ZLibError", Falcon::Ext::ZLibError_init )
      ->addParam( "code" )->addParam( "description")->addParam( "extra" );
   procerr_cls->setWKS( true );
   procerr_cls->getClassDef()->addInheritance(  new Falcon::InheritDef( error_class ) );

   return self;
}

/* end of zlib.cpp */


