# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# ssh_agent.py - Run an ssh agent when needed

import falcon
from gettext import gettext as _
import os, signal

class SshAgentPlugin(falcon.plugin.FalconPlugin):
    """SSH Agent controller plugin"""
    name = _("SSH Agent")
    desc = _("Start/stop an SSH agent before syncing if needed")
    api  = (1,1)

    def pre_action(self, action, args):
        # Do we need an agent?
        self.started_agent = False
        need_agent = False
        if action == 'sync':
            for m in falcon.mirror.Mirror.objects.all():
                if ':' in m.rsync and not 'rsync://' in m.rsync and not '::' in m.rsync:
                    need_agent = True
                    break
        elif action in ('build', 'build-queue'):
            need_agent = True # We only support scp/ssh buildds for now
        if need_agent:
            # Do we have SSH keys?
            if not os.path.exists('%s/.ssh/id_rsa' % os.getenv('HOME')) and \
                not os.path.exists('%s/.ssh/id_dsa' % os.getenv('HOME')) and \
                not os.path.exists('%s/.ssh/identity' % os.getenv('HOME')):
                falcon.util.warning(_("Couldn't find SSH key - falcon may ask for your password a lot"))
                return
            # Do we need to start/kill it
            if not os.getenv('SSH_AUTH_SOCK'):
                self.started_agent = True
                falcon.util.output(_('Setting up ssh-agent'))
                env = falcon.util.run(['ssh-agent']).split()
                for var in env:
                    if '=' in var:
                        (key, val) = var.split('=')
                        if key == 'SSH_AGENT_PID':
                            self.agent_pid = int(val[:-1])
                        os.environ[key] = val[:-1]
            # Shall we add our keys?
            try:
                ret = falcon.util.run(['ssh-add', '-l'])
            except RuntimeError:
                falcon.util.output(_('Trying ssh-add'))
                falcon.util.run(['ssh-add'],buffer=False)
            
    def post_action(self, action, args):
        if self.started_agent:
            falcon.util.output(_("Killing SSH agent (%d)") % self.agent_pid)
            os.kill(self.agent_pid, signal.SIGKILL)
