# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# falcon - Library for falcon et al with packaging functions

# Set up bugcatching very early. When crashes occur, reset the terminal if it changed
import sys, termios, errno, os, traceback
termattr = termios.tcgetattr(0)
def except_hook(etype, evalue, etraceback):
    """Custom exception hook that provides info for bug filing"""
    new_termattr = termios.tcgetattr(0)
    if new_termattr != termattr:
        termios.tcsetattr(0,termios.TCSANOW, termattr)
        os.system('tput init')
    if issubclass(etype, SystemExit) or issubclass(etype, KeyboardInterrupt):
        return
    if '_' not in locals():
        _ = lambda x: x
    if issubclass(etype, OSError) and evalue.args[0] == errno.EACCES:
        print _("E: You appear to lack privileges to manage this repository")
        print str(evalue)
        return
    if etype.__name__ == 'OperationalError' and 'locked' in evalue:
        print _("E: The falcon database is locked, maybe due to a previous crash")
        print _("E: Delete the data-journal file in .falcon if this is true")
        return

    # If it's a bug in falcon, say so
    file = etraceback.tb_frame.f_code.co_filename
    path = os.path.dirname(sys.argv[0])
    our_files = [os.path.join(path, x) for x in ('falcon', 'falcon-import', 'falcon-build-queue')]
    path = os.path.dirname(__file__)
    trace = traceback.format_exception(etype, evalue, etraceback)
    if file in our_files or file.startswith(path):
        print _("""\n === ERROR! ===

An error occured in %s. This is probably a bug in the software.
Please file a bug at https://launchpad.net/falcon/+filebug and
include the following backtrace:
""" % os.path.basename(sys.argv[0]))
    print ''.join(trace)
sys.excepthook = except_hook

import falcon.config
# falcon.config initializes the django configuration as well, so we can now import everything
import falcon.questions as questions
import falcon.validators as validators
import falcon.plugin as plugin
import falcon.util as util
import falcon.iso as iso
import falcon.bc as bc

import fcntl, time, shelve, warnings
from gettext import gettext as _
# Add our picklefield
from django.db import models as _models, get_creation_module, connection
from django.db.models.fields import Field
import django.conf
from django.core import management
import cPickle as pickle

class PickleField(Field):
    """Custom database field with automatic pickling"""
    def to_python(self, value):
        if type(value) == str:
            # Allow for unpickling failures for db upgrading
            try:
                return pickle.loads(value)
            except:
                pass
        return value
    def get_db_prep_lookup(self, value):
        return Field.get_db_prep_lookup(self, pickle.dumps(value))
    def get_db_prep_save(self, value):
        return Field.get_db_prep_save(self, pickle.dumps(value))
_models.PickleField = PickleField
m = get_creation_module()
m.DATA_TYPES['PickleField'] = 'text'

# These need the picklefield, so import after defining it
import falcon.pocket as pocket
import falcon.package as package
import falcon.mirror as mirror
import falcon.build as build

lockfd = None
screen = None
def lock():
    """Lock the current repository"""
    global lockfd
    lockfd = open(os.path.join('.falcon','lock'),'a+')
    try:
        fcntl.flock(lockfd,fcntl.LOCK_EX | fcntl.LOCK_NB)
        lockfd.truncate(0)
        lockfd.write(str(os.getpid()))
        lockfd.flush()
    except IOError:
        lockfd.seek(0)
        pid = lockfd.read()
        if not pid or not os.path.exists(os.path.join('/proc', pid)):
            # Removing stale lock
            lockfd.close()
            os.unlink(os.path.join('.falcon','lock'))
            lock()
        else:
            util.error(_("Repository is locked by another process (pid %s)") % pid)

def unlock():
    """Unlock the current repository"""
    global lockfd
    if lockfd:
        fcntl.flock(lockfd,fcntl.LOCK_UN)

import types
import falcon.models as models
conf = config.conf
def init_rootdir():
    """Perform sanity checks on the repository root and initialize database
       tables if needed"""
    # Chdir into the rootdir to avoid some os.path.join calls....
    os.chdir(conf.rootdir)

    # Check configuration database
    if not os.path.exists('.falcon'):
        os.mkdir('.falcon', 0700)
    else:
        # Previous versions of falcon created it world-readable. Fix that.
        os.chmod('.falcon', 0700)
    falcon.lock()

    # Sanity check
    if not os.path.isdir('pool'):
        util.error(_("Pool directory '%s' does not exist") % os.path.join(conf.rootdir, 'pool'))
    for d in ('morgue', 'changelogs', 'build', os.path.join('build','building'), 'incoming', os.path.join('build','logs')):
        if not os.path.isdir(os.path.join('.falcon', d)):
            os.mkdir(os.path.join('.falcon', d))

    # Configure django
    if django.conf.settings.DATABASE_ENGINE == 'sqlite3':
        setattr(django.conf.settings, 'DATABASE_NAME',  os.path.join('.falcon','data'))

    # Create databases
    management.disable_termcolors()
    all_sql = []
    all_models = [x for x in falcon.models.__dict__.values() if type(x) == django.db.models.base.ModelBase]
    for model in all_models:
        sql, ref = management._get_sql_model_create(model, all_models)
        all_sql += sql
        sql = management._get_many_to_many_sql_for_model(model)
        all_sql += sql
    cursor = connection.cursor()
    # Sigh, mysql, I'm telling you that it's ok if it exists....
    warnings.filterwarnings("ignore", message='Table .* already exists')
    for s in all_sql:
        cursor.execute(s.replace('CREATE TABLE', 'CREATE TABLE IF NOT EXISTS'))

    # Register our variables
    conf.register('my_database_abi', conf.database_abi)
    if conf.my_database_abi != conf.database_abi:
        falcon.bc.upgrade_db()

    conf.register('description', _('Falcon generated repository'),
                  questions.Entry(_("Please enter a short description of your repository")))
    conf.register('origin', 'falcon',
                  questions.Entry(_("Please enter a one-word origin for your repository (e.g. your nickname)"), [validators.nospaces]))
    conf.register('gpgkey', '',
                  questions.Entry(_("Please enter your 8-character GPG id"), [validators.blank_ok(validators.hex)]))
    conf.register('bugs', '', questions.Entry(_("Where can users file bugs about your packages")))
    all_architectures = ('i386','amd64','powerpc','sparc','arm','ia64','alpha','hppa','m68k','mips','mipsel','s390')
    conf.register('architectures', [conf.architecture],
                  questions.SelectMulti(_("Which architectures do you want to support"), all_architectures, [validators.atleastone]))
    conf.register('webbase', '',
                  questions.Entry(_("What is the webroot of your repository (Leave blank to disable html pages)"), [validators.blank_ok(validators.url)]))
    templatedirs = [os.path.join(conf.prefix,'share','falcon','templates'),
                    os.path.join(os.getenv('HOME'), '.falcon', 'templates'),
                    os.path.join('.falcon', 'templates')]
    templates = []
    _templatedirs = {}
    for dir in templatedirs:
        if os.path.exists(dir):
            ts = os.listdir(dir)
            templates += ts
            for template in ts:
                _templatedirs[template] = dir
    templatedirs = _templatedirs
    conf.register('template', 'default',
                  questions.Select(_("Which template do you want to use"), set(templates)))
    conf.register('long_description', '',
                  questions.Text(_("Please enter a description for on the webpage")))
    conf.register('create_filelist',True,
                  questions.Boolean(_("Create content listings")))
    conf.register('configured', False, None)
    conf.register('lastinstall', time.time(), None)
    conf.register('lastexport', time.time(), None)
    
    # Load/create pockets
    falcon.pockets = []
    for p in sorted(os.listdir('pool')):
        if os.path.isdir(os.path.join('pool',p)):
            pockets.append(pocket.Pocket.objects.get_or_create(name=p)[0])
            pockets[-1].load_components()
    # Remove pockets from the database that no longer exist
    for p in pocket.Pocket.objects.all():
        if not os.path.exists(p.poolpath):
            p.delete()

    # Load plugins
    plugin.load_plugins()

    # More django config (needs database)
    if conf.template:
        setattr(django.conf.settings, 'TEMPLATE_DIRS', (os.path.join(templatedirs[conf.template],conf.template),))
