/*
    ext2_unix_io.c -- ext2 unix I/O code
    Copyright (C) 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_unix_io_c[] = "$Id: ext2_unix_io.c,v 1.15 2004/09/30 14:04:04 sct Exp $";

#include "config.h"

#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "ext2.h"

#ifndef O_LARGEFILE
#define O_LARGEFILE 0
#endif

/* pseudo-header.... */

#ifndef BLKGETSIZE
#define BLKGETSIZE _IO(0x12,96)
#endif

#ifndef BLKGETSIZE64
#define BLKGETSIZE64 _IOR(0x12,114,sizeof(unsigned long long))
#endif

struct my_cookie
{
	int logsize;

	int fdread;
	int fdwrite;
	int fdroot;
	loff_t readoff;
	loff_t writeoff;
};

/* ...then this must be pseudo-code  :-) */


static void do_close(void *cookie)
{
	struct my_cookie *monster = cookie;

	fdatasync(monster->fdwrite);
	close(monster->fdread);
	close(monster->fdwrite);
	if (monster->fdroot >= 0)
		close(monster->fdroot);

	monster->fdread = -1;
	monster->fdwrite = -1;
	monster->fdroot = -1;
}

static void do_sync(void *cookie)
{
	struct my_cookie *monster = cookie;

	fdatasync(monster->fdwrite);
}

static int valid_offset(int fd, __loff_t offset)
{
	char c;

	if (ext2_llseek(fd, offset, SEEK_SET) < 0 || read(fd, &c, 1) < 1)
		return 0;

	return 1;
}

static blk_t do_get_size(void *cookie)
{
	struct my_cookie *monster = cookie;
	blk_t size;
#ifdef BLKGETSIZE64
	unsigned long long lsize;

	if (ioctl(monster->fdread, BLKGETSIZE64, &lsize) >= 0)
	{
		lsize >>= monster->logsize;
		return lsize;
	}
#endif /* BLKGETSIZE64 */
	/* Thanks to Rolf Jakob for this one (can't do SEEK_END on block dev) */
#ifdef BLKGETSIZE
	if (ioctl(monster->fdread, BLKGETSIZE, &size) >= 0) {
		/* printf("ioctl device size is %u blocks\n", size); */
	} else
#endif /* BLKGETSIZE */
	/* Do a binary search to find partition size (from e2fsprogs) */
	{
		__loff_t low, high;

		for (low = 0, high = 1024;
		     valid_offset (monster->fdread, high);
		     high *= 2)
			low = high;

		while (low < high - 1) {
			__loff_t mid = (low + high) / 2;

			if (valid_offset (monster->fdread, mid))
				low = mid;
			else
				high = mid;
		}

		/* printf("llseek device size is %u blocks\n", size); */

		/* Reset filesystem position? */
		ext2_llseek(monster->fdread, monster->readoff, SEEK_SET);

		size = high >> 9;
	}

	return size >> (monster->logsize - 9);
}

static void do_direct_read(void *cookie, void *ptr, loff_t offset,
			   size_t numbytes)
{
	struct my_cookie *monster = cookie;
	size_t ret;

	if (monster->readoff != offset) {
		if (ext2_llseek(monster->fdread, offset, SEEK_SET) < 0) {
			fprintf(stderr, "error: %s: seeking to %jd\n",
				strerror(errno), offset);
			exit(99);
		}
	}

	if ((ret = read(monster->fdread, ptr, numbytes)) != numbytes) {
		fprintf(stderr, "error: %s: read %zd of %zd bytes at %jd\n",
			ret == 0 ? "End of Device" : strerror(errno), 
			ret, numbytes, offset);
		exit(99);
	}
	monster->readoff = offset + numbytes;
}

static void do_read(void *cookie, void *ptr, blk_t block, blk_t numblocks)
{
	struct my_cookie *monster = cookie;
	loff_t offset;

	offset = ((loff_t)block) << monster->logsize;
	do_direct_read(cookie, ptr, offset, numblocks<<monster->logsize);
}

static void do_set_blocksize(void *cookie, int logsize)
{
	struct my_cookie *monster = cookie;

	monster->logsize = logsize;
}

static void do_direct_write(void *cookie, void *ptr, loff_t offset,
			      size_t numbytes)
{
	struct my_cookie *monster = cookie;
	size_t ret;

	if (monster->writeoff != offset) {
		if (ext2_llseek(monster->fdwrite, offset, SEEK_SET) < 0) {
			fprintf(stderr, "error: %s: seeking to %jd\n",
				strerror(errno), offset);
			exit(99);
		}
	}

	if ((ret = write(monster->fdwrite, ptr, numbytes)) != numbytes) {
		fprintf(stderr, "error: %s: wrote %zd of %zd bytes at %jd\n",
			strerror(errno), ret, numbytes, offset);
		exit(99);
	}
	monster->writeoff = offset + numbytes;
}

static void do_write(void *cookie, void *ptr, blk_t block, blk_t numblocks)
{
	struct my_cookie *monster = cookie;
	loff_t offset;

	offset = ((loff_t)block) << monster->logsize;
	do_direct_write(cookie, ptr, offset, numblocks<<monster->logsize);
}

static int do_ioctl(void *cookie, int ioc, void *ptr)
{
	struct my_cookie *monster = cookie;

	return ioctl(monster->fdroot, ioc, ptr);
}


struct ext2_dev_ops ops =
{
	close:		do_close,
	get_size:	do_get_size,
	read:		do_read,
	set_blocksize:	do_set_blocksize,
	sync:		do_sync,
	write:		do_write,
	direct_read:	do_direct_read,
	direct_write:	do_direct_write,
	ioctl:		do_ioctl,
};


struct ext2_dev_handle *ext2_make_dev_handle_from_file(char *dev, char *dir,
						       char *prog)
{
	struct ext2_dev_handle *dh;
	struct my_cookie *monster;

	if ((dh = malloc(sizeof(struct ext2_dev_handle))) == NULL)
		goto error;

	if ((monster = malloc(sizeof(struct my_cookie))) == NULL)
		goto error_free_dh;

	if ((monster->fdread = open(dev, O_RDONLY | O_LARGEFILE)) < 0) {
		char msg[4097];

		snprintf(msg, 4096, "%s: reading %s", prog, dev);
		perror(msg);
		goto error_free_cookie;
	}

	if ((monster->fdwrite = open(dev, O_WRONLY | O_LARGEFILE)) < 0) {
		char msg[4097];

		snprintf(msg, 4096, "%s: writing %s", prog, dev);
		perror(msg);
		goto error_close_read;
	}

	if (dir) {
#if 0
		/*
		 * If we are opening a mounted filesystem, we need a file
		 * therein for sending ioctls to the filesystem.  We unlink
		 * it immediately so that we don't have to clean it up later.
		 */
		char tmp[4097];

		snprintf(tmp, 4096, "%s/ioctlXXXXXX", dir);
		tmp[4097] = '\0';
		if ((monster->fdroot = mkstemp(tmp)) < 0) {
			fprintf(stderr, "%s: %s: %s for temp resize file",
				prog, dev, strerror(errno));
			goto error_close_write;
		}
		unlink(tmp);
#endif
		if ((monster->fdroot = open(dir, O_RDONLY)) < 0) {
			fprintf(stderr, "%s: %s: %s", prog, dev,
				strerror(errno));
			goto error_close_write;
		}
	} else
		monster->fdroot = -1;

	monster->readoff = 0;
	monster->writeoff = 0;
	monster->logsize = 9;

	dh->ops = &ops;
	dh->cookie = monster;
	dh->prog = prog;

	return dh;

 error_close_write:
	close(monster->fdread);

 error_close_read:
	close(monster->fdread);

 error_free_cookie:
	free(monster);

 error_free_dh:
	free(dh);

 error:
	return NULL;
}
