/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include "exrinput.h"
#include "exroutput.h"
#include "gaussian.h"

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    double *inputimg;
    double *outputimg;
    int width, height;
    double scale;
    int y;

    if( argc < 4 ) {
        fprintf( stderr, "Usage: exrblur input.exr output.exr scalefactor\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrblur: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrblur: %s [%dx%d]\n", argv[ 1 ], width, height );

    inputimg = malloc( width * height * 4 * sizeof( double ) );
    outputimg = malloc( width * height * 4 * sizeof( double ) );
    if( !inputimg || !outputimg ) {
        fprintf( stderr, "exrblur: Cannot allocate memory.\n" );
        if( inputimg ) free( inputimg );
        if( outputimg ) free( outputimg );
        exrinput_delete( exrin );
        return 1;
    }

    exrout = exroutput_new( argv[ 2 ], width, height );
    if( !exrout ) {
        exrinput_delete( exrin );
        fprintf( stderr, "exrblur: Cannot open %s.\n", argv[ 2 ] );
        return 1;
    }

    scale = atof( argv[ 3 ] );
    fprintf( stderr, "exrblur: Gaussian size %.3f of image width == %d pixels.\n",
             scale, (int) (((double) width) * scale) );

    for( y = 0; y < height; y++ ) {
        fprintf( stderr, "exrblur: Reading input %d\r", y );
        memcpy( inputimg + (y * width * 4), exrinput_get_scanline( exrin, y ),
                width * 4 * sizeof( double ) );
    }
    fprintf( stderr, "\nexrblur: Done, starting blur.\n" );
    exrinput_delete( exrin );

    gaussian_blur( outputimg, inputimg, width, height, (((double) width) * scale) / 2.0 );

    for( y = 0; y < height; y++ ) {
        fprintf( stderr, "exrblur: Writing scanline %d.\r", y );
        exroutput_scanline( exrout, outputimg + (y * width * 4) );
    }
    fprintf( stderr, "\nexrblur: Done.\n" );
    exroutput_delete( exrout );
    return 0;
}

