/*
 * Copyright (c) 2005 Sandia Corporation. Under the terms of Contract
 * DE-AC04-94AL85000 with Sandia Corporation, the U.S. Governement
 * retains certain rights in this software.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 * 
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.  
 * 
 *     * Neither the name of Sandia Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
/*****************************************************************************
*
* exgpem - ex_get_partial_elem_map
*
* entry conditions - 
*   input parameters:
*       int     exoid                   exodus file id
*       int     map_id                  element map id
*       int     ent_start               first entry in map
*       int     ent_count               number of entries in map
*
* exit conditions - 
*       int*    elem_map                element map
*
* revision history - 
*
*
*****************************************************************************/

#include <stdlib.h>
#include "exodusII.h"
#include "exodusII_int.h"

/*
 * reads the element map with specified ID
 */

int ex_get_partial_elem_map (int  exoid,
			     int  map_id,
			     int  ent_start,
			     int  ent_count, 
			     int *elem_map)
{
  int dimid, var_id, id_ndx, status;
  size_t num_elem, start[1], count[1]; 
  char errmsg[MAX_ERR_LENGTH];

  exerrval = 0; /* clear error code */

  /* See if file contains any elements...*/
  if (nc_inq_dimid (exoid, DIM_NUM_ELEM, &dimid) != NC_NOERR) {
    return (EX_NOERR);
  }

  if ((status = nc_inq_dimlen(exoid, dimid, &num_elem)) != NC_NOERR) {
    exerrval = status;
    sprintf(errmsg,
            "Error: failed to get number of elements in file id %d", exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  /* Check input parameters for a valid range of numbers */
  if (ent_start <= 0 || ent_start > (int)num_elem) {
    exerrval = EX_FATAL;
    sprintf(errmsg,
	    "Error: start count is invalid in file id %d",
	    exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  if (ent_count < 0) {
    exerrval = EX_FATAL;
    sprintf(errmsg,
	    "Error: Invalid count value in file id %d",
	    exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  if (ent_start+ent_count-1 > (int)num_elem) {
    exerrval = EX_FATAL;
    sprintf(errmsg,
	    "Error: start+count-1 is larger than element count in file id %d",
	    exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  /* first check if any element maps have been defined */
  if ((status = nc_inq_dimid (exoid, DIM_NUM_EM, &dimid)) != NC_NOERR) {
    exerrval = status;
    sprintf(errmsg,
            "Warning: no element maps defined in file id %d",
	    exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_WARN);
  }

  /* Lookup index of element map id property array */
  id_ndx = ex_id_lkup(exoid,EX_ELEM_MAP,map_id);
  if (exerrval != 0) {
    sprintf(errmsg,
	    "Error: failed to locate element map id %d in %s in file id %d",
	    map_id,VAR_EM_PROP(1),exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  /* inquire id's of previously defined dimensions and variables */
  if ((status = nc_inq_varid(exoid, VAR_ELEM_MAP(id_ndx), &var_id)) != NC_NOERR) {
    exerrval = status;
    sprintf(errmsg,
            "Error: failed to locate element map %d in file id %d",
	    map_id,exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }

  /* read in the element map */
  start[0] = ent_start-1;
  count[0] = ent_count;

  status = nc_get_vara_int(exoid, var_id, start, count, elem_map);

  if (status == -1) {
    exerrval = status;
    sprintf(errmsg,
            "Error: failed to get element map in file id %d",
	    exoid);
    ex_err("ex_get_partial_elem_map",errmsg,exerrval);
    return (EX_FATAL);
  }
  return (EX_NOERR);
}
