""" Demo of dynamic enabling and disabling of trait editors in an interface.

    Code sample showing a simple implementation of the dynamic 
    enabling and disabling of trait attribute editor interfaces 
    on the basis of another trait attribute's assigned value.

    Demo class "Person" has attributes that apply to all instances 
    ('first_name', 'last_name', 'age') and attributes that are 
    specific to age group ('marital_status' and 'registered_voter'
    for adults, 'legal_guardian' for children.)  The adult-specific 
    attributes are disabled if 'age' is less than 18; otherwise
    'legal_guardian' is disabled.

    NOTE: The 'enabled_when' expression for a given attribute must
    be a condition on some attribute, e.g. 'object.is_adult', so
    that the evaluation is triggered by its trait handler.
    Also valid are expressions such as 'age >= 18' but this is to be 
    avoided as it violates MVC -- mixes model logic, determination of 18 of
    adulthood, with view presentation.
"""

from enthought.traits.api \
    import HasTraits, Str, Range, Enum, Bool, Instance, Property
    
from enthought.traits.ui.api \
    import Group, View, Item, Include

    
class PersonView ( HasTraits ):
    """ Traits View for the Person class. """
    
    # Interface for attributes that are always enabled:
    _gen_group = Group( Item( name = 'first_name' ),
                        Item( name = 'last_name' ),
                        Item( name = 'age' ), 
                        label       = 'General Info', 
                        show_border = True )

    # Interface for adult-only attributes:
    _adult_group = Group( Item( name = 'marital_status' ), 
                          Item( name = 'is_registered_voter' ),
                          enabled_when = 'is_adult',
                          label        = 'Adults', 
                          show_border  = True )

    # Interface for child-only attribute:
    _child_group = Group( Item( name = 'legal_guardian', 
                          enabled_when = 'not is_adult' ),
                          label        ='Minors', 
                          show_border  = True )
    
    view = View( Group( _gen_group, _adult_group, _child_group ) )


class Person ( HasTraits ):
    """ Demo class for demonstrating enabling/disabling of trait editors.
    """
    # Person's first name:
    first_name = Str
    
    # Person's surname:
    last_name = Str
    
    # Person's age in years:
    age = Range( 0, 120 )
    
    # Marital status is valid when the person is an adult:
    marital_status = Enum( 'single', 'married', 'divorced', 'widowed' )

    # Registered voter flag is valid when the person is an adult:
    is_registered_voter = Bool
    
    # Name of the legal guardian; valied when person is not an adult:
    legal_guardian = Str

    # Is the person considered a minor or an adult?
    is_adult = Property

    # Trait properties:
    def _get_is_adult ( self, value ):
        """ Person is considered an adult if age is > 17. """
        return self.age > 17
        

if __name__ == "__main__":
    PersonView().configure_traits( context = Person() )
    
