#-------------------------------------------------------------------------------
#
#  Defines the OMComponent class of the Enable 'om' (Object Model) package.
#
#  The OMComponent class defines the base class for visual elements
#  representing objects in the underlying object model.
#
#  Written by: David C. Morrill
#
#  Date: 01/27/2005
#
#  (c) Copyright 2005 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.util.numerix import array

from om_base             import om_handler
from om_traits           import OffsetXY, SizeXY, StyleDelegate, \
                                SelectionBackgroundColor, SelectionBorderColor,\
                                InnerComponentPosition
from om_outer_component  import OMOuterComponent
from om_contact_manager  import OMContactManager

from enthought.enable    import Component, BaseContainer

from enthought.traits.api    import HasStrictTraits, Instance, Property, false, \
                                Any
from enthought.traits.ui.api import View

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

SelectionLineDash = array( [ 4.0, 4.0 ] )

#-------------------------------------------------------------------------------
#  Data:
#-------------------------------------------------------------------------------

# The default, singleton, OMComponentController object:
_default_component_controller = None

#-------------------------------------------------------------------------------
#  'OMComponentStyle' class:
#-------------------------------------------------------------------------------

class OMComponentStyle ( HasStrictTraits ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # Color used to draw the background of a selected component:
    selection_bg_color       = SelectionBackgroundColor

    # Color used to draw the border of a selected component:
    selection_border_color   = SelectionBorderColor

    # Position of the inner component relative to the outer component:
    inner_component_position = InnerComponentPosition

    # Context menu:
    menu = Property

#-- Property implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'menu' property (normally overridden):
    #---------------------------------------------------------------------------

    def _get_menu ( self ):
        return self.get_menu()

    def get_menu ( self ):
        return None

# Create a default component style:
default_component_style = OMComponentStyle()

#-------------------------------------------------------------------------------
#  'OMComponent' class:
#-------------------------------------------------------------------------------

class OMComponent ( Component, BaseContainer ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # The controller for the component:
    controller         = Instance( 'enthought.enable.om.OMComponentController' )

    # The style to use for drawing the component:
    style              = Instance( OMComponentStyle, default_component_style )

    # Context menu:
    menu               = StyleDelegate

    # Color used to draw the background of a selected component:
    selection_bg_color = StyleDelegate

    # Color used to draw the border of a selected component:
    selection_border_color   = StyleDelegate

    # Position of the inner component relative to the outer component:
    inner_component_position = StyleDelegate

    # The 'inner component' for the component:
    inner_component = Instance( Component )

    # The 'outer component' for the component:
    outer_component = Instance( OMOuterComponent )

    # The contact manager for the component:
    contact_manager = Instance( OMContactManager )

    # The location and size of the component (override of Component definition):
    bounds   = Property

    # Origin of the component relative to its containing canvas:
    origin   = OffsetXY

    # Size of the component:
    size     = SizeXY

    # Is the component selected?
    selected = false

    # List of all links connected to this component:
    links    = Property

    # List of all contacts associated with this component:
    contacts = Property

    # Component data (to help controller relate component back to object model):
    data     = Any

    #---------------------------------------------------------------------------
    #  Traits view definitions:
    #---------------------------------------------------------------------------

    traits_view = View( [ 'outer_component@', '|{Component}<>' ],
                        [ 'inner_component@', '|{Label}<>' ],
                        title   = 'Edit component properties',
                        handler = om_handler )

#-- Default Trait Value Handlers -----------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the default value for the 'controller' trait:
    #---------------------------------------------------------------------------

    def _controller_default ( self ):
        global _default_component_controller

        if _default_component_controller is None:
            from om_component_controller import OMComponentController
            _default_component_controller = OMComponentController()
        return _default_component_controller

#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'bounds' property:
    #---------------------------------------------------------------------------

    def _get_bounds ( self ):
        if self._bounds is None:
            x, y, dx, dy = self.container.bounds
            ox, oy       = self.origin
            self._bounds = ( x + ox, y + oy ) + self.size
        return self._bounds

    def _set_bounds ( self, bounds ):
        x, y, dx, dy        = self._bounds = bounds
        cx, cy, cdx, cdy    = self.container.bounds
        self.origin         = ( int( x - cx ), int( y - cy ) )
        self.size           = ( int( dx ), int( dy ) )
        self._drag_location = None

    #---------------------------------------------------------------------------
    #  Implementation of the 'links' property:
    #---------------------------------------------------------------------------

    def _get_links ( self ):
        return self.contact_manager.links

    #---------------------------------------------------------------------------
    #  Implementation of the 'contacts' property:
    #---------------------------------------------------------------------------

    def _get_contacts ( self ):
        return self.contact_manager.contacts

#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Allows the user to drag the component around its containing canvas:
    #---------------------------------------------------------------------------

    def drag ( self, event ):
        self.window.mouse_owner = None
        canvas     = self.container
        components = [ self ]
        if self.selected:
            components = canvas.selection[:]

        canvas.redraw()
        for component in components:
            for link in component.links:
                link.visible = False

        # Save the drag information so we can inform the canvas controller when
        # the drag is complete:
        canvas._drag_components = components
        canvas._drag_bounds     = [ c.bounds for c in components ]

        self.window.drag( components, canvas, event, alpha = -1.0 )

    #---------------------------------------------------------------------------
    #  Edit the properties of the component:
    #---------------------------------------------------------------------------

    def edit ( self ):
        self.edit_traits()

    #---------------------------------------------------------------------------
    #  Force the component to be updated:
    #---------------------------------------------------------------------------

    def update ( self, layout = False ):
        if layout:
            self.contact_manager.layout_contacts()
        self.redraw()

#-- Event Handlers -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Handles the 'origin' trait being changed:
    #---------------------------------------------------------------------------

    def _origin_changed ( self ):
        self._bounds = None

    #---------------------------------------------------------------------------
    #  Handles the 'size' trait being changed:
    #---------------------------------------------------------------------------

    def _size_changed ( self, old, new ):
        self._bounds = None

    #---------------------------------------------------------------------------
    #  Handles the 'outer_component' trait being changed:
    #---------------------------------------------------------------------------

    def _outer_component_changed ( self, oc ):
        if oc is not None:
            oc.container = self

    #---------------------------------------------------------------------------
    #  Handles the 'inner_component' trait being changed:
    #---------------------------------------------------------------------------

    def _inner_component_changed ( self, ic ):
        if ic is not None:
            ic.container = self

    #---------------------------------------------------------------------------
    #  Handles the 'contact_manager' trait being changed:
    #---------------------------------------------------------------------------

    def _contact_manager_changed ( self, cm ):
        if cm is not None:
            cm.component = self

#-- Overridden Methods ---------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Draws the component in a specified graphics context:
    #---------------------------------------------------------------------------

    def _draw ( self, gc ):
        # If the component is selected, draw the selection marker:
        if self.selected:
            gc.save_state()
            gc.set_fill_color( self.selection_bg_color_ )
            gc.set_stroke_color( self.selection_border_color_ )
            gc.set_line_dash( SelectionLineDash )
            gc.begin_path()
            x, y, dx, dy = self.bounds
            gc.rect( x + 0.5, y + 0.5, dx - 1.0, dy - 1.0 )
            gc.draw_path()
            gc.restore_state()

        self.outer_component.draw( gc )
        if self.inner_component is not None:
            self.inner_component.draw( gc )
        self.contact_manager.draw( gc )

    #---------------------------------------------------------------------------
    #  Returns the components at a specified (x,y) point:
    #---------------------------------------------------------------------------

    def _components_at ( self, x, y ):
        result = (self.outer_component.components_at( x, y ) +
                  self.inner_component.components_at( x, y ) +
                  self.contact_manager.components_at( x, y ))
        if len( result ) > 0:
            result[ 0: 0 ] = [ self ]
        return result
