#-------------------------------------------------------------------------------
#
#  Define an Annotater component that allows a user to annotate an underlying 
#  component.
#
#  Written by: David C. Morrill
#
#  Date: 10/29/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Annotater
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.traits.api    import Trait, TraitPrefixList, Event
from enthought.traits.ui.api import Group, View, Include

from component           import Component
from enable_traits       import color_trait

#-------------------------------------------------------------------------------
#  Constants: 
#-------------------------------------------------------------------------------

rect        = 'rectangular'
style_trait = Trait( rect, TraitPrefixList( [ rect, 'freehand' ] ) )

#-------------------------------------------------------------------------------
#  'Annotater' class:
#-------------------------------------------------------------------------------

class Annotater ( Component ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------
    
    color      = Trait( ( 0.0, 0.0, 0.0, 0.2 ), color_trait )
    style      = style_trait
    annotation = Event
    
    #---------------------------------------------------------------------------
    #  Trait editor definition:
    #---------------------------------------------------------------------------
    
    traits_view = View( Group( '<component>', id = 'component' ),
                        Group( '<links>',     id = 'links' ),
                        Group( 'color', 'style', 
                               id    = 'annotater',
                               style = 'custom' ) )
    
    #---------------------------------------------------------------------------
    #  Mouse event handlers:
    #---------------------------------------------------------------------------
    
    def _left_down_changed ( self, event ):
        event.handled            = True
        self.window.mouse_owner  = self
        self._cur_x, self._cur_y = self._start_x, self._start_y = ( 
           event.x, event.y )
        
    def _left_up_changed ( self, event ):
        event.handled           = True
        self.window.mouse_owner = None
        if self.xy_in_bounds( event ):
            self.annotation = ( min( self._start_x, event.x ), 
                                min( self._start_y, event.y ),
                                abs( self._start_x - event.x ),
                                abs( self._start_y - event.y ) )
        self._start_x = self._start_y = self._cur_x = self._cur_y = None
        self.redraw()
        
    def _mouse_move_changed ( self, event ):        
        event.handled = True
        if self._start_x is not None:
            x = max( min( event.x, self.right - 1.0 ), self.x )  
            y = max( min( event.y, self.top   - 1.0 ), self.y )
            if (x != self._cur_x) or (y != self._cur_y):
                self._cur_x, self._cur_y = x, y
                self.redraw()
        
    #---------------------------------------------------------------------------
    #  Draw the contents of the control:
    #---------------------------------------------------------------------------
    
    def _draw ( self, gc ):
        if self._start_x is not None:
            gc.save_state()
            gc.set_fill_color( self.color_ )
            gc.begin_path()
            gc.rect( min( self._start_x, self._cur_x ), 
                     min( self._start_y, self._cur_y ),
                     abs( self._start_x - self._cur_x ),
                     abs( self._start_y - self._cur_y ) )
            gc.fill_path()
            gc.restore_state()
        
