#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <glib.h>

#include "../include/disk.h"

#include "edv_types.h"
#include "edv_obj.h"
#include "edv_utils.h"


/* Comparing */
gboolean EDVObjectCompareIndex(
	edv_object_struct *obj1,
	edv_object_struct *obj2
);
gboolean EDVObjectCompareIndexWithStat(
	edv_object_struct *obj,
	struct stat *stat_buf
);

/* Object Property */
void EDVObjectPropSet(
	edv_object_struct *obj,
	const gchar *name, const gchar *value,
	const gboolean create_as_needed
);
void EDVObjectPropRemove(
	edv_object_struct *obj,
	const gchar *name
);
edv_object_prop_struct *EDVObjectPropGet(
	edv_object_struct *obj, 
	const gchar *name
);
edv_object_prop_struct *EDVObjectPropNew(void);
edv_object_prop_struct *EDVObjectPropCopy(edv_object_prop_struct *prop);
void EDVObjectPropDelete(edv_object_prop_struct *prop);

/* Object */
edv_object_struct *EDVObjectNew(void);
edv_object_struct *EDVObjectCopy(edv_object_struct *obj);
void EDVObjectSetPath(
	edv_object_struct *obj,
	const gchar *path
);
void EDVObjectSetPath2(
	edv_object_struct *obj,
	const gchar *parent,
	const gchar *name
);
void EDVObjectSetStat(
	edv_object_struct *obj,
	struct stat *lstat_buf
);
void EDVObjectSetToStat(
	struct stat *lstat_buf,
	edv_object_struct *obj
);
void EDVObjectUpdateLinkFlags(edv_object_struct *obj);
void EDVObjectDelete(edv_object_struct *obj);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if the objects are the same by comparing their
 *	index values.
 */
gboolean EDVObjectCompareIndex(
	edv_object_struct *obj1,
	edv_object_struct *obj2
)
{
	if((obj1 == NULL) || (obj2 == NULL))
	    return(FALSE);

	if((obj1->device_index == obj2->device_index) &&
	   (obj1->index == obj2->index)
	)
	    return(TRUE);
	else
	    return(FALSE);
}

/*
 *	Checks if the object is the same as the one specified by
 *	the struct stat values.
 */
gboolean EDVObjectCompareIndexWithStat(
	edv_object_struct *obj,
	struct stat *stat_buf
)
{
	if((obj == NULL) || (stat_buf == NULL))
	    return(FALSE);

	if((obj->device_index == (gulong)stat_buf->st_dev) &&
	   (obj->index == (gulong)stat_buf->st_ino)
	)
	    return(TRUE);
	else
	    return(FALSE);
}


/*
 *	Sets the propert value.
 *
 *	The obj specifies the object that the properties list is on.
 *
 *	The name specifies the name of the property to set.
 *
 *	The value specifies the property value to set. If value is NULL
 *	or an empty string then the property will be removed.
 *
 *	If create_as_needed is TRUE and the property does not yet exist
 *	then it will be created.
 */
void EDVObjectPropSet(
	edv_object_struct *obj,
	const gchar *name, const gchar *value,
	const gboolean create_as_needed
)
{
	edv_object_prop_struct *prop;

	if((obj == NULL) || STRISEMPTY(name))
	    return;

	/* Remove the property? */
	if(STRISEMPTY(value))
	{
	    EDVObjectPropRemove(obj, name);
	    return;
	}

	/* Find the property that matches the specified name */
	prop = EDVObjectPropGet(obj, name);
	if(prop != NULL)
	{
	    /* Update the property name for case-insensitive corrections */
	    g_free(prop->name);
	    prop->name = STRDUP(name);
	}
	else
	{
	    /* No such property exists, do not create it? */
	    if(!create_as_needed)
		return;

	    /* Create the property */
	    prop = EDVObjectPropNew();
	    if(prop == NULL)
		return;

	    /* Set the new property's initial values and append it to
	     * the properties list
	     */
	    prop->name = STRDUP(name);
	    obj->ext_props_list = g_list_append(
		obj->ext_props_list,
		prop
	    );
	}

	/* Set the new value */
	g_free(prop->value);
	prop->value = STRDUP(value);
}

/*
 *	Removes the property.
 *
 *	The obj specifies the object that the properties list is on.
 *
 *	The name specifies the name of the property to remove.
 */
void EDVObjectPropRemove(
	edv_object_struct *obj,
	const gchar *name
)
{
	edv_object_prop_struct *prop;

	if((obj == NULL) || STRISEMPTY(name))
	    return;

	/* Find the property that matches the specified name */
	prop = EDVObjectPropGet(obj, name);
	if(prop == NULL)
	    return;

	/* Remove this property from the list */
	obj->ext_props_list = g_list_remove(
	    obj->ext_props_list,
	    prop
	);
}

/*
 *	Gets the property.
 *
 *	The obj specifies the object that the properties list is on.
 *
 *	The name specifies the name of the property to get.
 *
 *	Returns the property or NULL on error.
 */
edv_object_prop_struct *EDVObjectPropGet(
	edv_object_struct *obj,
	const gchar *name
)
{
	GList *glist;
	edv_object_prop_struct *prop;

	if((obj == NULL) || STRISEMPTY(name))
	    return(NULL);

	for(glist = obj->ext_props_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    prop = EDV_OBJECT_PROP(glist->data);
	    if(prop == NULL)
		continue;

	    if(prop->name == NULL)
		continue;

	    if(!g_strcasecmp(prop->name, name))
		return(prop);
	}

	return(NULL);
}

/*
 *	Creates a new Property.
 */
edv_object_prop_struct *EDVObjectPropNew(void)
{
	return(EDV_OBJECT_PROP(g_malloc0(sizeof(edv_object_prop_struct))));
}

/*
 *	Coppies the Property.
 */
edv_object_prop_struct *EDVObjectPropCopy(edv_object_prop_struct *prop)
{
	edv_object_prop_struct *new_prop;

	if(prop == NULL)
	    return(NULL);

	new_prop = EDVObjectPropNew();
	if(new_prop == NULL)
	    return(NULL);

	new_prop->name = STRDUP(prop->name);
	new_prop->value = STRDUP(prop->value);

	return(new_prop);
}

/*
 *	Deletes the Property.
 */
void EDVObjectPropDelete(edv_object_prop_struct *prop)
{
	if(prop == NULL)
	    return;

	g_free(prop->name);
	g_free(prop->value);
	g_free(prop);
}


/*
 *	Creates a new Object.
 */
edv_object_struct *EDVObjectNew(void)
{
	return(EDV_OBJECT(g_malloc0(sizeof(edv_object_struct))));
}

/*
 *	Coppies the object.
 */
edv_object_struct *EDVObjectCopy(edv_object_struct *obj)
{
	edv_object_struct *new_obj;

	if(obj == NULL)
	    return(NULL);

	new_obj = EDVObjectNew();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->device_index = obj->device_index;
	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->full_path = STRDUP(obj->full_path);

	new_obj->size = obj->size;

	new_obj->link_target = STRDUP(obj->link_target);
	new_obj->link_flags = obj->link_flags;

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;

	new_obj->owner_id = obj->owner_id;
	new_obj->group_id = obj->group_id;

	new_obj->device_type = obj->device_type;

	new_obj->block_size = obj->block_size;
	new_obj->blocks = obj->blocks;

	new_obj->hard_links = obj->hard_links;

	if(obj->ext_props_list != NULL)
	{
	    GList *glist;
	    for(glist = obj->ext_props_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		new_obj->ext_props_list = g_list_append(
		    new_obj->ext_props_list,
		    EDVObjectPropCopy(EDV_OBJECT_PROP(glist->data))
		);
	}

	return(new_obj);
}


/*
 *	Sets the Object's name and full path from the specified path.
 *
 *	The specified path must be an absolute path.
 */
void EDVObjectSetPath(
	edv_object_struct *obj, const gchar *path
)
{
	if(obj == NULL)
	    return;

	/* Reset the name and path first */
	g_free(obj->name);
	obj->name = NULL;
	g_free(obj->full_path);
	obj->full_path = NULL;

	/* Enough information to set path? */
	if(!STRISEMPTY(path))
	{
	    const gchar *s;

	    /* Set full path, always assume it is an absolute path */
	    obj->full_path = STRDUP(path);

	    /* Remove tailing deliminators if any */
	    StripPath(obj->full_path);

	    /* Get name from path */
	    s = g_basename(obj->full_path);
	    obj->name = STRISEMPTY(s) ?
		STRDUP(obj->full_path) : STRDUP(s);
	}
}

/*
 *	Sets the Object's name and full path from the specified parent
 *	and name.
 */
void EDVObjectSetPath2(
	edv_object_struct *obj,
	const gchar *parent, const gchar *name
)
{
	if(obj == NULL)
	    return;

	/* Reset the names first */
	g_free(obj->name);
	obj->name = NULL;
	g_free(obj->full_path);
	obj->full_path = NULL;

	/* Enough info given to set full path? */
	if(!STRISEMPTY(parent) && !STRISEMPTY(name))
	    obj->full_path = STRDUP(PrefixPaths(parent, name));

	/* Enough info given to set name? */
	if(!STRISEMPTY(name))
	    obj->name = STRDUP(name);
}

/*
 *	Sets/converts the struct stat information to the object.
 *
 *	The obj specifies the target object to set/convert the
 *	information to.
 *
 *	The lstat_buf specifies the source struct stat containing
 *	the information to set/convert from.
 *
 *	Since the struct stat contains less information than the
 *	object, the object's name and full_path should be set by a
 *	prior call to EDVObjectSetPath*().
 *
 *	The object's type, permissions, access_time, modify_time,
 *	change_time, owner_id, group_id, hard_links, size, device,
 *	inode, device_type, block_size, and blocks will be set.
 */
void EDVObjectSetStat(
	edv_object_struct *obj,
	struct stat *lstat_buf
)
{
	mode_t m;

	if((obj == NULL) || (lstat_buf == NULL))
	    return;

	m = lstat_buf->st_mode;

	/* Type */
	obj->type = EDVStatModeToObjectType(m);

	/* Index */
	obj->device_index = (gulong)lstat_buf->st_dev;
	obj->index = (gulong)lstat_buf->st_ino;

	/* Size */
	obj->size = (gulong)lstat_buf->st_size;

	/* Assume link is valid when setting stats, calling function
	 * may update this afterwards
	 */
	obj->link_flags = EDV_OBJECT_LINK_VALID;

	/* Permissions */
	obj->permissions = EDVStatModeToEDVPermissions(m);

	/* Time Stamps */
	obj->access_time = (gulong)lstat_buf->st_atime;
	obj->modify_time = (gulong)lstat_buf->st_mtime;
	obj->change_time = (gulong)lstat_buf->st_ctime;

	/* Ownership */
	obj->owner_id = (gint)lstat_buf->st_uid;
	obj->group_id = (gint)lstat_buf->st_gid;

	/* Device Type */
	obj->device_type = (gint)lstat_buf->st_rdev;

	/* Block transfer size and blocks allocated */
#if !defined(_WIN32)
	obj->block_size = (gulong)lstat_buf->st_blksize;
	obj->blocks = (gulong)lstat_buf->st_blocks;
#endif

	/* Number of hard links */
	obj->hard_links = (gint)lstat_buf->st_nlink;

#ifdef S_ISLNK
	/* If this is a link then set the link target */
	if(S_ISLNK(m))
	{
	    g_free(obj->link_target);
	    obj->link_target = EDVGetLinkTarget(obj->full_path);
	}
#endif
}

/*
 *	Sets/converts the object information to the struct stat.
 *
 *	The lstat_buf specifies the target struct stat to set/convert
 *	the information to.
 *
 *	The obj specifies the source object containing the information
 *	to set/convert from.
 *
 *	Note that the struct stat contains less information than the
 *	object so not all the information will be set/converted.
 */
void EDVObjectSetToStat(
	struct stat *lstat_buf,
	edv_object_struct *obj
)
{
	if((obj == NULL) || (lstat_buf == NULL))
	    return;

	lstat_buf->st_dev = (dev_t)obj->device_index;
	lstat_buf->st_ino = (ino_t)obj->index;

	lstat_buf->st_mode = EDVObjectTypeToStatType(obj->type) |
	    EDVEDVPermissionsToStatMode(obj->permissions);

	lstat_buf->st_nlink = (nlink_t)obj->hard_links;

	lstat_buf->st_uid = (uid_t)obj->owner_id;
	lstat_buf->st_gid = (gid_t)obj->group_id;

	lstat_buf->st_rdev = (dev_t)obj->device_type;

	lstat_buf->st_size = (off_t)obj->size;

#if !defined(_WIN32)
	lstat_buf->st_blksize = (unsigned long)obj->block_size;
	lstat_buf->st_blocks = (unsigned long)obj->blocks;
#endif

	lstat_buf->st_atime = (time_t)obj->access_time;
	lstat_buf->st_mtime = (time_t)obj->modify_time;
	lstat_buf->st_ctime = (time_t)obj->change_time;
}

/*
 *	Updates the object's link_flags.
 *
 *	The specified object must be of type EDV_OBJECT_TYPE_LINK and
 *	its full_path and link_value must be set by a prior call to
 *	EDVObjectSetStat().
 */
void EDVObjectUpdateLinkFlags(edv_object_struct *obj)
{
	gchar *path, *parent;
	struct stat stat_buf;

	if(obj == NULL)
	    return;

	obj->link_flags = 0;

	if(!EDV_OBJECT_IS_LINK(obj) ||
	   (obj->full_path == NULL) || (obj->link_target == NULL)
	)
	    return;

	parent = g_dirname(obj->full_path);
	if(parent == NULL)
	    return;

	path = EDVEvaluatePath(parent, obj->link_target);
	if(path == NULL)
	{
	    g_free(parent);
	    return;
	}

	if(!stat((const char *)path, &stat_buf))
	{
	    obj->link_flags |= EDV_OBJECT_LINK_VALID;
#ifdef S_ISDIR
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		obj->link_flags |= EDV_OBJECT_LINK_TAR_DIRECTORY;
		if(EDVIsLinkInfinatelyRecursive(obj->full_path))
		    obj->link_flags |= EDV_OBJECT_LINK_TAR_GRAND_PARENT;
	    }
#endif
	}

	g_free(path);
	g_free(parent);
}

/*
 *	Deletes the Object.
 */
void EDVObjectDelete(edv_object_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->full_path);

	g_free(obj->link_target);

	if(obj->ext_props_list != NULL)
	{
	    g_list_foreach(obj->ext_props_list, (GFunc)EDVObjectPropDelete, NULL);
	    g_list_free(obj->ext_props_list);
	}

	g_free(obj);
}
