#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "fb.h"
#include "progressdialog.h"
#include "toolbar.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_obj.h"
#include "edv_confirm.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_stat.h"
#include "edv_recbin_delete.h"
#include "edv_device.h"
#include "edv_device_mount.h"
#include "edv_devices_list.h"
#include "edv_mount_bar.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "obj_op_dlg.h"
#include "browser.h"
#include "browser_cb.h"
#include "browser_op_cb.h"
#include "browser_contents_list.h"
#include "browser_dir_tree.h"
#include "browser_contents_list.h"
#include "endeavour2.h"
#include "edv_obj_create.h"
#include "edv_obj_op.h"
#include "edv_cb.h"
#include "edv_help.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_wildcards_32x32.xpm"


void EDVBrowserOPCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void EDVBrowserOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void EDVBrowserOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
);

void EDVBrowserOPClose(edv_browser_struct *browser);
void EDVBrowserOPExit(edv_browser_struct *browser);

void EDVBrowserOPSyncDisks(edv_browser_struct *browser);
void EDVBrowserOPRun(edv_browser_struct *browser);
void EDVBrowserOPWriteProtect(edv_browser_struct *browser);
void EDVBrowserOPDeleteMethodRecycle(edv_browser_struct *browser);
void EDVBrowserOPDeleteMethodPurge(edv_browser_struct *browser);

static void EDVBrowserOPNewObjectNexus(
	edv_browser_struct *browser,
	edv_object_type type, GtkWidget *toplevel
);
void EDVBrowserOPNewObject(
	edv_browser_struct *browser, toolbar_item_struct *item
);
void EDVBrowserOPNewFile(edv_browser_struct *browser);
void EDVBrowserOPNewDirectory(edv_browser_struct *browser);
void EDVBrowserOPNewLink(edv_browser_struct *browser);
void EDVBrowserOPNewFifo(edv_browser_struct *browser);
void EDVBrowserOPNewDeviceBlock(edv_browser_struct *browser);
void EDVBrowserOPNewDeviceCharacter(edv_browser_struct *browser);
void EDVBrowserOPNewSocket(edv_browser_struct *browser);

void EDVBrowserOPOpen(edv_browser_struct *browser);
void EDVBrowserOPOpenTo(edv_browser_struct *browser, toolbar_item_struct *item);
void EDVBrowserOPOpenWith(edv_browser_struct *browser);

void EDVBrowserOPCopyPath(edv_browser_struct *browser);
void EDVBrowserOPCopyURL(edv_browser_struct *browser);
void EDVBrowserOPPaste(edv_browser_struct *browser);

void EDVBrowserOPMove(edv_browser_struct *browser);
void EDVBrowserOPCopy(edv_browser_struct *browser);
void EDVBrowserOPLink(edv_browser_struct *browser);
void EDVBrowserOPRename(edv_browser_struct *browser);
void EDVBrowserOPChMod(edv_browser_struct *browser);
void EDVBrowserOPChOwn(edv_browser_struct *browser);
void EDVBrowserOPChTime(edv_browser_struct *browser);
void EDVBrowserOPDelete(edv_browser_struct *browser);
void EDVBrowserOPProperties(edv_browser_struct *browser);

void EDVBrowserOPSelectAll(edv_browser_struct *browser);
void EDVBrowserOPUnselectAll(edv_browser_struct *browser);
void EDVBrowserOPInvertSelection(edv_browser_struct *browser);

void EDVBrowserOPDownload(edv_browser_struct *browser);

void EDVBrowserOPRefresh(edv_browser_struct *browser);
void EDVBrowserOPRefreshAll(edv_browser_struct *browser);
void EDVBrowserOPGoToParent(edv_browser_struct *browser);
void EDVBrowserOPGoToHome(edv_browser_struct *browser);
void EDVBrowserOPExpand(edv_browser_struct *browser);

void EDVBrowserContentsFilter(edv_browser_struct *browser);

void EDVBrowserOPDirectoryTreeOrigin(edv_browser_struct *browser);

void EDVBrowserOPMount(edv_browser_struct *browser);
void EDVBrowserOPEject(edv_browser_struct *browser);

void EDVBrowserMIMETypes(edv_browser_struct *browser);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Operation ID callback nexus.
 *
 *	The data must be a edv_browser_opid_struct *.
 */
void EDVBrowserOPCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	gint dev_num;
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_device_struct *dev;
	edv_browser_struct *browser;
	edv_core_struct *core;
	edv_browser_opid_struct *opid = EDV_BROWSER_OPID(data);
	if(opid == NULL)
	    return;

	browser = opid->browser;
	if((browser == NULL) || (opid->flags & EDV_OPID_NO_OP))
	    return;

	if(browser->processing || (browser->freeze_count > 0))
	    return;

	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	browser->freeze_count++;

	/* Get last selected device (if any) */
	dev_num = browser->selected_dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;

	/* Handle by operation id code */
	switch(opid->op)
	{
	  case EDV_BROWSER_OP_NONE:
	  case EDV_BROWSER_OP_SEPARATOR:
	    break;

	  case EDV_BROWSER_OP_CLOSE:
	    EDVBrowserOPClose(browser);
	    break;
	  case EDV_BROWSER_OP_EXIT:
	    EDVBrowserOPExit(browser);
	    break;

	  case EDV_BROWSER_OP_SYNC_DISKS:
	    EDVBrowserOPSyncDisks(browser);
	    break;
	  case EDV_BROWSER_OP_HISTORY:
	    EDVMapHistoryListWin(core, toplevel);
	    break;
	  case EDV_BROWSER_OP_RUN:
	    EDVBrowserOPRun(browser);
	    break;
	  case EDV_BROWSER_OP_RUN_TERMINAL:
	    if(core != NULL)
	    {
		gchar *wd = STRDUP(EDVBrowserCurrentLocation(browser));
		EDVRunTerminal(core, NULL, wd, toplevel);
		g_free(wd);
	    }
	    break;
	  case EDV_BROWSER_OP_WRITE_PROTECT:
	    EDVBrowserOPWriteProtect(browser);
	    break;
	  case EDV_BROWSER_OP_DELETE_METHOD_RECYCLE:
	    EDVBrowserOPDeleteMethodRecycle(browser);
	    break;
	  case EDV_BROWSER_OP_DELETE_METHOD_PURGE:
	    EDVBrowserOPDeleteMethodPurge(browser);
	    break;

	  case EDV_BROWSER_OP_NEW:
	    EDVBrowserOPNewObject(browser, item);
	    break;
	  case EDV_BROWSER_OP_NEW_FILE:
	    EDVBrowserOPNewFile(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_DIRECTORY:
	    EDVBrowserOPNewDirectory(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_LINK:
	    EDVBrowserOPNewLink(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_FIFO:
	    EDVBrowserOPNewFifo(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_DEVICE_BLOCK:
	    EDVBrowserOPNewDeviceBlock(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_DEVICE_CHARACTER:
	    EDVBrowserOPNewDeviceCharacter(browser);
	    break;
	  case EDV_BROWSER_OP_NEW_SOCKET:
	    EDVBrowserOPNewSocket(browser);
	    break;

	  case EDV_BROWSER_OP_OPEN:
	    EDVBrowserOPOpen(browser);
	    break;
	  case EDV_BROWSER_OP_OPEN_TO:
	    EDVBrowserOPOpenTo(browser, item);
	    break;
	  case EDV_BROWSER_OP_OPEN_WITH:
	    EDVBrowserOPOpenWith(browser);
	    break;

	  case EDV_BROWSER_OP_COPY_PATH:
	    EDVBrowserOPCopyPath(browser);
	    break;
	  case EDV_BROWSER_OP_COPY_URL:
	    EDVBrowserOPCopyURL(browser);
	    break;
	  case EDV_BROWSER_OP_PASTE:
	    EDVBrowserOPPaste(browser);
	    break;

	  case EDV_BROWSER_OP_MOVE:
	    EDVBrowserOPMove(browser);
	    break;
	  case EDV_BROWSER_OP_COPY:
	    EDVBrowserOPCopy(browser);
	    break;
	  case EDV_BROWSER_OP_LINK:
	    EDVBrowserOPLink(browser);
	    break;
	  case EDV_BROWSER_OP_RENAME:
	    EDVBrowserOPRename(browser);
	    break;
	  case EDV_BROWSER_OP_CHMOD:
	    EDVBrowserOPChMod(browser);
	    break;
	  case EDV_BROWSER_OP_CHOWN:
	    EDVBrowserOPChOwn(browser);
	    break;
	   case EDV_BROWSER_OP_CHTIME:
	    EDVBrowserOPChTime(browser);
	    break;
	  case EDV_BROWSER_OP_DELETE:
	    EDVBrowserOPDelete(browser);
	    break;
	  case EDV_BROWSER_OP_PROPERTIES:
	    EDVBrowserOPProperties(browser);
	    break;

	  case EDV_BROWSER_OP_SELECT_ALL:
	    EDVBrowserOPSelectAll(browser);
	    break;
	  case EDV_BROWSER_OP_UNSELECT_ALL:
	    EDVBrowserOPUnselectAll(browser);
	    break;
	  case EDV_BROWSER_OP_INVERT_SELECTION:
	    EDVBrowserOPInvertSelection(browser);
	    break;
	  case EDV_BROWSER_OP_FIND:
	    EDVMapBrowserFindWin(core, browser);
	    break;

	  case EDV_BROWSER_OP_DOWNLOAD:
	    EDVBrowserOPDownload(browser);
	    break;

	  case EDV_BROWSER_OP_REFRESH:
	    EDVBrowserOPRefresh(browser);
	    break;
	  case EDV_BROWSER_OP_REFRESH_ALL:
	    EDVBrowserOPRefreshAll(browser);
	    break;
	  case EDV_BROWSER_OP_GOTO_PARENT:
	    EDVBrowserOPGoToParent(browser);
	    break;
	  case EDV_BROWSER_OP_GOTO_HOME:
	    EDVBrowserOPGoToHome(browser);
	    break;
	  case EDV_BROWSER_OP_EXPAND:
	    EDVBrowserOPExpand(browser);
	    break;

	  case EDV_BROWSER_OP_SHOW_TOOL_BAR:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR, state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;
	  case EDV_BROWSER_OP_SHOW_LOCATION_BAR:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR, state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;
	  case EDV_BROWSER_OP_SHOW_MOUNT_BAR:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR, state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;
	  case EDV_BROWSER_OP_SHOW_FIND_BAR:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR, state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;
	  case EDV_BROWSER_OP_SHOW_STATUS_BAR:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR, state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_BROWSER_OP_CONTENTS_LIST_FILTER:
	    EDVBrowserContentsFilter(browser);
	    break;

	  case EDV_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS:
	    if(core != NULL)
	    {
		const gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS,
		    state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_BROWSER_OP_SHOW_HIDDEN_OBJECTS:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN, state
		);
		EDVBrowserOPRefresh(browser);
	    }
	    break;
	  case EDV_BROWSER_OP_SHOW_NOACCESS_OBJECTS:
	    if(core != NULL)
	    {
		gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
		);
		EDV_SET_B(
		    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS, state
		);
		EDVBrowserOPRefresh(browser);
	    }
	    break;
	  case EDV_BROWSER_OP_DIRECTORY_TREE_ORIGIN:
	    EDVBrowserOPDirectoryTreeOrigin(browser);
	    break;

	  case EDV_BROWSER_OP_MOUNT:
	    EDVBrowserOPMount(browser);
	    break;
	  case EDV_BROWSER_OP_EJECT:
	    EDVBrowserOPEject(browser);
	    break;
	  case EDV_BROWSER_OP_DEVICE_CHECK:
	    EDVRunDeviceCheck(core, dev, toplevel);
	    break;
	  case EDV_BROWSER_OP_DEVICE_TOOLS:
	    EDVRunDeviceTools(core, dev, toplevel);
	    break;
	  case EDV_BROWSER_OP_DEVICE_FORMAT:
	    EDVRunDeviceFormat(core, dev, toplevel);
	    break;

	  case EDV_BROWSER_OP_MIME_TYPES:
	    EDVBrowserMIMETypes(browser);
	    break;
	  case EDV_BROWSER_OP_DEVICES:
	    EDVMapDevicesListWin(core, toplevel);
	    break;

	  case EDV_BROWSER_OP_NEW_BROWSER:
	    EDVNewBrowser(core, NULL);
	    break;
	  case EDV_BROWSER_OP_NEW_IMBR:
	    EDVNewImbr(core, NULL);
	    break;
	  case EDV_BROWSER_OP_NEW_ARCHIVER:
	    EDVNewArchiver(core, NULL, NULL);
	    break;
	  case EDV_BROWSER_OP_RECYCLE_BIN:
	    EDVMapRecBin(core);
	    break;

	  case EDV_BROWSER_OP_OPTIONS:
	    EDVMapOptionsWin(core, toplevel);
	    break;
	  case EDV_BROWSER_OP_CUSTOMIZE:
	    EDVMapCustomizeWin(core, toplevel);
	    break;

	  case EDV_BROWSER_OP_HELP_CONTENTS:
	    EDVHelp(core, "Contents", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_FILE_BROWSER:
	    EDVHelp(core, "File Browser", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_IMAGE_BROWSER:
	    EDVHelp(core, "Image Browser", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_ARCHIVER:
	    EDVHelp(core, "Archiver", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_RECYCLE_BIN:
	    EDVHelp(core, "Recycle Bin", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_KEYS_LIST:
	    EDVHelp(core, "Keys List", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_MIME_TYPES:
	    EDVHelp(core, "MIME Types", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_DEVICES:
	    EDVHelp(core, "Devices", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_COMMON_OPERATIONS:
	    EDVHelp(core, "Common Operations", toplevel);
	    break;
	  case EDV_BROWSER_OP_HELP_ABOUT:
	    EDVAbout(core, toplevel);
	    break;
	}

	browser->freeze_count--;
}

/*
 *	Operation ID enter notify callback nexus.
 *
 *	The data must be a edv_browser_opid_struct *.
 */
void EDVBrowserOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	const gchar *tooltip;
	edv_browser_opid_struct *opid = EDV_BROWSER_OPID(data);
	edv_browser_struct *browser = (opid != NULL) ? opid->browser : NULL;
	if(browser == NULL)
	    return;

	tooltip = opid->tooltip;
	if(!STRISEMPTY(tooltip))
	    EDVStatusBarMessage(browser->status_bar, tooltip, FALSE);
}

/*
 *	Operation ID leave notify callback nexus.
 */
void EDVBrowserOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	edv_browser_opid_struct *opid = EDV_BROWSER_OPID(data);
	edv_browser_struct *browser = (opid != NULL) ? opid->browser : NULL;
	if(browser == NULL)
	    return;

	EDVStatusBarMessage(browser->status_bar, NULL, FALSE);
}


/*
 *	Sync Disks.
 */
void EDVBrowserOPSyncDisks(edv_browser_struct *browser)
{
	edv_status_bar_struct *sb;

	if(browser == NULL)
	    return;

	sb = browser->status_bar;

	EDVBrowserSetBusy(browser, TRUE);
	EDVStatusBarMessage(
	    sb,
	    "Syncing disks...",
	    TRUE
	);

	/* Sync disks */
	EDVSyncDisks(browser->core);

	EDVStatusBarMessage(
	    sb,
	    "Disk sync done",
	    FALSE
	);
	EDVStatusBarProgress(sb, 0.0f, FALSE);
	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Run.
 */
void EDVBrowserOPRun(edv_browser_struct *browser)
{
	gchar *s, *cmd;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_object_struct *obj;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Format command to contain the list of selected objects */
	cmd = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = EDV_OBJECT(gtk_clist_get_row_data(
		clist, (gint)glist->data
	    ));
	    if(obj == NULL)
		continue;

	    if(STRISEMPTY(obj->full_path))
		continue;

	    if(cmd != NULL)
		s = g_strconcat(
		    cmd,
		    " ",
		    obj->full_path,
		    NULL
		);
	    else
		s = STRDUP(obj->full_path);
	    g_free(cmd);
	    cmd = s;
	}

	EDVMapRunDialogCommand(
	    core,
	    cmd,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	g_free(cmd);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Write Protect.
 */
void EDVBrowserOPWriteProtect(edv_browser_struct *browser)
{
	gboolean write_protect;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get current write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Toggle write protect */
	write_protect = !write_protect;

	/* Set new write protect state */
	CFGItemListSetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
	    write_protect, FALSE
	);

	/* Emit write protect changed signal */
	EDVWriteProtectChangedEmit(core, write_protect);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Delete Method: Recycle.
 */
void EDVBrowserOPDeleteMethodRecycle(edv_browser_struct *browser)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_RECYCLE
	);
	EDVReconfiguredEmit(core);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Delete Method: Purge.
 */
void EDVBrowserOPDeleteMethodPurge(edv_browser_struct *browser)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_PURGE
	);
	EDVReconfiguredEmit(core);

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Close.
 */
void EDVBrowserOPClose(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserSyncConfiguration(browser);
	EDVBrowserUnmap(browser);
}

/*
 *	Exit.
 */
void EDVBrowserOPExit(edv_browser_struct *browser)
{
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	core = browser->core;

	EDVBrowserSyncConfiguration(browser);
	EDVBrowserUnmap(browser);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	New Object Nexus.
 */
static void EDVBrowserOPNewObjectNexus(
	edv_browser_struct *browser,
	edv_object_type type, GtkWidget *toplevel
)
{
	gboolean yes_to_all = FALSE;
	gint status;
	const char *obj_type_name;
	gchar *new_path = NULL;
	const gchar *cur_path, *error_msg;
	edv_core_struct *core = browser->core;
	if(core == NULL)
	    return;

#define CLEANUP_RETURN	{	\
 g_free(new_path);		\
				\
 return;			\
}

	EDVBrowserSetBusy(browser, TRUE);

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get current location as the path to create the new object
	 * at
	 */
	cur_path = EDVBrowserCurrentLocation(browser);
	if(cur_path == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    CLEANUP_RETURN;
	}

	/* Create new object by type */
	status = -1;
	obj_type_name = NULL;
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
	  case EDV_OBJECT_TYPE_ERROR:
	  case EDV_OBJECT_TYPE_FILE:
	    obj_type_name = "file";
	    status = EDVObjCreateFile(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
	    obj_type_name = "directory";
	    status = EDVObjCreateDirectory(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_LINK:
	    obj_type_name = "symbolic link";
	    status = EDVObjCreateLink(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
	    obj_type_name = "block device";
	    status = EDVObjCreateDeviceBlock(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
	    obj_type_name = "character device";
	    status = EDVObjCreateDeviceCharacter(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_FIFO:
	    obj_type_name = "fifo pipe";
	    status = EDVObjCreateFifo(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	  case EDV_OBJECT_TYPE_SOCKET:
	    obj_type_name = "socket";
	    status = EDVObjCreateSocket(
		core, cur_path, &new_path,
		toplevel, TRUE, TRUE, &yes_to_all
	    );
	    break;
	}

	/* Unmap progress dialog since it may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Error creating new object? */
	if(status != 0)
	{
	    error_msg = EDVObjCreateGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Create Failed",
		    error_msg,
		    NULL,
		    browser->toplevel
		);
	    }

	    EDVStatusBarMessage(browser->status_bar, NULL, FALSE);
	}
	else
	{
	    /* Successfully created new object */
	    gchar *msg;
	    struct stat lstat_buf;

	    /* Get local stats of new object */
	    if(!lstat(new_path, &lstat_buf))
	    {
		gint row;
		GtkCList *clist = GTK_CLIST(browser->contents_clist);

		/* Emit a disk object added signal to all of endeavour's
		 * resources
		 */
		EDVObjectAddedEmit(
		    core, new_path, &lstat_buf
		);

		/* Select new row on contents clist that is listing the
		 * new object.
		 */
		row = EDVBrowserContentsFindRowByPath(browser, new_path);
		if((row > -1) && (clist != NULL))
		{
		    gtk_clist_freeze(clist);
		    gtk_clist_unselect_all(clist);
		    gtk_clist_select_row(clist, row, 0);
		    gtk_clist_thaw(clist);
		}
	    }

	    msg = g_strdup_printf(
		"Created new %s",
		obj_type_name
	    );
	    EDVStatusBarMessage(browser->status_bar, msg, FALSE);
	    g_free(msg);
	}

	EDVBrowserSetBusy(browser, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	New Object.
 */
void EDVBrowserOPNewObject(
	edv_browser_struct *browser, toolbar_item_struct *item
)
{
	if(browser == NULL)
	    return;

	EDVMenuButtonMapMenu(
	    browser->new_object_menu,
	    (item != NULL) ? item->w : NULL
	);
}

/*
 *	New File.
 */
void EDVBrowserOPNewFile(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_FILE, browser->toplevel
	);
}

/*
 *	New Directory.
 */
void EDVBrowserOPNewDirectory(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_DIRECTORY, browser->toplevel  
	);
}

/*
 *	New Link.
 */
void EDVBrowserOPNewLink(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_LINK, browser->toplevel  
	);
}

/*
 *	New FIFO Pipe.
 */
void EDVBrowserOPNewFifo(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_FIFO, browser->toplevel  
	);
}

/*
 *	New Block Device.
 */
void EDVBrowserOPNewDeviceBlock(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_DEVICE_BLOCK, browser->toplevel  
	);
}

/*
 *	New Character Device.
 */
void EDVBrowserOPNewDeviceCharacter(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_DEVICE_CHARACTER, browser->toplevel  
	);
}

/*
 *	New Socket.
 */
void EDVBrowserOPNewSocket(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserOPNewObjectNexus(
	    browser, EDV_OBJECT_TYPE_SOCKET, browser->toplevel  
	);
}


/*
 *	Open.
 */
void EDVBrowserOPOpen(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserContentsOpen(
	    browser,
	    -1, -1,		/* Use selected object(s) */
	    0
	);
}

/*
 *	Open To.
 */
void EDVBrowserOPOpenTo(edv_browser_struct *browser, toolbar_item_struct *item)
{
	if(browser == NULL)
	    return;

	EDVMenuButtonMapMenu(
	    browser->open_to_menu,
	    (item != NULL) ? item->w : NULL
	);
}

/*
 *	Open With.
 */
void EDVBrowserOPOpenWith(edv_browser_struct *browser)
{
	GtkCList *clist;

	if(browser == NULL)
	    return;

	/* Is there an object selected on the Contents List? */
	clist = GTK_CLIST(browser->contents_clist);
	if(clist->selection != NULL)
	{
	    EDVBrowserContentsOpenWith(
		browser,
		-1, -1		/* Use selected object(s) */
	    );
	}
	else
	{
	    /* Since no object was selected on the Contents List then
	     * check if an object is selected on the Directory Tree
	     */
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCTreeNode *node = EDVCTreeGetSelectedLast(ctree, NULL);
	    if(node != NULL)
		EDVBrowserDirTreeOpenWith(browser, node);
	}
}


/*
 *	Copy Path To DDE.
 */
void EDVBrowserOPCopyPath(edv_browser_struct *browser)
{
	GList *objs_list;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	EDVBrowserSyncData(browser);

	/* Get the list of selected objects */
	objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Copy the paths of the selected objects to the dde */
	EDVCopyDiskObjectsToDDEPath(
	    browser->core,
	    objs_list,
	    browser->toplevel
	);

	/* Delete the selected objects list */
	g_list_free(objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Copy Location To DDE.
 */
void EDVBrowserOPCopyURL(edv_browser_struct *browser)
{
	GList *objs_list;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	EDVBrowserSyncData(browser);

	/* Get the list of selected objects */
	objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Copy the urls of the selected objects to the dde */
	EDVCopyDiskObjectsToDDEURL(
	    browser->core,
	    objs_list,
	    browser->toplevel
	);

	/* Delete the selected objects list */
	g_list_free(objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Paste.
 */
void EDVBrowserOPPaste(edv_browser_struct *browser)
{
	gboolean yes_to_all = FALSE;
	gint status = -1;
	const gchar *protocol;
	gchar *buf;
	GList *glist, *url_list = NULL;
	GtkWidget *toplevel;
	url_struct *url;
	edv_object_struct *obj = NULL;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	EDVBrowserSyncData(browser);

	/* Get the data from the clipboard as a string */
	buf = GUIDDEGetString(
	    toplevel,
	    GDK_SELECTION_PRIMARY,
	    GDK_CURRENT_TIME
	);
	if(buf == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the last selected object from the contents list only
	 * if it is a directory, otherwise get last selected object
	 * from the directory tree
	 */
	/* Contents clist was last selected? */
	if(browser->contents_clist_selected_row > -1)
	{
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    GList *glist = (clist != NULL) ? clist->selection_end : NULL;
	    if(glist != NULL)
	    {
		obj = EDV_OBJECT(
		    gtk_clist_get_row_data(clist, (gint)glist->data)
		);
		if((obj != NULL) ?
		    (obj->type != EDV_OBJECT_TYPE_DIRECTORY) : TRUE
		)
		    obj = NULL;
	    }
	}
	/* Directory ctree was last selected? */
	if(obj == NULL)
	{
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCList *clist = GTK_CLIST(ctree);
	    GList *glist = (clist != NULL) ? clist->selection_end : NULL;
	    if(glist != NULL)
	    {
		GtkCTreeNode *node = (GtkCTreeNode *)glist->data;
		obj = EDV_OBJECT(
		    gtk_ctree_node_get_row_data(ctree, node)
		);
		if((obj != NULL) ?
		    (obj->type != EDV_OBJECT_TYPE_DIRECTORY) : TRUE
		)
		    obj = NULL;
	    }
	}
	/* No objects selected? */
	if(obj == NULL)
	{
	    g_free(buf);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Decode the URL list string into a list of URLs */
	url_list = URLDecodeString(buf);
	g_free(buf);
	buf = NULL;

	/* Make confirmation */
	if(url_list != NULL)
	{
	    url = URL(url_list->data);
	    if(EDVConfirmCopy(
		core, toplevel,
		(url != NULL) ? url->path : NULL, g_list_length(url_list),
		obj->full_path
	    ) != CDIALOG_RESPONSE_YES)
	    {
		g_list_foreach(url_list, (GFunc)URLDelete, NULL);
		g_list_free(url_list);
		EDVBrowserSetBusy(browser, FALSE);
		return;
	    }
	}

	/* Paste each URL */
	for(glist = url_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    url = URL(glist->data);
	    if(url == NULL)
		continue;

	    protocol = url->protocol;
	    if(STRISEMPTY(protocol))
		protocol = "file";

	    /* File? */
	    if(!g_strcasecmp(protocol, "file"))
	    {
		const gchar *error_msg;
		gchar *new_path = NULL;

		/* Copy this object */
		status = EDVObjectOPCopy(
		    core,
		    url->path, obj->full_path,
		    &new_path, toplevel,
		    TRUE, TRUE, &yes_to_all
		);

		/* Check for error */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageObjectOPError(
			"Copy Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		if(new_path != NULL)
		{
		    /* Notify about this object being added */
		    struct stat lstat_buf;
		    if(!lstat((const char *)new_path, &lstat_buf))
			EDVObjectAddedEmit(
			    core, new_path, &lstat_buf
			);
		    g_free(new_path);
		}

		/* Skip handling of the rest of the objects on error
		 * (status != 0) and that the error was not a user
		 * response of no (status != -5)
		 */
		if((status != 0) && (status != -5))
		    break;
	    }
	    /* Download? */
	    else if(!g_strcasecmp(protocol, "http") ||
		    !g_strcasecmp(protocol, "ftp") ||
		    !g_strcasecmp(protocol, "https")
	    )
	    {
		EDVInternetDownloadObject(
		    core, url, obj->full_path, toplevel
		);
	    }
	}

	/* Delete the URL list */
	if(url_list != NULL)
	{
	    g_list_foreach(url_list, (GFunc)URLDelete, NULL);
	    g_list_free(url_list);
	}

	/* Unmap progress dialog, it may have been mapped if any
	 * operations occured in the above loop
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Move.
 */
void EDVBrowserOPMove(edv_browser_struct *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Map the object operations dialog to move */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_MOVE,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Copy.
 */
void EDVBrowserOPCopy(edv_browser_struct *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Map the object operations dialog to copy */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_COPY,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Link.
 */
void EDVBrowserOPLink(edv_browser_struct *browser)
{
	GList *glist, *sel_objs_list;
	GtkWidget *toplevel;
	edv_core_struct *core;
	edv_obj_op_dlg_struct *d;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Linking only allows one object in the list, so create
	 * a list with only one object
	 */
	glist = NULL;
	glist = g_list_append(glist, sel_objs_list->data);
	g_list_free(sel_objs_list);
	sel_objs_list = glist;

	/* Map the object operations dialog to link */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_LINK,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Rename.
 */
void EDVBrowserOPRename(edv_browser_struct *browser)
{
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	toplevel = browser->toplevel;
	core = browser->core;

/* This will actually be checked again when the rename fprompt map call
 * but we check it first initially anyways
 */
	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	/* Check which list contains the last selected object, that will
	 * indicate the list to map the rename floating prompt for
	 */

	/* Contents clist was last selected? */
	if(browser->contents_clist_selected_row > -1)
	{
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    const gint row = EDVCListGetSelectedLast(clist, NULL);
	    if(row > -1)
		EDVBrowserContentsPromptRename(
		    browser, row, -1
		);
	}
	/* Directory ctree was last selected? */
	else if(browser->directory_ctree_selected_node != NULL)
	{
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCTreeNode *node = EDVCTreeGetSelectedLast(ctree, NULL);
	    if(node != NULL)
		EDVBrowserDirTreePromptRename(
		    browser, node
		);
	}
}

/*
 *	Change Permissions.
 */
void EDVBrowserOPChMod(edv_browser_struct *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Map the object operations dialog to change permissions */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHMOD,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Change Ownership.
 */
void EDVBrowserOPChOwn(edv_browser_struct *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Map the object operations dialog to change ownership */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHOWN,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Set Time Stamps.
 */
void EDVBrowserOPChTime(edv_browser_struct *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_core_struct *core;
	edv_obj_op_dlg_struct *d;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Sync data to ensure current values to operate on */
	EDVBrowserSyncData(browser);

	/* Get the selected objects list */
	sel_objs_list = EDVBrowserGetSelectedObjectsList(browser, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Map the object operations dialog to set time stamps */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHTIME,
	    EDV_LOCATION_TYPE_VFS,
	    sel_objs_list,
	    EDVBrowserCurrentLocation(browser),
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Delete.
 */
void EDVBrowserOPDelete(edv_browser_struct *browser)
{
	struct stat lstat_buf;
	gboolean yes_to_all = FALSE;
	gint status, response, npaths, objects_deleted = 0;
	const gchar *path, *error_msg;
	GList *glist, *paths_list, *index_list;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

#define CLEANUP_RETURN	{			\
 if(paths_list != NULL)	{			\
  g_list_foreach(				\
   paths_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(paths_list);			\
 }						\
						\
 return;					\
}

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the list of selected objects */
	paths_list = EDVBrowserGetSelectedPathsList(browser);

	/* No objects selected? */
	if(paths_list == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    CLEANUP_RETURN;
	}

	npaths = g_list_length(paths_list);

	/* Confirm delete */
	response = EDVConfirmDelete(
	    core, toplevel,
	    (npaths == 1) ? (const gchar *)paths_list->data : NULL,
	    npaths
	);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	    yes_to_all = TRUE;
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_OK:
	    break;

	  default:
	    EDVBrowserSetBusy(browser, FALSE);
	    CLEANUP_RETURN;
	    break;
	}

	/* Delete each selected object */
	status = 0;
	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    path = (const gchar *)glist->data;
	    if(path == NULL)
		continue;

	    /* Delete this object */
	    index_list = NULL;
	    status = EDVRecBinDeleteObject(
		core, path,
		&index_list,
		toplevel,
		TRUE, TRUE, &yes_to_all
	    );

	    /* Check for errors */
	    error_msg = EDVRecBinDeleteObjectGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Delete Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Check if the object still exists */
	    if(lstat(path, &lstat_buf))
	    {
#ifdef ENOENT
		/* Report the object being removed */
		const gint error_code = (gint)errno;
		if(error_code == ENOENT)
		    EDVObjectRemovedEmit(core, path);
#endif
	    }

	    /* Any objects deleted? */
	    if(index_list != NULL)
	    {
		guint index;
		GList *glist;

		for(glist = index_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    index = (guint)glist->data;
		    if(index != 0)
			EDVRecycledObjectAddedEmit(core, index);

		    objects_deleted++;
		}

		/* Delete the recycle objects index list */
		g_list_free(index_list);
	    }

	    /* User aborted? */
	    if(status == -4)
		break;
	}

	/* Update status bar */
	if(TRUE)
	{
	    gchar *s;
	    if(status == -4)
		s = STRDUP(
		    "Delete operation canceled"
		);
	    else if(objects_deleted > 0)
		s = g_strdup_printf(
		    "Deleted %i %s",
		    objects_deleted,
		    (objects_deleted == 1) ? "object" : "objects"
		);
	    else
		s = g_strdup_printf(
		    "Unable to delete %s",
		    (npaths == 1) ? "object" : "objects"
		);
	    EDVStatusBarMessage(browser->status_bar, s, FALSE);
	    g_free(s);
	}

	/* Unmap progress dialog, it may have been mapped if any
	 * operations occured in the above loop
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	EDVBrowserSetBusy(browser, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Properties.
 */
void EDVBrowserOPProperties(edv_browser_struct *browser)
{
	GtkWidget *toplevel;
	edv_object_struct *obj = NULL;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	EDVBrowserSyncData(browser);

	/* Get the selected object */
	if(browser->contents_clist_selected_row > -1)
	{
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    GList *glist = clist->selection_end;
	    if(glist != NULL)
		obj = EDV_OBJECT(gtk_clist_get_row_data(
		    clist,
		    (gint)glist->data
		));
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCList *clist = GTK_CLIST(ctree);
	    GList *glist = clist->selection_end;
	    if(glist != NULL)
		obj = EDV_OBJECT(gtk_ctree_node_get_row_data(
		    ctree,
		    (GtkCTreeNode *)glist->data
		));
	}

	/* No selected object */
	if(obj == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Create a new properties dialog displaying the object */
	EDVNewPropertiesDialogVFS(
	    core,
	    obj,
	    toplevel
	);

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Select All.
 */
void EDVBrowserOPSelectAll(edv_browser_struct *browser)
{
	edv_core_struct *core;
	GtkCList *clist;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Select all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Mark node as unselected on directory ctree */
	browser->directory_ctree_selected_node = NULL;

	/* Assume highest row index as the last selected row */
	browser->contents_clist_selected_row = clist->rows - 1;

	EDVStatusBarMessage(
	    browser->status_bar, "All objects selected", FALSE
	);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Unselect All.
 */
void EDVBrowserOPUnselectAll(edv_browser_struct *browser)
{
	edv_core_struct *core;
	GtkCList *clist;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Unselect all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark node as unselected on directory ctree */
	browser->directory_ctree_selected_node = NULL;

	/* Mark contents clist's row as unselected */
	browser->contents_clist_selected_row = -1;

	EDVStatusBarMessage(
	    browser->status_bar, "All objects unselected", FALSE
	);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Invert Selection.
 */
void EDVBrowserOPInvertSelection(edv_browser_struct *browser)
{
	gint row, total_rows;
	GList *glist, *selection;
	GtkCList *clist;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Get a copy of the selected rows list */
	selection = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	/* Invert the selection */
	gtk_clist_freeze(clist);
	for(row = 0, total_rows = clist->rows;
	    row < total_rows;
	    row++
	)
	{
	    for(glist = selection;
	        glist != NULL;
	        glist = g_list_next(glist)
	    )
	    {
		if(row == (gint)glist->data)
		{
		    gtk_clist_unselect_row(clist, row, 0);
		    break;
		}
	    }
	    /* Row not selected? */
	    if(glist == NULL)
		gtk_clist_select_row(clist, row, 0);
	}
	gtk_clist_thaw(clist);

	g_list_free(selection);

	EDVStatusBarMessage(
	    browser->status_bar, "Selection inverted", FALSE
	);
	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Download.
 */
void EDVBrowserOPDownload(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);
	EDVInternetDownloadObject(
	    browser->core,
	    NULL,
	    EDVBrowserCurrentLocation(browser),
	    browser->toplevel
	);
	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Refresh.
 */
void EDVBrowserOPRefresh(edv_browser_struct *browser)
{
	gchar *cur_path;
	GtkWidget *toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *sb;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	toplevel = browser->toplevel;
	cur_path = STRDUP(EDVBrowserCurrentLocation(browser));
	core = browser->core;
	cfg_list = core->cfg_list;
	sb = browser->status_bar;

	EDVBrowserSetBusy(browser, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Directory GtkCTree */
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* Get the selected node */
	node = EDVCTreeGetSelectedLast(ctree, &row_ptr);
	if((node != NULL) && (row_ptr != NULL))
	{
	    const gboolean was_expanded = row_ptr->expanded;

	    /* Record last scroll position */
	    const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
				last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	    gtk_clist_freeze(clist);

	    /* Get a new listing of child nodes for the selected node */
	    EDVBrowserDirTreeGetListing(browser, node, TRUE);

	    /* Realize all the nodes */
	    EDVBrowserDirTreeRealizeListing(browser, NULL);

	    /* Reget the selected node */
	    node = EDVCTreeGetSelectedLast(ctree, &row_ptr);
	    if(node != NULL)
	    {
		/* Expand it as needed */
		if(was_expanded)
		    gtk_ctree_expand(ctree, node);
	    }

	    gtk_clist_thaw(clist);

	    /* Scroll back to the original position */
	    EDVScrollCListToPosition(clist, last_x, last_y);

	}

	/* Update the Contents GtkCList */
	clist = GTK_CLIST(browser->contents_clist);
	if(cur_path != NULL)
	{
	    /* Record the last scroll positions */
	    const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
				last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	    gtk_clist_freeze(clist);

	    /* Reget the listing */
	    EDVBrowserContentsGetListing(
		browser,
		cur_path,
		EDV_GET_B(EDV_CFG_PARM_LISTS_ANIMATE_UPDATES)
	    );

	    gtk_clist_thaw(clist);

	    /* Scroll back to the original position */
	    EDVScrollCListToPosition(clist, last_x, last_y);
	}

	EDVBrowserUpdateMenus(browser);
	EDVStatusBarMessage(sb, "Refreshed contents listing", FALSE);

	g_free(cur_path);

	GUIBlockInput(toplevel, FALSE);
	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Refresh All.
 */
void EDVBrowserOPRefreshAll(edv_browser_struct *browser)
{
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	core = browser->core;

	/* Refresh device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Refresh File Browser */
	EDVBrowserOPRefresh(browser);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Go To Parent.
 */
void EDVBrowserOPGoToParent(edv_browser_struct *browser)
{
	const gchar *path;
	GtkWidget *toplevel;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree;
	edv_object_struct *obj;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;

	/* Get the selected node */
	node = EDVCTreeGetSelectedLast(ctree, &row_ptr);
	if((node == NULL) || (row_ptr == NULL))
	{
	    /* No node selected */
	    EDVPlaySoundBeep(core);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the selected node's Object */
	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	path = obj->full_path;
	if(STRISEMPTY(path))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Already at toplevel? */
	if(!strcmp((const char *)path, "/"))
	{
	    EDVPlaySoundBeep(core);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Get the parent node
	 *
	 * If the node does not have a parent then it implies that
	 * we need to update the Directory GtkCTree's Origin path
	 */
	GUIBlockInput(toplevel, TRUE);
	node = row_ptr->parent;
	if(node != NULL)
	{
	    /* Select the parent node */
	    gtk_ctree_select(ctree, node);
	}
	else
	{
	    gchar *parent_path = g_dirname(path);
	    if(parent_path != NULL)
	    {
		/* Update the Directory Tree Origin to be the parent
		 * and select the parent
		 */
		EDVBrowserDirTreeSetOriginPath(browser, parent_path);
		g_free(parent_path);
	    }
	}
	GUIBlockInput(toplevel, FALSE);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Go To Home.
 */
void EDVBrowserOPGoToHome(edv_browser_struct *browser)
{
	gchar *path;
	GtkWidget *toplevel;
	GtkCTree *ctree;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;

	/* Get home directory */
	path = STRDUP(core->home_dir);
	if(STRISEMPTY(path))
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A En Casa Gua Fallada",
"Incapaz de encontrar la gua de hogar, cercirese que el\n\
ambiente HOME variable se pone.\n",
		NULL,
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A L'Annuaire De Maison Echou",
"Incapable de trouver l'annuaire de maison, s'assure que\n\
l'environnement HOME variable est rgl.\n",
		NULL,
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Sie Zu Heim Verzeichnis Hat Versagt",
"Unfhig, das heim verzeichnis zu finden, vergewissert\n\
sich, da die umwelt vernderliche HOME gesetzt ist.\n",
		NULL,
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare All'Elenco Di Casa Fallito",
"Incapace per trovare l'elenco di casa, si assicura che\n\
l'ambiente HOME variabile  regolato.\n",
		NULL,
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te Huis Gids Verzuimde",
"Onbekwame de huis gids te vinden, vergewist zich ervan dat de\n\
omgeving, die veranderlijke HOME gezet is.\n",
		NULL,
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A Guia De Lar Fracassado",
"Incapaz de achar o guia de lar, assegura-se que o ambiente\n\
HOME varivel  posto.\n",
		NULL,
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til Hjem Sviktet Katalog",
"Maktesls finne hjemkatalogen, sjekker at miljet variabel\n\
HOME setter.\n",
		NULL,
#else
"Go To Home Directory Failed",
"Unable to find the home directory, make sure that the\n\
environment variable HOME is set.\n",
		NULL,
#endif
		toplevel
	    );
	    g_free(path);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Change the Directory Tree origin as needed */
	GUIBlockInput(toplevel, TRUE);
	if(EDVBrowserIsPathFromDirTreeOrigin(browser, path))
	{
	    /* Select the home directory */
	    EDVBrowserDirTreeSelectPath(browser, path);
	}
	else
	{
	    /* Set the Directory Tree origin to the home directory
	     * and select the home directory
	     */
	    EDVBrowserDirTreeSetOriginPath(browser, path);
	}
	GUIBlockInput(toplevel, FALSE);

	g_free(path);

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Expands/Colapse.
 */
void EDVBrowserOPExpand(edv_browser_struct *browser)
{
	GtkCTree *ctree;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Get selected node */
	node = EDVCTreeGetSelectedLast(ctree, &row_ptr);
	if((node == NULL) || (row_ptr == NULL))
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Collapse or expand node? Note that appropriate callbacks
	 * will be made when expanded or collapsed.
	 */
	if(row_ptr->expanded)
	    gtk_ctree_collapse(ctree, node);
	else
	    gtk_ctree_expand(ctree, node);

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Sets the contents list filter.
 */
void EDVBrowserContentsFilter(edv_browser_struct *browser)
{
	gchar **strv;
	gint strc;
	GtkWidget *toplevel;

	if((browser == NULL) || PDialogIsQuery())
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;

	PDialogDeleteAllPrompts();
	PDialogSetTransientFor(toplevel);
	PDialogAddPrompt(NULL, "Filter:", browser->contents_list_filter);
	PDialogSetSize(320, -1);
	strv = PDialogGetResponseIconData(
	    "Set Filter",
	    NULL, NULL,
	    (guint8 **)icon_wildcards_32x32_xpm,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	PDialogSetTransientFor(NULL);
	if((strv != NULL) && (strc > 0))
	{
	    if(strc > 0)
	    {
		g_free(browser->contents_list_filter);
		browser->contents_list_filter = STRDUP(strv[0]);
	    }

	    EDVBrowserOPRefresh(browser);
	}

	PDialogDeleteAllPrompts();

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Directory Tree Origin.
 */
void EDVBrowserOPDirectoryTreeOrigin(edv_browser_struct *browser)
{
	struct stat stat_buf;
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL, *new_origin_path;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;
	if((browser == NULL) || FileBrowserIsQuery())
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Create new file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);
	  
	/* Query the user for the origin directory */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Set Directory Tree Origin",
	    "Set", "Cancel",
	    browser->directory_ctree_origin_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	); 
	FileBrowserSetTransientFor(NULL);

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Got new directory tree origin path and it exists and it leads
	 * to a directory?
	 */
	if(response)
	{
	    new_origin_path = (npaths > 0) ?
		STRDUP(paths_list[0]) : NULL;
	}
	else
	{
	    new_origin_path = NULL;
	}

	if(STRISEMPTY(new_origin_path))
	{
	    g_free(new_origin_path);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}
	if(!g_path_is_absolute(new_origin_path))
	{
	    gchar *msg = g_strdup_printf(
"Not an absolute path:\n\
\n\
    %s\n",
		new_origin_path
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Set Directory Tree Origin Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    g_free(new_origin_path);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}
	if(stat((const char *)new_origin_path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s\n",
		g_strerror(error_code), new_origin_path
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Set Directory Tree Origin Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    g_free(new_origin_path);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}
#ifdef S_ISDIR
	if(!S_ISDIR(stat_buf.st_mode))
#else
	if(TRUE)
#endif
	{
	    gchar *msg = g_strdup_printf(
"Object is not a directory:\n\
\n\
    %s\n",
		new_origin_path
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Set Directory Tree Origin Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    g_free(new_origin_path);
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Set new Directory Tree Origin path */
	GUIBlockInput(toplevel, TRUE);
	EDVBrowserDirTreeSetOriginPath(browser, new_origin_path);
	GUIBlockInput(toplevel, FALSE);

	g_free(new_origin_path);

	EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	Mount/Unmount.
 */
void EDVBrowserOPMount(edv_browser_struct *browser)
{
	gboolean original_mount_state;
	gint status, dev_num;
	GtkWidget *toplevel;
	edv_device_struct *dev;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Get number of last selected device index (if any) */
	dev_num = browser->selected_dev_num;
	if((dev_num >= 0) && (dev_num < core->total_devices))
	    dev = core->device[dev_num];
	else
	    dev = NULL;
	if(dev == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	    status = EDVDeviceUnmount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );
	else
	    status = EDVDeviceMount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );

	/* Update all device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Mount error? */
	if(status != 0)
	{
	    const gchar *last_error = EDVDeviceMountGetError(core);
	    if(!STRISEMPTY(last_error))
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
		    original_mount_state ?
			"Unmount Failed" : "Mount Failed",
		    last_error,
		    NULL,
		    browser->toplevel
		);
	    }
	}
	else
	{
	    /* Report un mount signal to all of endeavour's resources */
	    EDVObjectMountEmit(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *	Eject.
 */
void EDVBrowserOPEject(edv_browser_struct *browser)
{
	gboolean original_mount_state;
	gint status, dev_num;
	GtkWidget *toplevel;
	edv_device_struct *dev;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Get number of last selected device index (if any) */
	dev_num = browser->selected_dev_num;
	if((dev_num >= 0) && (dev_num < core->total_devices))
	    dev = core->device[dev_num];
	else
	    dev = NULL;
	if(dev == NULL)
	{
	    EDVBrowserSetBusy(browser, FALSE);
	    return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	    status = EDVDeviceUnmount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );

	/* Eject */
	status = EDVDeviceEject(
	    core, dev,
	    TRUE, TRUE,
	    toplevel
	);

	/* Update all device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Eject error? */
	if(status != 0)
	{
	    const gchar *last_error = EDVDeviceMountGetError(core);
	    if(!STRISEMPTY(last_error))
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Eject Failed",
		    last_error,
		    NULL,
		    browser->toplevel
		);
	    }
	}
	else
	{
	    /* Report eject (unmount) signal to all of endeavour's
	     * resources
	     */
	    EDVObjectMountEmit(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	EDVBrowserSetBusy(browser, FALSE);
}

/*
 *      MIME Types.
 */
void EDVBrowserMIMETypes(edv_browser_struct *browser)
{
	gchar *type_str = NULL;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	EDVBrowserSetBusy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	if(browser->contents_clist_selected_row > -1)
	{
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    const gint i = EDVCListGetSelectedLast(clist, NULL);
	    if(i > -1)
	    {
		edv_object_struct *obj = EDV_OBJECT(
		    gtk_clist_get_row_data(clist, i)
		);
		if(obj != NULL)
		    EDVMatchObjectTypeString(
			core->mimetype, core->total_mimetypes,
			obj->type,
			obj->permissions,
			obj->full_path,
			&type_str
		);
	    }
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCTreeNode *node = EDVCTreeGetSelectedLast(ctree, NULL);
	    if(node != NULL)
	    {
		edv_object_struct *obj = EDV_OBJECT(
		    gtk_ctree_node_get_row_data(ctree, node)
		);
		if(obj != NULL)
		    EDVMatchObjectTypeString(
			core->mimetype, core->total_mimetypes,
			obj->type,
			obj->permissions,
			obj->full_path,
			&type_str
		);
	    }
	}

	EDVMapMIMETypesListWin(core, type_str, toplevel);

	EDVBrowserSetBusy(browser, FALSE);
}
