#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "progressdialog.h"
#include "fprompt.h"

#include "edv_types.h"
#include "cfg.h"
#include "edv_obj.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "browser.h"
#include "browser_cb.h"
#include "browser_dir_tree.h"
#include "endeavour2.h"
#include "edv_obj_op.h"
#include "edv_cb.h"
#include "edv_open.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVBrowserDirTreeFPromptData	EDVBrowserDirTreeFPromptData;
#define EDV_BROWSER_DIR_TREE_FPROMPT_DATA(p)	((EDVBrowserDirTreeFPromptData *)(p))


/* Utilities */
static edv_object_struct *EDVBrowserDirTreeHasSubDirs(
	const gchar *path,
	const gboolean hide_object_hidden
);
static edv_object_struct *EDVBrowserNewErrorObject(
	const gchar *path, const gchar *error_msg
);
static GtkCTreeNode *EDVBrowserDirTreeInsertDummyNode(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node, GtkCTreeNode *sibling_node
);
static gboolean EDVBrowserDirTreeIsPathNoScan(
	edv_core_struct *core, const gchar *path
);

/* Node Setting */
static void EDVBrowserDirTreeSetNode(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCTree *ctree,
	GtkCTreeNode *node,
	edv_object_struct *obj
);

/* Node Utilities */
GtkCTreeNode *EDVBrowserDirTreeGetToplevel(edv_browser_struct *browser);
static GtkCTreeNode *EDVBrowserDirTreeInsertNode(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCTree *ctree,
	GtkCTreeNode *parent_node, GtkCTreeNode *sibling_node,
	edv_object_struct *obj
);

/* Origin Path */
void EDVBrowserDirTreeSetOriginPath(
	edv_browser_struct *browser,
	const gchar *path
);

/* Get Listing */
static void EDVBrowserDirTreeGetChildrenListIterate(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node,
	const gboolean update_progress, const gboolean recurse,
	const gboolean hide_object_hidden,
	const gboolean hide_object_noaccess,
	const gboolean probe_only
);
static void EDVBrowserDirTreeGetChildrenList(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node,
	const gboolean update_progress, const gboolean recurse
);
void EDVBrowserDirTreeCreateToplevels(edv_browser_struct *browser);
void EDVBrowserDirTreeGetListing(
	edv_browser_struct *browser,
	GtkCTreeNode *node,
	const gboolean update_progress
);
void EDVBrowserDirTreeClear(edv_browser_struct *browser);

/* Expanding */
void EDVBrowserDirTreeExpand(
	edv_browser_struct *browser, GtkCTreeNode *node,
	const gboolean update_progress
);

/* Realize Listing */
static void EDVBrowserDirTreeRealizeListingIterate(
	edv_browser_struct *browser,
	GtkCTreeNode *node,
	const gboolean parent_is_expanded
);
void EDVBrowserDirTreeRealizeListing(
	edv_browser_struct *browser,
	GtkCTreeNode *node
);

/* Selecting */
static void EDVBrowserDirTreeSelectPathIterate(
	edv_browser_struct *browser,
	GtkCTree *ctree, GtkCList *clist,
	GtkCTreeNode *node, const gchar *path,
	GtkCTreeNode **matched_node_rtn,
	GtkCTreeNode **partial_matched_node_rtn
);
void EDVBrowserDirTreeSelectPath(
	edv_browser_struct *browser,
	const gchar *path
);

/* Finding */
static void EDVBrowserDirTreeFindNodeByIndexIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gulong device_index, const gulong index,
	GtkCTreeNode **matched_node
);
GtkCTreeNode *EDVBrowserDirTreeFindNodeByIndex(
	edv_browser_struct *browser,
	const gulong device_index, const gulong index
);
static void EDVBrowserDirTreeFindNodeByPathIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node
);
GtkCTreeNode *EDVBrowserDirTreeFindNodeByPath(
	edv_browser_struct *browser,
	const gchar *path
);

/* Removing */
void EDVBrowserDirTreeRemoveGrandChildrenNodes(
	edv_browser_struct *browser, GtkCTreeNode *node
);

/* Opening */
void EDVBrowserDirTreeOpenWith(
	edv_browser_struct *browser, GtkCTreeNode *node
);

/* Renaming */
static void EDVBrowserDirTreeFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
static void EDVBrowserDirTreeFPromptRenameCancelCB(gpointer data);
void EDVBrowserDirTreePromptRename(
	edv_browser_struct *browser,
	GtkCTreeNode *node
);

/* Object Callbacks */
void EDVBrowserDirTreeObjectAddedNotify(
	edv_browser_struct *browser, const gchar *path,
	struct stat *lstat_buf
);
void EDVBrowserDirTreeObjectModifiedNotify(
	edv_browser_struct *browser,
	const gchar *path, const gchar *new_path,
	struct stat *lstat_buf
);
void EDVBrowserDirTreeObjectRemovedNotify(
	edv_browser_struct *browser, const gchar *path
);

/* Mount Callbacks */
void EDVBrowserDirTreeMountNotify(
	edv_browser_struct *browser, edv_device_struct *dev,
	const gboolean is_mounted
);


struct _EDVBrowserDirTreeFPromptData {
	edv_browser_struct	*browser;
	GtkCTreeNode	*node;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if the directory has any subdirectories.
 *
 *	The path specifies the full path to the directory.
 *
 *	If hide_object_hidden is TRUE then hidden objects will be
 *	excluded.
 *
 *	Returns a new object describing the first subdirectory found
 *	in the directory or NULL if there were no subdirectories.
 */
static edv_object_struct *EDVBrowserDirTreeHasSubDirs(
	const gchar *path,
	const gboolean hide_object_hidden
)
{
	const gchar *name;
	gchar *child_path;
	DIR *dir;
	struct dirent *dent;
	struct stat stat_buf;

	if(STRISEMPTY(path))
	    return(NULL);

	/* Open the directory */
	dir = opendir((const char *)path);
	if(dir == NULL)
	    return(NULL);

	/* Check if there is a subdirectory */
	for(dent = readdir(dir); dent != NULL; dent = readdir(dir))
	{
	    name = (const gchar *)dent->d_name;

	    /* Skip special notations */
	    if(!strcmp((const char *)name, ".") ||
	       !strcmp((const char *)name, "..")
	    )
		continue;

	    /* Skip hidden objects? */
	    if(hide_object_hidden)
	    {
		if(EDVIsObjectNameHidden(name))
		    continue;
	    }

	    /* Get the full path to this object */
	    child_path = g_strconcat(
		path,
		G_DIR_SEPARATOR_S,
		name,
		NULL
	    );
	    if(child_path == NULL)
		continue;

	    /* Does this object lead to a directory? */
	    if(!stat((const char *)child_path, &stat_buf))
	    {
#ifdef S_ISDIR
		if(S_ISDIR(stat_buf.st_mode))
		{
		    /* Create a new object describing this subdirectory */
		    edv_object_struct *obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			struct stat lstat_buf;

			EDVObjectSetPath(obj, child_path);
			if(!lstat((const char *)child_path, &lstat_buf))
			    EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
		    }

		    /* Close the directory and return the new object */
		    g_free(child_path);
		    closedir(dir); 
		    return(obj);
		}
#endif	/* S_ISDIR */
	    }

	    g_free(child_path);
	}

	/* Close the directory */
	closedir(dir);

	return(NULL);
}

/*
 *	Creates a new error object.
 *
 *	Used to create an error object to indicate that the actual
 *	object on disk was not obtainable.
 *
 *	The path specifies the full path to the error object.
 */
static edv_object_struct *EDVBrowserNewErrorObject(
	const gchar *path, const gchar *error_msg
)
{
	edv_object_struct *obj = EDVObjectNew();
	if(obj == NULL)
	    return(NULL);

	obj->type = EDV_OBJECT_TYPE_ERROR;
	EDVObjectSetPath(obj, path);
	EDVObjectUpdateLinkFlags(obj);
	if(!STRISEMPTY(error_msg))
	    EDVObjectPropSet(
		obj,
		EDV_OBJECT_PROP_NAME_ERROR,
		error_msg,
		TRUE
	    );

	return(obj);
}

/*
 *	Inserts a dummy node.
 *
 *	Used to mark an unexpanded directory that it has contents.
 *
 *	Returns the dummy GtkCTreeNode or NULL on error.
 */
static GtkCTreeNode *EDVBrowserDirTreeInsertDummyNode(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node, GtkCTreeNode *sibling_node
)
{
	gint i;
	gchar **strv;
	GtkCTreeNode *node;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the new node's cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Insert the new node */
	node = gtk_ctree_insert_node(
	    ctree,
	    parent_node,
	    sibling_node,
	    strv,				/* Cell values */
	    EDV_LIST_PIXMAP_TEXT_SPACING,	/* Pixmap text spacing */
	    NULL, NULL,				/* Closed pixmap, mask */
	    NULL, NULL,				/* Opened pixmap, mask */
	    TRUE,				/* Is leaf */
	    FALSE				/* Not expanded */
	);

	/* Delete the node's cell values */
	g_free(strv);

	gtk_clist_thaw(clist);

	return(node);
}

/*
 *	Checks if the path should not be scanned for its contents.
 *
 *	All inputs assumed valid.
 */
static gboolean EDVBrowserDirTreeIsPathNoScan(
	edv_core_struct *core, const gchar *path
)
{
	/* Check if this directory is a mounted device that should
	 * not be scanned for its contents
	 */
#if 1
	edv_device_struct *dev = EDVDevicesListMatchObject(
	    core->device, core->total_devices,
	    NULL,
	    path
	);
#else
	edv_device_struct *dev = EDVDevicesListMatchMountPath(
	    core->device, core->total_devices,
	    NULL,
	    path			/* Mount path */
	);
#endif
	if(dev == NULL)
	    return(FALSE);

	if(EDV_DEVICE_IS_NO_SCAN(dev))
	{
	    if(EDV_DEVICE_IS_MOUNTED(dev))
		return(TRUE);
	}

	return(FALSE);
}

/*
 *	Sets the Directory Tree node's icon, text, and style.
 *
 *	The ctree specifies the Directory Tree.
 *
 *	The node specifies the node.
 *
 *	The obj specifies the object who's values will be used to
 *	set the node's icon, text, and style.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeSetNode(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCTree *ctree,
	GtkCTreeNode *node,
	edv_object_struct *obj
)
{
	const gchar *text = obj->name;
	GdkPixmap	*pixmap,
			*pixmap_exp,
			*pixmap_ext,
			*pixmap_hid;
	GdkBitmap	*mask,
			*mask_exp,
			*mask_ext,
			*mask_hid;
	GtkStyle	**cell_styles = browser->cell_style,
			*style = NULL;
	GtkCList *clist = GTK_CLIST(ctree);
	GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);

	if(row_ptr == NULL)
	    return;

	/* Get the appropriate icon for this object */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    EDV_OBJECT_IS_LINK_VALID(obj),
	    obj->permissions,
	    0,			/* Small icons */
	    &pixmap, &mask,
	    &pixmap_exp, &mask_exp,
	    &pixmap_ext, &mask_ext,
	    &pixmap_hid, &mask_hid
	);

	/* Hidden? */
	if(EDVIsObjectHidden(obj))
	{
	    /* Use the hidden icon if it is available */
	    if(pixmap_hid != NULL)
	    {
		pixmap = pixmap_hid;
		mask = mask_hid;
	    }
	}

	/* Directory not accessable? */
	if(!EDVIsObjectAccessable(core, obj))
	{
	    /* Use the directory not accessable icon if it is available */
	    if(pixmap_ext != NULL)
	    {
		pixmap = pixmap_ext;
		mask = mask_ext;
	    }

	    /* Use the directory not accessable style */
	    if(style == NULL)
		style = cell_styles[EDV_BROWSER_CELL_STYLE_NO_ACCESS];
	}

	/* Link target is a grand parent? */
	if(EDV_OBJECT_IS_LINK_TAR_GRAND_PARENT(obj))
	{
	    /* Use the recursive link style */
	    if(style == NULL)
		style = cell_styles[EDV_BROWSER_CELL_STYLE_RECURSIVE_LINK];
	}

	/* If the expanded icon is not available then use the standard
	 * icon as the expanded icon
	 */
	if(pixmap_exp == NULL)
	{
	    pixmap_exp = pixmap;
	    mask_exp = mask;
	}

	/* Set this node */
	gtk_clist_freeze(clist);
	gtk_ctree_set_node_info(
	    ctree, node,
	    (text != NULL) ? text : "(null)",
	    EDV_LIST_PIXMAP_TEXT_SPACING,
	    pixmap, mask,		/* Closed pixmap, mask */
	    pixmap_exp, mask_exp,	/* Opened pixmap, mask */
	    row_ptr->is_leaf,
	    row_ptr->expanded
	);
	gtk_ctree_node_set_row_style(ctree, node, style);
	gtk_clist_thaw(clist);
}


/*
 *	Gets the toplevel node.
 */
GtkCTreeNode *EDVBrowserDirTreeGetToplevel(edv_browser_struct *browser)
{
	GtkCTree *ctree;

	if(browser == NULL)
	    return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* For now just get the first toplevel, later on we may need
	 * to return a node other than the first toplevel if multiple
	 * vfs are to be supported
	 */
	return(EDVNodeGetToplevel(ctree));
}


/*
 *	Inserts a node.
 *
 *	The ctree specifies the Directory Tree.
 *
 *	The parent specifies the parent node that the new node will
 *	be inserted to. If parent is NULL then the new noed will be
 *	a new toplevel node.
 *
 *	The sibling specifies the sibling node that the new node will
 *	be inserted after. If sibling is NULL then the new node will
 *	be appended to the parent's list of child nodes.
 *
 *	The obj specifies the object that will be set as the new
 *	node's data. The obj should not be referenced again after
 *	this call.
 *
 *	Returns the new node or NULL on error.
 *
 *	All inputs assumed valid.
 */
static GtkCTreeNode *EDVBrowserDirTreeInsertNode(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCTree *ctree,
	GtkCTreeNode *parent_node, GtkCTreeNode *sibling_node,
	edv_object_struct *obj
)
{
	gint i;
	gchar **strv;
	GtkCTreeNode *node;
	GtkCList *clist = GTK_CLIST(ctree);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the new node's cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Insert the new node */
	node = gtk_ctree_insert_node(
	    ctree,
	    parent_node,
	    sibling_node,
	    strv,				/* Cell values */
	    EDV_LIST_PIXMAP_TEXT_SPACING,	/* Pixmap text spacing */
	    NULL, NULL,				/* Closed pixmap, mask */
	    NULL, NULL,				/* Opened pixmap, mask */
	    FALSE,				/* Not leaf */
	    FALSE				/* Not expanded */
	);

	/* Delete the cell values */
	g_free(strv);

	/* Failed to insert a new node? */
	if(node == NULL)
	{
	    gtk_clist_thaw(clist);
	    EDVObjectDelete(obj);
	    return(NULL);
	}

	/* Set the new node's text, pixmaps, and style */
	EDVBrowserDirTreeSetNode(
	    core, browser, ctree,
	    node,
	    obj
	);

	/* Set the new node's row data as the specified object */
	gtk_ctree_node_set_row_data_full(
	    ctree, node,
	    obj, EDVBrowserDirTreeItemDestroyCB
	);

	gtk_clist_thaw(clist);

	return(node);
}


/*
 *	Sets the origin path and recreates the toplevel nodes.
 */
void EDVBrowserDirTreeSetOriginPath(
	edv_browser_struct *browser,
	const gchar *path
)
{
	const gchar *origin_path;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;

	if((browser == NULL) || STRISEMPTY(path))
	    return;

	origin_path = browser->directory_ctree_origin_path;
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* No change? */
	if(path == origin_path)
	    return;
	if(origin_path != NULL)
	{
	    if(!strcmp((const char *)origin_path, (const char *)path))
		return;
	}

	/* Set the new origin path */
	g_free(browser->directory_ctree_origin_path);
	browser->directory_ctree_origin_path = STRDUP(path);

	/* Delete all existing nodes and recreate toplevels */
	EDVBrowserDirTreeCreateToplevels(browser);

	/* Select and expand the toplevel node */
	node = EDVBrowserDirTreeGetToplevel(browser);
	if(node != NULL)
	{
	    gtk_clist_freeze(clist);
	    gtk_ctree_expand(ctree, node);
	    gtk_ctree_select(ctree, node);
	    gtk_clist_thaw(clist);
	}
}


/*
 *	Called by EDVBrowserDirTreeGetChildrenList() to get a list
 *	of child nodes.
 *
 *	The parent_node specifies the parent GtkCTreeNode.
 *
 *	If update_progress is TRUE then the status bar will be updated
 *	with the progress informatio nduring this call.
 *
 *	If recurse is TRUE then this call will recurse no more than
 *	2 levels from the parent node. If recurse is FALSE then no
 *	recursion will take place.
 *
 *	If hide_object_hidden is TRUE then hidden objects will be
 *	excluded.
 *
 *	If hide_object_noaccess is TRUE then no access objects will
 *	be excluded.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeGetChildrenListIterate(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node,
	const gboolean update_progress, const gboolean recurse,
	const gboolean hide_object_hidden,
	const gboolean hide_object_noaccess,
	const gboolean probe_only
)
{
	struct stat lstat_buf, stat_buf;
	gint i, strc;
	gchar *parent_path, **strv;
	const gchar *s;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	edv_status_bar_struct *sb = browser->status_bar;
	edv_core_struct *core = browser->core;

	/* Get the parent node's object */
	edv_object_struct *obj = EDV_OBJECT(
	    gtk_ctree_node_get_row_data(ctree, parent_node)
	);
	if(obj == NULL)
	    return;

	/* Get the object's parent path */
	parent_path = STRDUP(obj->full_path);
	if(parent_path == NULL)
	    return;

	/* Do not reference the object past this point */
/*	obj = NULL; */

	/* Update the status bar? */
	if(update_progress)
	{
	    const gchar *name = g_basename(parent_path);
	    gchar *msg = g_strdup_printf(
		"Scanning directory: %s",
		(name != NULL) ? name : parent_path
	    );
	    EDVStatusBarMessage(sb, msg, FALSE);
	    g_free(msg);
	}

	/* Probe only? */
	if(probe_only)
	{
	    /* Check if this directory has at least one subdirectory */
	    edv_object_struct *obj = EDVBrowserDirTreeHasSubDirs(
		parent_path, hide_object_hidden
	    );
	    if(obj != NULL)
	    {
		/* Add a child node with the object describing the
		 * first subdirectory found in this directory
		 */
		EDVBrowserDirTreeInsertNode(
		    core, browser, ctree,
		    parent_node, NULL,
		    obj
		);
	    }
	    g_free(parent_path);
	    return;
	}

	/* Get this directory's contents listing */
	strv = GetDirEntNames2(parent_path, &strc);
	if(strv != NULL)
	{
	    gchar *full_path;

	    /* Sort the contents listing */
	    strv = StringQSort(strv, strc);
	    if(strv == NULL)
	    {
		g_free(parent_path);
		return;
	    }

	    /* Iterate through each child object */
	    full_path = NULL;
	    for(i = 0; i < strc; i++)
	    {
#define FREE_AND_CONTINUE	{	\
 g_free(full_path);			\
 full_path = NULL;			\
					\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
					\
 continue;				\
}
		s = strv[i];
		if(s == NULL)
		    FREE_AND_CONTINUE

		/* Skip special dir notations */
		if(!strcmp((const char *)s, "..") ||
		   !strcmp((const char *)s, ".")
		)
		    FREE_AND_CONTINUE

		/* Get full path to child object */
		full_path = STRDUP(PrefixPaths(
		    (const char *)parent_path,
		    (const char *)s
		));
		if(full_path == NULL)
		    FREE_AND_CONTINUE

		/* Get the local and destination stats of child
		 * object
		 */
		if(lstat((const char *)full_path, &lstat_buf))
		{
#if 0
/* Do not add error objects on the directory tree */
		    obj = EDVBrowserNewErrorObject(full_path, NULL);
		    node = EDVBrowserDirTreeInsertNode(
			core, browser, ctree,
			parent_node, NULL,
			obj
		    );
#endif
		    FREE_AND_CONTINUE
		}
		if(stat((const char *)full_path, &stat_buf))
		{
#if 0
/* Do not add error objects on the tree */
		    obj = EDVBrowserNewErrorObject(full_path, NULL);
		    node = EDVBrowserDirTreeInsertNode(
			core, browser, ctree,
			parent_node, NULL,
			obj
		    );
#endif
		    FREE_AND_CONTINUE
		}

		/* Is this child object's destination a directory? */
#ifdef S_ISDIR
		if(S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    gboolean no_scan_device;
		    GtkCTreeNode *node;
		    edv_object_struct *obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, full_path);
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);

			/* Begin filter checks */
			if((hide_object_hidden ?
			    EDVIsObjectHidden(obj) : FALSE) ||
			   (hide_object_noaccess ?
			    !EDVIsObjectAccessable(core, obj) : FALSE)
			)
			{
			    EDVObjectDelete(obj);
			    node = NULL;
			    no_scan_device = FALSE;
			}
			else
			{
			    /* Check if this directory should not be
			     * scanned for its contents
			     */
			    no_scan_device = EDVBrowserDirTreeIsPathNoScan(
				core, obj->full_path
			    );

			    /* Add/transfer the object to the tree */
			    node = EDVBrowserDirTreeInsertNode(
				core, browser, ctree,
				parent_node, NULL,
				obj
			    );
			}
		    }
		    else
		    {
			node = NULL;
			no_scan_device = FALSE;
		    }

		    /* Report progress */
		    if(update_progress)
			EDVStatusBarProgress(sb, -1.0f, TRUE);

		    /* Recurse into subdirectories? */
		    if(recurse && (node != NULL))
		    {
			if(no_scan_device)
			    EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
			else
			    EDVBrowserDirTreeGetChildrenListIterate(
				browser,
				node,
				update_progress,
				FALSE,		/* Do not recurse */
				hide_object_hidden,
				hide_object_noaccess,
				TRUE		/* Always probe for the
						 * second recursion */
			    );
		    }
		}	/* Is this child object's destination a directory? */

		FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
	    }

	    /* Delete the object names list */
	    g_free(strv);
	}

	g_free(parent_path);
}

/*
 *	Gets a list of child nodes from the parent node, any existing
 *	child nodes in the parent node will be deleted.
 *
 *	The parent_node specifies the parent GtkCTreeNode.
 *
 *	If update_progress is TRUE then the status bar will be updated
 *	with the progress informatio nduring this call.
 *
 *	If recurse is TRUE then this call will recurse no more than
 *	2 levels from the parent node. If recurse is FALSE then no
 *	recursion will take place.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeGetChildrenList(
	edv_browser_struct *browser,
	GtkCTreeNode *parent_node,
	const gboolean update_progress, const gboolean recurse
)
{
	gboolean hide_object_hidden, hide_object_noaccess;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	edv_core_struct *core = browser->core;
	const cfg_item_struct *cfg_list = core->cfg_list;
	edv_status_bar_struct *sb = browser->status_bar;

	/* Get the display options */
	hide_object_hidden = !EDV_GET_B(
	    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDV_GET_B(
	    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
	);

	/* Report the initial progress? */
	if(update_progress)
	{
	    EDVStatusBarMessage(
		sb,
		"Scanning directories...",
		FALSE
	    );
	    EDVStatusBarProgress(sb, -1.0f, TRUE);
	}

	gtk_clist_freeze(clist);

	/* Delete all the child nodes from the parent node */
	row_ptr = GTK_CTREE_ROW(parent_node);
	if(row_ptr != NULL)
	{
	    while(row_ptr->children != NULL)
		gtk_ctree_remove_node(ctree, row_ptr->children);
	}

	/* Get the new listing */
	EDVBrowserDirTreeGetChildrenListIterate(
	    browser,
	    parent_node,
	    update_progress,
	    recurse,
	    hide_object_hidden,
	    hide_object_noaccess,
	    FALSE				/* Do not probe */
	);

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(update_progress)
	{
	    EDVStatusBarMessage(sb, NULL, FALSE);
	    EDVStatusBarProgress(sb, 0.0f, FALSE);
	}
}

/*
 *	Creates the toplevel nodes.
 *
 *	Any existing nodes will be deleted.
 */
void EDVBrowserDirTreeCreateToplevels(edv_browser_struct *browser)
{
	const gchar *path;
	GtkCList *clist;
	GtkCTree *ctree;
	edv_object_struct *obj;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	gtk_clist_freeze(clist);

	/* Delete all the existing nodes */
	EDVBrowserDirTreeClear(browser);

	/* Get the origin path as the path for the toplevel node */
	path = browser->directory_ctree_origin_path;
	if(STRISEMPTY(path))
#if defined(_WIN32)
	    path = "C:\\";
#else
	    path = "/";
#endif

	/* Create the object for the toplevel node */
	obj = EDVObjectNew();
	if(obj != NULL)
	{
	    struct stat lstat_buf;
	    GtkCTreeNode *node;

	    /* Get the toplevel's local statistics */
	    if(lstat((const char *)path, &lstat_buf))
	    {
		const gint error_code = (gint)errno;
		EDVObjectDelete(obj);
		obj = EDVBrowserNewErrorObject(path, g_strerror(error_code));
	    }
	    else
	    {
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(obj);
	    }

	    /* Add/transfer the object to the tree */
	    node = EDVBrowserDirTreeInsertNode(
		core, browser, ctree,
		NULL, NULL,
		obj
	    );
	    if(node != NULL)
	    {
		/* Get one level of child nodes */
		if(EDVBrowserDirTreeIsPathNoScan(core, path))
		    EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
		else
		    EDVBrowserDirTreeGetChildrenList(
			browser,
			node,			/* Parent node */
			TRUE,			/* Update progress */
			FALSE			/* Do not recurse */
		    );
	    }
	}

	gtk_clist_thaw(clist);
}

/*
 *	Gets a listing of child nodes from the specified node,
 *	recursing no more than 2 levels.
 *
 *	The node specifies the GtkCTreeNode at which to get the
 *	listing of child nodes for. If node is NULL then the toplevel
 *	GtkCTreeNode will be used.
 *
 *	Any existing child nodes in the specified node will be deleted
 *	first before getting the new list of child nodes.
 */
void EDVBrowserDirTreeGetListing(
	edv_browser_struct *browser,
	GtkCTreeNode *node,
	const gboolean update_progress
)
{
	GtkCList *clist;
	GtkCTree *ctree;

	if(browser == NULL)
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	gtk_clist_freeze(clist);

	/* Use the toplevel node if the specified node is NULL */
	if(node == NULL)
	    node = EDVBrowserDirTreeGetToplevel(browser);

	/* Delete any existing child nodes from the specified node
	 * and then get a new list of child nodes
	 */
	if(node != NULL)
	    EDVBrowserDirTreeGetChildrenList(
		browser,
		node,				/* Parent node */
		update_progress,
		TRUE				/* Recurse */
	    );

	gtk_clist_thaw(clist);
}

/*
 *	Deletes all nodes on the Directory Tree.
 */
void EDVBrowserDirTreeClear(edv_browser_struct *browser)
{
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;

	if(browser == NULL)
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	node = EDVBrowserDirTreeGetToplevel(browser);
	if(node != NULL)
	{
	    gtk_clist_freeze(clist);
	    gtk_ctree_remove_node(ctree, node);
	    gtk_clist_thaw(clist);
	}
}


/*
 *	Expands the node and gets a listing of child nodes that
 *	recurse no more than 2 levels.
 *
 *	The node specifies the GtkCTreeNode to expand and to
 *	get a listing of child nodes for.
 */
void EDVBrowserDirTreeExpand(
	edv_browser_struct *browser, GtkCTreeNode *node,
	const gboolean update_progress
)
{
	GtkCList *clist;
	GtkCTree *ctree;
	edv_core_struct *core;

	if((browser == NULL) || (node == NULL))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	gtk_clist_freeze(clist);

	/* Get child directories as child nodes of the given expanded
	 * node, recursing at most 2 levels
	 */
	EDVBrowserDirTreeGetChildrenList(
	    browser,
	    node,				/* Parent node */
	    update_progress,
	    TRUE				/* Recurse */
	);

	gtk_clist_thaw(clist);
}


/*
 *	Called by EDVBrowserDirTreeRealizeListing() to update existing
 *	nodes and to remove all non-existant nodes, then recurse into
 *	any child nodes.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeRealizeListingIterate(
	edv_browser_struct *browser,
	GtkCTreeNode *node,
	const gboolean parent_is_expanded
)
{
	const gchar *path;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCTreeRow *row_ptr;
	GtkCTreeNode *next_node;
	edv_object_struct *obj;
	edv_core_struct *core = browser->core;

	/* Iterate from the specified node through its siblings */
	while(node != NULL)
	{
	    row_ptr = GTK_CTREE_ROW(node);
	    if(row_ptr == NULL)
		break;

	    /* Get this node's object */
	    obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	    path = (obj != NULL) ? obj->full_path : NULL;

	    /* Check this node's child objects only if this node's
	     * device is not marked "no scan"
	     */
	    if(!EDVBrowserDirTreeIsPathNoScan(core, path))
	    {
		/* Recurse into and realize each child node */
		EDVBrowserDirTreeRealizeListingIterate(
		    browser,
		    row_ptr->children,	/* Child node */
		    row_ptr->expanded
		);

		/* Does this node no longer have any children? */
		if(row_ptr->children == NULL)
		{
		    /* If the parent was expanded then check if there
		     * are any child nodes now by getting a new
		     * children listing
		     *
		     * This should only be done if the parent is
		     * expanded so that we do not get child listings
		     * for nodes that do not need to be indicated as
		     * expandable
		     *
		     * Recurse one level
		     */
		    if(parent_is_expanded && (path != NULL))
			EDVBrowserDirTreeGetChildrenList(
			    browser,
			    node,		/* Parent node */
			    FALSE,		/* Do not update progress */
			    FALSE		/* Do not recurse */
			);
		}
	    }

	    /* Get this node's sibling */
	    next_node = row_ptr->sibling;

	    /* Does this node have an object and its path is set? */
	    if(path != NULL)
	    {
		/* Check if this node's object no longer exists or
		 * its destination is no longer a directory
		 */
		struct stat lstat_buf;

		/* Check if this object no longer exists locally */
		if(lstat((const char *)path, &lstat_buf))
		{
		    /* No longer exists, remove this node */
		    gtk_ctree_remove_node(ctree, node);
#if 0
		    node = NULL;
		    row_ptr = NULL;
		    obj = NULL;
		    path = NULL;
#endif
		}
		else
		{
		    /* Exists locally, check if its destination exists */
		    struct stat stat_buf;
		    if(stat((const char *)path, &stat_buf))
		    {
			/* Destination no longer exists, remove this node */
			gtk_ctree_remove_node(ctree, node);
#if 0
			node = NULL;
			row_ptr = NULL;
			obj = NULL;
			path = NULL;
#endif
		    }
		    /* Is the destination no longer a directory? */
#ifdef S_ISDIR
		    else if(!S_ISDIR(stat_buf.st_mode))
#else
		    else if(TRUE)
#endif
		    {
			/* Destination is no longer a directory, remove this node */
			gtk_ctree_remove_node(ctree, node);
#if 0
			node = NULL;
			row_ptr = NULL;
			obj = NULL;
			path = NULL;
#endif
		    }
		    else
		    {
			/* All checks passed, update this object and
			 * the node
			 */
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
			EDVBrowserDirTreeSetNode(
			    core,
			    browser,
			    ctree,
			    node,
			    obj
			);
		    }
		}
	    }
	    /* Does this node not have an object? */
	    else if(obj == NULL)
	    {
		/* Remove this node */
		gtk_ctree_remove_node(ctree, node);
#if 0
		node = NULL;
		row_ptr = NULL;
#endif
	    }

	    /* The node, row_ptr, obj, and path should be considered
	     * invalid after this point
	     */

	    /* Get the next sibling */
	    node = next_node;
	}
}

/*
 *	Realizes all the nodes from the specified node.
 *
 *	Each node will be checked to see if its object exists and is
 *	a directory, if it is not then the node will be removed.
 *
 *	If a node currently does not have any children then a new
 *	listing of child objects will be obtained for that node only
 *	if the node's parent is expanded. If the specified node does
 *	not have any children then a new listing of child objects will
 *	be obtained for the specified node regardless of if its parent
 *	is expanded or not.
 *
 *	The node specifies the GtkCTreeNode at which to realize the
 *	listing at. If node is NULL then the toplevel GtkCTreeNode
 *	will be used.
 */
void EDVBrowserDirTreeRealizeListing(
	edv_browser_struct *browser,
	GtkCTreeNode *node
)
{
	GtkCList *clist;
	GtkCTree *ctree;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Use toplevel node if the specified node is NULL */
	if(node == NULL)
	    node = EDVBrowserDirTreeGetToplevel(browser);
	if(node == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Begin updating the nodes and removing any non-existant
	 * nodes
	 */
	EDVBrowserDirTreeRealizeListingIterate(
	    browser,
	    node,
	    TRUE				/* Always indicate the
						 * specified node's
						 * parent as expanded */
	);

	gtk_clist_thaw(clist);
}


/*
 *	Called by EDVBrowserDirTreeSelectPath() to seek through all
 *	child directories for a prefix match to the given full path and
 *	recurse into child directories.
 *
 *	The given path is the full path that is intended to be selected
 *	if found.
 */
static void EDVBrowserDirTreeSelectPathIterate(
	edv_browser_struct *browser,
	GtkCTree *ctree, GtkCList *clist,
	GtkCTreeNode *node, const gchar *path,
	GtkCTreeNode **matched_node_rtn,
	GtkCTreeNode **partial_matched_node_rtn	
)
{
	const gchar *cur_path;
	edv_object_struct *obj;

	/* Got match? If so then we should not continue the search */
	if(*matched_node_rtn != NULL)
	    return;

	if(node == NULL)
	    return;

	/* Get this node's object */
	obj = EDV_OBJECT(
	    gtk_ctree_node_get_row_data(ctree, node)
	);

	/* Get this object's full path */
	cur_path = (obj != NULL) ? obj->full_path : NULL;
	if(STRISEMPTY(cur_path))
	    return;

	/* Is the cur_path a parent of the search path? */
	if(EDVIsParentPath(cur_path, path))
	{
	    GtkCTreeRow *row_ptr;

	    /* Update the partial matched node */
	    *partial_matched_node_rtn = node;

	    /* Check if the search path and the current path are a
	     * complete match
	     */
	    if(!strcmp((const char *)path, (const char *)cur_path))
	    {
		/* Got complete match, set matched node and return */
		*matched_node_rtn = node;
		return;
	    }
	    else
	    {
		/* Not a complete match, but cur_path is a parent of
		 * the search path so we are on the right track
		 *
		 * Get child node and iterate through all its siblings
		 * and children
		 */

		/* Expand current node as needed
		 *
		 * Note that this may call the expand callback and that
		 * it will load more child nodes
		 */
		row_ptr = GTK_CTREE_ROW(node);
		if(row_ptr != NULL)
		{
		    if(!row_ptr->expanded)
		    {
			GtkCList *clist = GTK_CLIST(ctree);
			gtk_clist_freeze(clist);
			gtk_ctree_expand(ctree, node);
			gtk_clist_thaw(clist);
		    }
		}

		/* Get the child nodes */
		node = EDVNodeGetChild(node);
		while(node != NULL)
		{
		    /* Check this node, recurse if it is a match */
		    EDVBrowserDirTreeSelectPathIterate(
			browser,
			ctree, clist,
			node, path,
			matched_node_rtn,
			partial_matched_node_rtn
		    );
		    /* If we got a match, then do not continue search */
		    if((matched_node_rtn != NULL) ? (*matched_node_rtn != NULL) : TRUE)
			return;

		    /* Get the next sibling */
		    node = EDVNodeGetSibling(node);
		}
	    }
	}
}

/*
 *	Selects the node that corresponds to the path or a parent
 *	node of the path if the exact node is not found. Nodes will
 *	be expanded and updated as needed.
 *
 *	The path specifies the full path.
 */
void EDVBrowserDirTreeSelectPath(
	edv_browser_struct *browser,
	const gchar *path
)
{
	gchar *dpath;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node, *matched_node, *partial_matched_node;

	if((browser == NULL) || STRISEMPTY(path))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* Path must be absolute */
	if(!g_path_is_absolute(path))
	    return;

	/* Make a copy of the path */
	dpath = STRDUP(path);
	if(dpath == NULL)
	    return;

	/* Strip any tailing deliminators in the path */
	StripPath(dpath);

	/* Get the toplevel node to start the search from */
	node = EDVBrowserDirTreeGetToplevel(browser);
	if(node == NULL)
	{
	    g_free(dpath);
	    return;
	}

	/* Search for the path, expanding and updating nodes as
	 * needed
	 */
	matched_node = NULL;
	partial_matched_node = NULL;
	EDVBrowserDirTreeSelectPathIterate(
	    browser,
	    ctree, clist,
	    node,
	    dpath,
	    &matched_node,
	    &partial_matched_node
	);

	g_free(dpath);

	/* Found the node? */
	if(matched_node != NULL)
	{
	    /* Select the matched node */
	    gtk_clist_freeze(clist);
	    gtk_ctree_select(ctree, matched_node);
	    gtk_clist_thaw(clist);
	}
	/* Found a partially matched parent node? */
	else if(partial_matched_node != NULL)
	{
	    /* Select the matched parent node */
	    gtk_clist_freeze(clist);
	    gtk_ctree_select(ctree, partial_matched_node);
	    gtk_clist_thaw(clist);
	}
}


/*
 *	Performs one iteration of finding the Directory Tree node by
 *	location index.
 */
static void EDVBrowserDirTreeFindNodeByIndexIterate(
	GtkCTree *ctree, 
	GtkCTreeNode *node,
	const gulong device_index, const gulong index,
	GtkCTreeNode **matched_node
)
{
	edv_object_struct *obj;

	if((*matched_node != NULL) || (node == NULL))
	    return;

	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	    return;

	/* Does this object's index location match the specified index
	 * location?
	 */
	if((obj->device_index == device_index) &&
	   (obj->index == index)
	)
	{
	    *matched_node = node;
	    return;
	}
	else
	{
	    /* No match, now check if any of this node's child nodes
	     * match
	     */
	    GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
	    if(row_ptr == NULL)
		return;

	    node = row_ptr->children;
	    while(node != NULL)
	    {
		EDVBrowserDirTreeFindNodeByIndexIterate(
		    ctree,
		    node,
		    device_index, index,
		    matched_node
		);
		if(*matched_node != NULL)
		    return;

		/* Get the sibling */
		row_ptr = GTK_CTREE_ROW(node);
		if(row_ptr == NULL)
		    break;

		node = row_ptr->sibling;
	    }
	}
}

/*
 *	Finds the node by index.
 *
 *	The device_index specifies the index of the device that the
 *	object resides on.
 *
 *	The index specifies the index on the device that the object
 *	resides at.
 *
 *	Returns the matched node or NULL on error.
 */
GtkCTreeNode *EDVBrowserDirTreeFindNodeByIndex(
	edv_browser_struct *browser,                  
	const gulong device_index, const gulong index
)
{
	GtkCTreeNode *node, *matched_node;
	GtkCTree *ctree;

	if(browser == NULL)
	    return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Get the toplevel node to start the search at */
	node = EDVBrowserDirTreeGetToplevel(browser);

	/* Begin the search */
	matched_node = NULL;
	EDVBrowserDirTreeFindNodeByIndexIterate(
	    ctree,
	    node,
	    device_index, index,
	    &matched_node
	);

	return(matched_node);
}

/*
 *	Performs one iteration of finding the Directory Tree node by
 *	path.
 */
static void EDVBrowserDirTreeFindNodeByPathIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node
)
{
	const gchar *cur_path;
	edv_object_struct *obj;

	if((*matched_node != NULL) || (node == NULL))
	    return;

	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	    return;

	/* Get this object's full path */
	cur_path = obj->full_path;
	if(STRISEMPTY(cur_path))
	    return;

	/* Is the specified node's path a prefix of the specified path? */
	if(EDVIsParentPath(cur_path, path))
	{
	    /* Yes it is, now check if it is a complete match */
	    if(!strcmp((const char *)path, (const char *)cur_path))
	    {
		*matched_node = node;
		return;
	    }
	    else
	    {
		/* Not a complete match, check the child nodes */
		node = EDVNodeGetChild(node);
		while(node != NULL)
		{
		    EDVBrowserDirTreeFindNodeByPathIterate(
			ctree,
			node,
			path,
			matched_node
		    );
		    if(*matched_node != NULL)
		        return;

		    /* Get the sibling */
		    node = EDVNodeGetSibling(node);
		}
	    }
	}
}

/*
 *	Finds the node by path.
 *
 *	The path specifies the string describing the full path to
 *	match a node that has an object who has the same path.
 *
 *	Returns the matched node or NULL on error.
 */
GtkCTreeNode *EDVBrowserDirTreeFindNodeByPath(
	edv_browser_struct *browser,
	const gchar *path
)
{
	gchar *full_path;
	GtkCTreeNode *node, *matched_node;
	GtkCTree *ctree;

	if((browser == NULL) || STRISEMPTY(path))
	    return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Path must be absolute */
	if(!g_path_is_absolute(path))
	    return(NULL);

	/* Make a copy of the given path as the full path */
	full_path = STRDUP(path);
	if(full_path == NULL)
	    return(NULL);

	/* Strip tailing deliminators */
	StripPath(full_path);

	/* Get the toplevel node to start search from */
	node = EDVBrowserDirTreeGetToplevel(browser);

	/* Begin searching for a node who's object's fuil path matches
	 * the specified path
	 *
	 * matched_node_ptr will point to the matched node if a match
	 * was made
	 */
	matched_node = NULL;
	EDVBrowserDirTreeFindNodeByPathIterate(
	    ctree,
	    node,
	    full_path,
	    &matched_node
	);

	g_free(full_path);

	return(matched_node);
}


/*
 *	Removes all the grandchild nodes.
 *
 *	The node specifies the parent node.
 */
void EDVBrowserDirTreeRemoveGrandChildrenNodes(
	edv_browser_struct *browser, GtkCTreeNode *node
)
{
	GtkCList *clist;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree;

	if((browser == NULL) || (node == NULL))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	row_ptr = GTK_CTREE_ROW(node);
	if(row_ptr == NULL)
	    return;

	/* Get the specified node's childrens list */
	node = row_ptr->children;

	/* No children? */
	if(node == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Iterate through the specified node's childrens list */
	while(node != NULL)
	{
	    row_ptr = GTK_CTREE_ROW(node);
	    if(row_ptr == NULL)
		break;

	    /* Remove any child nodes in this node */
	    while(row_ptr->children != NULL)
		gtk_ctree_remove_node(ctree, row_ptr->children);

	    /* Get the next node */
	    node = row_ptr->sibling;
	}

	gtk_clist_thaw(clist);
}


/*
 *	Open with.
 */
void EDVBrowserDirTreeOpenWith(
	edv_browser_struct *browser, GtkCTreeNode *node
)
{
	const gchar *name;
	gchar *full_path;
	struct stat stat_buf;
	GtkWidget *toplevel;
	GtkCTree *ctree;
	edv_object_struct *obj;
	edv_core_struct *core;

	if((browser == NULL) || (node == NULL))
	    return;

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;
	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	    return;

	/* Check the name of the object for special notations */
	name = obj->name;
	if(name != NULL)
	{
	    if(!strcmp(name, "."))
	    {
		full_path = NULL;
	    }
	    else if(!strcmp(name, ".."))
	    {
		full_path = g_dirname(
		    EDVBrowserCurrentLocation(browser)
		);
	    }
	    else
	    {
		full_path = STRDUP(obj->full_path);
	    }
	}
	else
	{
	    /* No name available, use the path */
	    full_path = STRDUP(obj->full_path);
	}
	if(full_path == NULL)
	    return;

	/* Get stats of destination object, the object structure may
	 * have that information already but we want to ensure we have
	 * the most up to date information
	 */
	if(stat((const char *)full_path, &stat_buf))
	{
	    g_free(full_path);
	    return;
	}

	if(TRUE)
	{
	    gchar       *stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
	    GList *paths_list = NULL;

	    paths_list = g_list_append(
		paths_list,
		STRDUP(full_path)
	    );

	    EDVOpenWith(
		core,
		paths_list,		/* Paths List */
		NULL,			/* Command Name */
		toplevel,		/* Toplevel */
		TRUE,			/* Verbose */
		&stdout_path_rtn,
		&stderr_path_rtn
	    );

	    g_list_foreach(paths_list, (GFunc)g_free, NULL);
	    g_list_free(paths_list);

	    g_free(stdout_path_rtn);
	    g_free(stderr_path_rtn);
	}

	g_free(full_path);
}


/*
 *	FPrompt rename apply callback.
 */
static void EDVBrowserDirTreeFPromptRenameApplyCB(
	gpointer data, const gchar *value
)
{
	GtkCTreeNode *node;
	edv_browser_struct *browser;
	EDVBrowserDirTreeFPromptData *d = EDV_BROWSER_DIR_TREE_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	browser = d->browser;
	node = d->node;

	/* Inputs valid? */
	if((browser != NULL) && (node != NULL) && (value != NULL))
	{
	    GtkWidget *toplevel = browser->toplevel;
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    edv_status_bar_struct *sb = browser->status_bar;
	    edv_core_struct *core = browser->core;

	    /* Get the object from the selected node */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, node)
	    );
	    if((obj != NULL) ? (obj->full_path != NULL) : FALSE)
	    {
		gboolean yes_to_all = FALSE;
		const gchar *error_msg;
		gchar *old_full_path = STRDUP(obj->full_path);
		GList *modified_paths_list;

		/* Rename */
		EDVObjectOPRename(
		    core,
		    old_full_path, value,
		    &modified_paths_list,
		    toplevel,
		    FALSE,			/* Do not show progress */
		    TRUE,			/* Interactive */
		    &yes_to_all
		);

		/* Unmap the progress dialog */
		ProgressDialogBreakQuery(FALSE);
		ProgressDialogSetTransientFor(NULL);

		/* Check for errors */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Rename Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Notify about the renamed objects */
		if(modified_paths_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *modified_path;
		    GList *glist;

		    for(glist = modified_paths_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			modified_path = (const gchar *)glist->data;
			if(modified_path == NULL)
			    continue;

			if(!lstat(modified_path, &lstat_buf))
			{
			    gchar *msg = g_strdup_printf(
				"Object \"%s\" renamed to \"%s\"",
				g_basename(old_full_path),
				g_basename(modified_path)
			    );
			    EDVStatusBarMessage(sb, msg, FALSE);
			    g_free(msg);

			    EDVObjectModifiedEmit(
				core,
				old_full_path,
				modified_path,
				&lstat_buf
			    );
			}
		    }

		    if(modified_paths_list != NULL)
		    {
			g_list_foreach(
			    modified_paths_list, (GFunc)g_free, NULL
			);
			g_list_free(modified_paths_list);
		    }
		}
		else
		{
		    /* Did not get the modified object path so this
		     * implies failure
		     */
		    EDVStatusBarMessage(
			sb,
			"Rename object failed",
			FALSE
		    );
		}

		g_free(old_full_path);
	    }
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void EDVBrowserDirTreeFPromptRenameCancelCB(gpointer data)
{
	EDVBrowserDirTreeFPromptData *d = EDV_BROWSER_DIR_TREE_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	g_free(d);
}

/*
 *	Maps the FPrompt over the File Browser directory tree item
 *	to rename.
 */
void EDVBrowserDirTreePromptRename(
	edv_browser_struct *browser,
	GtkCTreeNode *node
)
{
	gint row, column;
	gint cx, cy, px, py, pwidth, pheight;
	GtkWidget *toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	edv_object_struct *obj;
	edv_core_struct *core;

	if((browser == NULL) || (node == NULL) || FPromptIsQuery())
	    return;

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	EDVBrowserSyncData(browser);

	/* Get the row and column of the specified node */
	if(!EDVNodeGetIndex(ctree, node, &row, &column))
	    return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
	    clist, column, row,
	    &cx, &cy, &pwidth, &pheight
	))
	    return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
	    clist->clist_window, &px, &py
	);
/*	px += cx + 0; */
	py += cy - 2;	/* Move up a little */

	/* Set the prompt width to match width of the GtkCTree */
	pwidth = GTK_WIDGET(ctree)->allocation.width;

	/* Get the object */
	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	    return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(obj->name != NULL)
	{
	    const gchar *name = obj->name;
	    if(!strcmp(name, ".") || !strcmp(name, "..") ||
	       !strcmp(name, "/")
	    )
		return;
	}

	if(TRUE)
	{
	    gchar *value = STRDUP(obj->name);
	    EDVBrowserDirTreeFPromptData *d = EDV_BROWSER_DIR_TREE_FPROMPT_DATA(
		g_malloc(sizeof(EDVBrowserDirTreeFPromptData))
	    );
	    if(d != NULL)
	    {
		d->browser = browser;
		d->node = node;
	    }

	    /* Map floating prompt to change values */
	    FPromptSetTransientFor(browser->toplevel);
	    FPromptSetPosition(px, py);
	    FPromptMapQuery(
		NULL,			/* No label */
		value,
		NULL,			/* No tooltip message */
		FPROMPT_MAP_NO_MOVE,	/* Map code */
		pwidth, -1,		/* Width and height */
		0,			/* No buttons */
		d,			/* Callback data */
		NULL,			/* No browse callback */
		EDVBrowserDirTreeFPromptRenameApplyCB,
		EDVBrowserDirTreeFPromptRenameCancelCB
	    );

	    g_free(value);
	}
}


/*
 *	Object added callback.
 */
void EDVBrowserDirTreeObjectAddedNotify(
	edv_browser_struct *browser, const gchar *path,
	struct stat *lstat_buf
)
{
	struct stat stat_buf;
	gchar *parent_path;
	GtkCList *clist;
	GtkCTreeNode *node, *parent_node;
	GtkCTree *ctree;
	edv_core_struct *core;

	if((browser == NULL) || STRISEMPTY(path) || (lstat_buf == NULL))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Get the added object's destination statistics */
	if(stat((const char *)path, &stat_buf))
	    return;

	/* Skip if the added object's destination is not a directory */
#ifdef S_ISDIR
	if(!S_ISDIR(stat_buf.st_mode))
#else
	if(TRUE)
#endif
	    return;

	/* Update the object who's path matches the added path */
	node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	if(node != NULL)
	{
	    /* Update this object */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, node)
	    );
	    if(obj != NULL)
	    {
		GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
		const gboolean was_expanded = (row_ptr != NULL) ?
		    row_ptr->expanded : FALSE;

		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		gtk_clist_freeze(clist);

		/* Update the node */
		EDVBrowserDirTreeSetNode(
		    core, browser, ctree,
		    node,
		    obj
		);

		/* Get the new listing of child objects for the
		 * updated node, recurse no more than 2 levels
		 */
		if(EDVBrowserDirTreeIsPathNoScan(core, path))
		    EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
		else
		    EDVBrowserDirTreeGetChildrenList(
			browser,
			node,			/* Parent node */
			TRUE,			/* Update progress */
			TRUE			/* Recurse */
		    );

		if(was_expanded)
		    gtk_ctree_expand(ctree, node);

		gtk_clist_thaw(clist);
	    }
	    return;
	}

	/* Get the added object's parent */
	parent_path = g_dirname(path);
	if(parent_path == NULL)
	    return;

	/* Get the parent node of the added object */
	parent_node = EDVBrowserDirTreeFindNodeByPath(browser, parent_path);
	if(parent_node != NULL)
	{
	    /* Create the new object and node for the added object */
	    edv_object_struct *obj = EDVObjectNew();
	    if(obj != NULL)
	    {
		/* Set the new object */
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		gtk_clist_freeze(clist);

		/* Create the new node */
		node = EDVBrowserDirTreeInsertNode(
		    core, browser, ctree,
		    parent_node, NULL,
		    obj
		);
		if(node != NULL)
		{
		    /* Get the new listing of child objects for the
		     * new node, recursing no more than 2 levels
		     */
		    if(EDVBrowserDirTreeIsPathNoScan(core, path))
			EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
		    else
			EDVBrowserDirTreeGetChildrenList(
			    browser,
			    node,		/* Parent node */
			    TRUE,		/* Update progress */
			    TRUE		/* Recurse */
			);
		}

		gtk_clist_thaw(clist);
	    }
	}

	g_free(parent_path);
}

/*
 *	Object modified callback.
 */
void EDVBrowserDirTreeObjectModifiedNotify(
	edv_browser_struct *browser,
	const gchar *path, const gchar *new_path,
	struct stat *lstat_buf
)
{
	struct stat stat_buf;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	edv_core_struct *core;
	if((browser == NULL) || STRISEMPTY(path) || (lstat_buf == NULL))
	    return;

	if(new_path == NULL)
	    new_path = path;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Check if the object's destination is no longer a directory */
	if(!stat((const char *)new_path, &stat_buf))
	{
#ifdef S_ISDIR
	    if(!S_ISDIR(stat_buf.st_mode))
	    {
		/* Remove all the nodes who's object's path matches
		 * the modified path
		 */
		GtkCTreeNode *node = EDVBrowserDirTreeFindNodeByPath(browser, path);
		if(node != NULL)
		{
		    gtk_clist_freeze(clist);
		    do {
			gtk_ctree_remove_node(ctree, node);
			node = EDVBrowserDirTreeFindNodeByPath(browser, path);
		    } while(node != NULL);
		    gtk_clist_thaw(clist);
		}
		return;
	    }
#endif	/* S_ISDIR */
	}

	/* Update the node who's object's path matches the modified
	 * path
	 */
	node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	if(node != NULL)
	{
	    /* Update this object */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, node)
	    );
	    if(obj != NULL)
	    {
		GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
		const gboolean was_expanded = (row_ptr != NULL) ?
		    row_ptr->expanded : FALSE;

		EDVObjectSetPath(obj, new_path);	/* Use the new path */
		EDVObjectSetStat(obj, lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		gtk_clist_freeze(clist);

		/* Update this node */
		EDVBrowserDirTreeSetNode(
		    core, browser, ctree,
		    node,
		    obj
		);

		/* Get the new listing of child objects for the updated
		 * node, recurse no more than 2 levels
		 */
		if(EDVBrowserDirTreeIsPathNoScan(core, path))
		    EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
		else
		    EDVBrowserDirTreeGetChildrenList(
			browser,
			node,			/* Parent node */
			TRUE,			/* Update progress */
			TRUE			/* Recurse */
		    );

		if(was_expanded)
		    gtk_ctree_expand(ctree, node);

		gtk_clist_thaw(clist);
	    }
	}
	else
	{
	    /* No node existed for the modified path, create a new
	     * node for the new path
	     */
	    gchar *new_parent_path = g_dirname(new_path);
	    if(new_parent_path != NULL)
	    {
		/* Find the node for the new path's parent */
		GtkCTreeNode *parent_node = EDVBrowserDirTreeFindNodeByPath(
		    browser, new_parent_path
		);
		if(parent_node != NULL)
		{
		    /* Create a new node for the new path */
		    edv_object_struct *obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, new_path);
			EDVObjectSetStat(obj, lstat_buf);
			EDVObjectUpdateLinkFlags(obj);

			gtk_clist_freeze(clist);

			/* Add/transfer the object to the tree */
			node = EDVBrowserDirTreeInsertNode(
			    core, browser, ctree,
			    parent_node, NULL,
			    obj
			);
			if(node != NULL)
			{
			    /* Get new listing of child disk objects
			     * for the new node, recurse no more than 2
			     * levels
			     */
			    if(EDVBrowserDirTreeIsPathNoScan(core, path))
				EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
			    else
				EDVBrowserDirTreeGetChildrenList(
				    browser,
				    node,	/* Parent node */
				    TRUE,	/* Update progress */
				    TRUE	/* Recurse */
				);
			}

			gtk_clist_thaw(clist);
		    }
		}
		g_free(new_parent_path);
	    }
	}
}

/*
 *	Object removed callback.
 */
void EDVBrowserDirTreeObjectRemovedNotify(
	edv_browser_struct *browser, const gchar *path
)
{
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	edv_core_struct *core;

	if((browser == NULL) || STRISEMPTY(path))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
 
	/* Remove all the nodes that match the removed path */
	node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	if(node != NULL)
	{
	    gtk_clist_freeze(clist);
	    do {
		gtk_ctree_remove_node(ctree, node);
		node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	    } while(node != NULL);
	    gtk_clist_thaw(clist);
	}
}


/*
 *	Mount/unmount notify callback.
 */
void EDVBrowserDirTreeMountNotify(
	edv_browser_struct *browser, edv_device_struct *dev,
	const gboolean is_mounted
)
{
	gchar *mount_path;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	edv_core_struct *core;

	if((browser == NULL) || (dev == NULL))
	    return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Get the mount path */
	mount_path = STRDUP(dev->mount_path);
	if(mount_path == NULL)
	    return;

	/* Simplify the mount path */
	EDVSimplifyPath(mount_path);

	/* Find the node that matches this mount path */
	node = EDVBrowserDirTreeFindNodeByPath(browser, mount_path);
	if(node != NULL)
	{
	    /* Update this node and its object */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, node)
	    );
	    if(obj != NULL)
	    {
		struct stat lstat_buf;
		if(!lstat((const char *)mount_path, &lstat_buf))
		{
		    EDVObjectSetPath(obj, mount_path);
		    EDVObjectSetStat(obj, &lstat_buf);
		    EDVObjectUpdateLinkFlags(obj);

		    gtk_clist_freeze(clist);

		    EDVBrowserDirTreeSetNode(
			core, browser, ctree,
			node,
			obj
		    );

		    /* Get new listing of child objects for the
		     * updated node, recurse no more than 2 levels
		     */
		    if(EDVBrowserDirTreeIsPathNoScan(core, mount_path))
			EDVBrowserDirTreeInsertDummyNode(browser, node, NULL);
		    else
			EDVBrowserDirTreeGetChildrenList(
			    browser,
			    node,		/* Parent node */
			    TRUE,		/* Update progress */
			    TRUE		/* Recurse */
			);

		    gtk_clist_thaw(clist);
		}
	    }
	}

	/* Unmounted? */
	if(!is_mounted)
	{
	    /* Is the current location within the mount path? */
	    gchar *cur_loc = STRDUP(EDVBrowserCurrentLocation(browser));
	    if(cur_loc != NULL)
	    {
		if(strpfx((const char *)cur_loc, (const char *)mount_path))
		{
		    /* Select the unmounted device's node as needed */
		    GtkCTreeNode *cur_node = EDVBrowserDirTreeFindNodeByPath(
			browser, cur_loc
		    );
		    if((node != NULL) && (node != cur_node))
		    {
			gtk_clist_freeze(clist);
			gtk_ctree_select(ctree, node);
			gtk_clist_thaw(clist);
		    }
		}
		g_free(cur_loc);
	    }
	}

	g_free(mount_path);
}
