/*
  emilda-print
  Part of the Emilda project (http://www.emilda.org/)
	
  Copyright (C) 2003-2004 Realnode Ltd

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  
  Authors:
    Mattias Nordstrom <nordstrom (at) realnode.com>

  $Id: print.cpp,v 1.7 2005/01/26 17:23:11 mnordstr Exp $
    This file provides the printing functions.

*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <iostream>

#include "print.h"

using namespace std;

int print_epd(epd *epd, DOTCONFDocument *conf)
{
	int result = 0;
	
	switch (epd->document.type) {
		case EPD_TYPE_RECEIPT:
			result = print_receipt(epd, conf);
			break;
		case EPD_TYPE_LABEL:
			result = print_label(epd, conf);
			break;
		case EPD_TYPE_PS:
			result = print_ps(epd, conf);
			break;
	}
	
	return result;
}

// This is for the Star TSP600 series.
int print_receipt(epd *epd, DOTCONFDocument *conf)
{
	const DOTCONFDocumentNode *pnode = conf->findNode("ReceiptPrinter");
	const DOTCONFDocumentNode *node = conf->findNode("Path", pnode);

	string::size_type pos = 0;
	string header((char[5]) {0x1b, 0x69, '1', '1', '\0'});

	if ((pos = epd->document.data.find("\n", pos)) != string::npos) {
		header += epd->document.data.substr(0, pos);
		header += (char[5]) {0x1b, 0x69, '0', '0', '\0'};
		epd->document.data.erase(0, pos+1);
	}
	
	pos = 0;
	
	if ((pos = epd->document.data.find("<HEADER>", pos)) != string::npos) {
		epd->document.data.replace(pos, 8, header);
	}
	
	pos = 0;
	
	while ((pos = epd->document.data.find_first_of("", pos)) != string::npos) {
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 93);
		}
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 92);
		}
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 91);
		}
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 125);
		}
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 124);
		}
		if (!epd->document.data.compare(pos, 1, "")) {
			epd->document.data.replace(pos, 1, 1, 123);
		}
	}

	// Feed n' cut.
	epd->document.data += (char[6]) {0x0a, 0x0a, 0x1b, 0x64, 0x03, '\0'};

	string path = node->getValue();

	return print_data(&path, &epd->document.data);	
}

// This is for the Zebra TLP2844.
int print_label(epd *epd, DOTCONFDocument *conf)
{
	const DOTCONFDocumentNode *pnode = conf->findNode("LabelPrinter");
	const DOTCONFDocumentNode *node = conf->findNode("Path", pnode);

	string::size_type pos = 0;
	string header("\nD12\nS2\nq608\nN\nA45,5,0,4,1,1,N,\"");

	if ((pos = epd->document.data.find("\n", pos)) != string::npos) {
		header += epd->document.data.substr(0, pos);
		epd->document.data.erase(0, pos+1);
	}
	header += "\"\n";
	
	string barcode, prn_data;
	pos = 0;		
	
	while ((pos = epd->document.data.find("\n", 0)) != string::npos) {
		barcode = "B45,35,0,3,3,6,100,N,\"";
		barcode += epd->document.data.substr(0, pos);
		barcode += "\"\nA210,145,0,4,1,1,N,\"";
		barcode += epd->document.data.substr(0, pos);
		barcode += "\"\nP1\n";
		epd->document.data.erase(0, pos+1);
		
		prn_data += header;
		prn_data += barcode;
	}
	
	string path = node->getValue();

	return print_data(&path, &prn_data);
}

// This is for any PostScript capable printer.
int print_ps(epd *epd, DOTCONFDocument *conf)
{
	const DOTCONFDocumentNode *pnode = conf->findNode("PSPrinter");
	const DOTCONFDocumentNode *node = conf->findNode("Path", pnode);

	string path = node->getValue();
	
	node = conf->findNode("UsePJL", pnode);
	string pjl = node->getValue();

	if (pjl == "yes") { // Add PJL headers and footers.
		string ps_data = epd->document.data;
		
		epd->document.data = ((char [49])
			{0x1b, 0x25, 0x2d, 0x31, 0x32, 0x33, 0x34, 0x35, 0x58, 0x40, 0x50, 0x4a, 0x4c, 0x20, 0x0a, 0x40,
			0x50, 0x4a, 0x4c, 0x20, 0x45, 0x4e, 0x54, 0x45, 0x52, 0x20, 0x4c, 0x41, 0x4e, 0x47, 0x55, 0x41,
			0x47, 0x45, 0x20, 0x3d, 0x20, 0x50, 0x4f, 0x53, 0x54, 0x53, 0x43, 0x52, 0x49, 0x50, 0x54, 0x0a, '\0'});

		epd->document.data += ps_data;
	 
		epd->document.data += ((char [25])
			{0x1b, 0x25, 0x2d, 0x31, 0x32, 0x33, 0x34, 0x35, 0x58, 0x40, 0x50, 0x4a, 0x4c, 0x20, 0x0a, 0x1b,
			0x25, 0x2d, 0x31, 0x32, 0x33, 0x34, 0x35, 0x58, '\0'});

	}
	
	if (print_data(&path, &epd->document.data)) {
		return 1;
	} else {
		sleep(8);
		return 0;
	}
}

int print_data(string *path, string *data)
{
	int result = 0;
	
	FILE *prn = fopen(path->c_str(), "w");
	if (prn == NULL) {
		return 1;
	}
	
	size_t written = fwrite((void *)data->c_str(), sizeof(char[data->length()]), 1, prn);
	if (written != 1) {
		result = 1;
	}
	
	if (fclose(prn) != 0 || result == 1) {
		result = 1;
	}
	
	return result;
}
