// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"

template< typename T, int Size, typename MatrixType, typename VectorType >
void helper_checkBasicStuff( const char *strType )
{
    MatrixType mat1(Size); mat1.loadIdentity();
    MatrixType mat2(Size); mat2.loadRandom();
    VectorType vec1(Size); vec1.loadZero();
    VectorType vec2(Size); vec2.loadRandom();;

    for( int i = 0; i < Size; i++ )
        QVERIFY( test_isApprox( vec2[i], vec2(i) ) );

    for( int i = 0; i < Size; i++ )
        QVERIFY( test_isApprox( ((const VectorType)vec2)[i], ((const VectorType)vec2)(i) ) );

    for( int i = 0; i < Size; i++ )
        for( int j = 0; j < Size; j++ )
            QVERIFY( test_isApprox( mat2[i + j * Size], mat2(i, j) ) );

    for( int i = 0; i < Size; i++ )
        for( int j = 0; j < Size; j++ )
            QVERIFY( test_isApprox( ((const MatrixType)mat2)[i + j * Size], ((const MatrixType)mat2)(i, j) ) );

    QVERIFY( compareMatrices( mat2, mat1 * mat2, strType ) );
    QVERIFY( compareMatrices( mat2, mat2 * mat1, strType ) );
    QVERIFY( compareVectors( vec2, mat1 * vec2, strType ) );
    QVERIFY( compareVectors( vec2, vec2 * mat1, strType ) );
    QVERIFY( compareVectors( vec2, vec2 *= mat1, strType ) );
    QVERIFY( compareVectors( vec2, vec1 + vec2, strType ) );

    mat1.multiply( vec2, &vec1 );
    QVERIFY( compareVectors( vec2, vec1, strType ) );
    mat1.leftmultiply( vec2, &vec1 );
    QVERIFY( compareVectors( vec2, vec1, strType ) );

    MatrixType mat3(Size); pickRandomMatrix(mat3);
    VectorType vec3(Size); pickRandomVector(vec3);
    pickRandomMatrix(mat1);
    pickRandomVector(vec1);

    QVERIFY( compareMatrices( mat1 * (mat2 + mat3) , mat1 * mat2 + mat1 * mat3, strType ) );
    QVERIFY( compareMatrices( mat1 - mat2 - mat3 + mat2, mat1 - mat3, strType ) );
    QVERIFY( compareVectors( vec1 - vec2 - vec3 + vec2, vec1 - vec3, strType ) );

    T a1 = static_cast<T>(1);
    QVERIFY( compareMatrices( mat1 * a1, mat1, strType ) );
    QVERIFY( compareMatrices( a1 * mat1, mat1, strType ) );
    QVERIFY( compareVectors( vec1 * a1, vec1, strType ) );
    QVERIFY( compareVectors( a1 * vec1, vec1, strType ) );

    T a2;
    pickRandom(a1);
    pickRandom(a2);
    QVERIFY( compareMatrices( mat1 * ( a1 + a2 ), mat1 * a1 + mat1 * a2, strType ) );
    QVERIFY( compareMatrices( ( a1 - a2 ) * mat1, a1 * mat1 - a2 * mat1, strType ) );
    QVERIFY( compareVectors( vec1 * ( a1 + a2 ), vec1 * a1 + vec1 * a2, strType ) );
    QVERIFY( compareVectors( ( a1 - a2 ) * vec1, a1 * vec1 - a2 * vec1, strType ) );
    QVERIFY( compareMatrices( ( mat1 * mat2 ) * ( a1 * a2 ), ( mat1 * a1 ) * ( mat2 * a2 ), strType ) );
    QVERIFY( compareMatrices( ( mat1 / a1 ) * a1, mat1, strType ) );
    QVERIFY( compareVectors( ( vec1 / a1 ) * a1, vec1, strType ) );

    mat1.multiply( mat2, & mat3 );
    QVERIFY( compareMatrices( mat1 * mat2, mat3, strType ) );

    mat3 = mat1; mat3 *= mat2;
    QVERIFY( compareMatrices( mat1 * mat2, mat3, strType ) );
    mat3 = mat1; mat3 += mat2;
    QVERIFY( compareMatrices( mat1 + mat2, mat3, strType ) );
    mat3 = mat1; mat3 -= mat2;
    QVERIFY( compareMatrices( mat1 - mat2, mat3, strType ) );
    mat3 = mat1; mat3 *= a1;
    QVERIFY( compareMatrices( mat1 * a1, mat3, strType ) );
    mat3 = mat1; mat3 /= a1;
    QVERIFY( compareMatrices( mat1 / a1, mat3, strType ) );

    vec3 = vec1; vec3 += vec2;
    QVERIFY( compareVectors( vec1 + vec2, vec3, strType ) );
    vec3 = vec1; vec3 -= vec2;
    QVERIFY( compareVectors( vec1 - vec2, vec3, strType ) );
    vec3 = vec1; vec3 *= a1;
    QVERIFY( compareVectors( vec1 * a1, vec3, strType ) );
    vec3 = vec1; vec3 /= a1;
    QVERIFY( compareVectors( vec1 / a1, vec3, strType ) );

    mat3.loadZero();
    QVERIFY( compareMatrices( mat1 + mat3, mat1, strType ) );

    vec1.loadRandom();
    vec2 = -vec1;
    vec3.loadRandom();
    QVERIFY( compareVectors( vec1 + vec2 + vec3, vec3, strType ) );
    vec2 = vec1;
    vec2.replaceWithOpposite();
    QVERIFY( compareVectors( vec1 + vec2 + vec3, vec3, strType ) );

    mat1.loadRandom();
    mat2 = -mat1;
    mat3.loadRandom();
    QVERIFY( compareMatrices( mat1 + mat2 + mat3, mat3, strType ) );
    mat2 = mat1;
    mat2.replaceWithOpposite();
    QVERIFY( compareMatrices( mat1 + mat2 + mat3, mat3, strType ) );

    mat1.loadIdentity();
    mat2.loadIdentity();
    mat1.multiplyEntries( mat2, & mat3 );
    QVERIFY( compareMatrices( mat3, mat1, strType ) );
}

void MainTest::checkBasicStuff()
{
#define HELPER_CHECKBASICSTUFF_FIX( T, Size ) \
    helper_checkBasicStuff< T, Size, \
        Matrix< T, Size >, Vector< T, Size > >\
            ("fixed<"__STRING(T)","__STRING(Size)">")

#define HELPER_CHECKBASICSTUFF_DYN( T, Size ) \
    helper_checkBasicStuff< T, Size, \
        MatrixX<T>, VectorX<T> >\
            ("dynamic<"__STRING(T)">("__STRING(Size)")")

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKBASICSTUFF_FIX( float, 1 );
        HELPER_CHECKBASICSTUFF_FIX( double, 2 );
        HELPER_CHECKBASICSTUFF_FIX( float, 3 );
        HELPER_CHECKBASICSTUFF_FIX( double, 4 );
        HELPER_CHECKBASICSTUFF_FIX( double, 7 );
        HELPER_CHECKBASICSTUFF_FIX( float, 8 );
        HELPER_CHECKBASICSTUFF_FIX( double, 12 );
        HELPER_CHECKBASICSTUFF_FIX( complex<double>, 7 );
        HELPER_CHECKBASICSTUFF_FIX( complex<float>, 8 );
        HELPER_CHECKBASICSTUFF_FIX( complex<double>, 12 );
    
        HELPER_CHECKBASICSTUFF_DYN( float, 1 );
        HELPER_CHECKBASICSTUFF_DYN( double, 2 );
        HELPER_CHECKBASICSTUFF_DYN( float, 3 );
        HELPER_CHECKBASICSTUFF_DYN( double, 4 );
        HELPER_CHECKBASICSTUFF_DYN( double, 7 );
        HELPER_CHECKBASICSTUFF_DYN( float, 8 );
        HELPER_CHECKBASICSTUFF_DYN( double, 12 );
        HELPER_CHECKBASICSTUFF_DYN( complex<double>, 7 );
        HELPER_CHECKBASICSTUFF_DYN( complex<float>, 8 );
        HELPER_CHECKBASICSTUFF_DYN( complex<double>, 12 );
    }
}
