/*
    Eiciel - GNOME editor of ACL file permissions.
    Copyright (C) 2004-2005 Roger Ferrer Ibez

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "eiciel_container.hpp"

EicielContainer::EicielContainer()
	: escollirFitxer(Gtk::Stock::OPEN), 
	etiquetaFitxer(_("<b>File name</b>")),
	nomFitxer(_("No file opened")),
	sortir(Gtk::Stock::QUIT),
	ajuda(Gtk::Stock::HELP),
	quantA(_("About..."))
{

	set_title("Eiciel");
	set_border_width(4);
	
	// set_icon_from_file(PKGDATADIR "/img/icona_eiciel.png");
	std::vector<Glib::RefPtr<Gdk::Pixbuf> > llista;
	Glib::RefPtr<Gdk::Pixbuf> icona;

	icona = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/icona_eiciel_16.png");
	llista.push_back(icona);
	icona = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/icona_eiciel_24.png");
	llista.push_back(icona);
	icona = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/icona_eiciel_32.png");
	llista.push_back(icona);
	icona = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/icona_eiciel_48.png");
	llista.push_back(icona);
	icona = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/icona_eiciel_64.png");
	llista.push_back(icona);

	set_icon_list(llista);

	add(principal);

	principal.pack_start(superior, Gtk::PACK_SHRINK, 2);
	etiquetaFitxer.set_use_markup();
	superior.pack_start(etiquetaFitxer, Gtk::PACK_SHRINK, 2);
	superior.pack_start(nomFitxer, Gtk::PACK_EXPAND_WIDGET, 4);
	superior.pack_start(escollirFitxer, Gtk::PACK_SHRINK, 4);
	superior.set_homogeneous(false);

	principal.add(paginador);
	paginador.append_page(paginaACL, _("Access Control List"));

	controlador_widget = new EicielMainControler();
	widget_principal = new EicielWindow(controlador_widget);

	paginaACL.pack_start(*widget_principal, Gtk::PACK_EXPAND_WIDGET, 0);

#ifdef ENABLE_USER_XATTR
	paginador.append_page(paginaXAttr, _("Extended user attributes"));

	controlador_xattr = new EicielXAttrControler();
	widget_xattr = new EicielXAttrWindow(controlador_xattr);

	paginaXAttr.pack_start(*widget_xattr, Gtk::PACK_EXPAND_WIDGET, 0);
#endif	

	principal.pack_start(inferior, Gtk::PACK_SHRINK, 2);
	inferior.set_spacing(2);
	inferior.add(quantA);
	inferior.add(ajuda);
	inferior.add(sortir);

	sortir.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielContainer::sortidaAplicacio)
			);

	quantA.signal_clicked().connect (
			sigc::mem_fun(*this, &EicielContainer::mostrarQuantA)
			);

	ajuda.signal_clicked().connect (
			 sigc::mem_fun(*this, &EicielContainer::mostrarAjuda)
			);

	escollirFitxer.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielContainer::oberturaFitxer)
			);

	show_all_children();
}

void EicielContainer::mostrarAjuda()
{
    GError* error = NULL;

    gnome_help_display("eiciel", NULL, &error);
    if (error != NULL)
    {
        g_warning(_("Could not show the help file: %s"), error->message);
        g_error_free(error);
    }
}

void EicielContainer::sortidaAplicacio()
{
	Gtk::Main::quit();
}

void EicielContainer::mostrarQuantA()
{
	EicielAboutBox about(*this);
	about.run();
}

void EicielContainer::chooser_file_activated(Gtk::FileChooserDialog* dialeg)
{
	dialeg->response(Gtk::RESPONSE_NONE);
}

void EicielContainer::oberturaFitxer()
{
    // Gtk::FileSelection dialog(_("Choose a file or a directory"));
	Gtk::FileChooserDialog dialog(_("Choose a file or a directory"), 
			Gtk::FILE_CHOOSER_ACTION_OPEN);
	dialog.set_local_only(true);
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_NONE);
	dialog.set_transient_for(*this);

	dialog.signal_file_activated().connect(
			sigc::bind<Gtk::FileChooserDialog*>(
				sigc::mem_fun(*this, &EicielContainer::chooser_file_activated), &dialog)
			);
	
	int result = dialog.run();
	dialog.hide();
	switch (result)
	{
		case(Gtk::RESPONSE_NONE):
			{
				if (!this->obrirFitxer(dialog.get_filename()))
				{
					Gtk::MessageDialog missatge(_("Could not open the file \"") 
							+ Glib::locale_to_utf8(dialog.get_filename()) + Glib::ustring("\" ") 
							+ Glib::ustring("(") + controlador_widget->ultimError() + Glib::ustring(")"),
							false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
					missatge.run();
				}
				break;
			}
		case(Gtk::RESPONSE_CANCEL):
			{
				break;
			}
		default:
			{
				break;
			}
	}
}

bool EicielContainer::obrirFitxer(Glib::ustring nom)
{
	bool resultat;
	controlador_widget->obreFitxer(nom);

	resultat = controlador_widget->fitxerObert();

#ifdef ENABLE_USER_XATTR
	controlador_xattr->obrirFitxer(nom);

	resultat |= controlador_xattr->fitxerObert();
#endif

	if (resultat)
	{
		nomFitxer.set_text(nom);
	}
	else
	{
		nomFitxer.set_text(_("No file opened"));
	}

	return resultat;
}

/* About box */
EicielAboutBox::EicielAboutBox(Gtk::Window& parent)
	: Gtk::Dialog (_("About..."), parent, true, true),
	titol("<span size=\"xx-large\"><b>Eiciel " PACKAGE_VERSION "</b></span>"),
	autor("<small>Copyright © 2004-2005 Roger Ferrer Ibáñez</small>")
{
	set_border_width(4);
	get_vbox()->set_spacing(4);
	titol.set_use_markup();
	autor.set_use_markup();
	get_vbox()->add(titol);
	get_vbox()->add(autor);
	add_button(Gtk::Stock::OK, 0);
	show_all_children();
}
