/* Copyright (C) 2005 to 2010 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#ifndef CGU_WINDOW_H
#define CGU_WINDOW_H

#include <gtk/gtk.h>

#include <c++-gtk-utils/cgu_config.h>

/** 
 * @class Cgu::WinBase window.h c++-gtk-utils/window.h
 * @brief This is a class for managing the lifetime of top level
 * widgets.
 * @sa MainWidgetBase
 *
 * This class provides a base class for lifetime management of top
 * level widgets which can make GTK+ exception safe.  It would be
 * possible to use GTK+ to control the lifetime of the contained GTK+
 * window object, through the delete event handler and the destroy
 * signal.  However in the case of a blocking window (say a dialog)
 * this would result in an invalid object remaining in scope.  It is
 * better to use the C++ lifetime handling mechanisms to control
 * object status, and this class does that.  Where a window is created
 * as an auto (local) object it will be valid while it remains in
 * scope.  For a window which is not blocking (on which exec() is not
 * called) and which is therefore created on free store the WinBase
 * class will delete its own memory when it is closed (see below about
 * the close() function).
 *
 * If a NULL pointer is passed as the last argument of its
 * constructor, a new window object will be created with
 * gtk_window_new(GTK_WINDOW_TOPLEVEL).  However, alternatively a
 * pre-formed object deriving from GtkWindow (such as a GtkDialog or
 * GtkMessageDialog object) can be passed as that argument, in which
 * case the class will manage that object.
 *
 * A window will block with a call to exec(), which returns an int
 * value.  The exec() method obtains its return value by calling the
 * virtual protected function get_exec_val(), which by default returns
 * 0 but can be overridden to return something more meaningful.  If
 * something other than an int needs to be returned, it will be
 * necessary for a derived class to provide an extractor function to
 * obtain the data from the object after exec() has returned.
 *
 * A convenience virtual protected on_delete_event() method is
 * provided for any derived class which needs to respond to that
 * signal.  It does not return a value (it does not cause a destroy
 * event to be emitted as in the case of a GTK+ delete event handler
 * returning false).
 *
 * If the class is constructed on free store and exec() has not been
 * called on it, it self-destroys and any memory allocated to it freed
 * by calling the protected close() function.  If exec() has been
 * called for an object, a call to close() will only terminate the
 * window event loop (that is, cause exec() to unblock), which is
 * correct for an auto (local) object as it will destroy itself when
 * it goes out of scope.  By default (that is, if not overridden by
 * any derived classes) on_delete_event() calls close().
 *
 * No special memory management for GTK+ widgets contained in the
 * WinBase object (or in an object of a class derived from WinBase) is
 * needed.  The GTK+ object system takes care of that, and they will
 * be released when the life of the WinBase object finishes.  For this
 * reason, the WinBase class can be used to make GTK+ exception safe:
 * as its constructor does not throw, exceptions thrown in the
 * constructors of classes derived from it will be correctly dealt
 * with without causing GTK+ to leak memory if a widget is put into
 * its container in the derived class's constructor immediately after
 * creation on a "top down" basis, or it is placed in another widget
 * class derived from MainWidgetBase.
 *
 * If a C string is passed to the caption argument of the constructor,
 * then the window will display that text as its caption.  A NULL
 * pointer can be passed if no caption is required.  Likewise if a
 * pointer to a GdkPixbuf object is passed as the second parameter, it
 * will be used to set a window icon in the window bar (if the user's
 * window manager supports this).
 *
 * The constructor of WinBase does not call gtk_widget_show() because
 * the constructor of a derived class may want to do this which
 * requires the widget not to be visible.  Accordingly the constructor
 * of the derived class (or the library user) should call
 * gtk_widget_show()/gtk_widget_show_all() via WinBase::get_win()) .
 *
 * A small compilable example is as follows:
 *
 * @anchor WinBaseExampleAnchor
 * @code
 *   #include <gtk/gtk.h>
 *   #include <c++-gtk-utils/window.h>
 *
 *
 *   // *** class headers ***
 *
 *   extern "C" void message_button_clicked(GtkWidget*, void*);
 *
 *   class Message: public Cgu::WinBase {
 *   public:
 *     friend void message_button_clicked(GtkWidget*, void*);
 *     Message(const char* text);
 *   };
 *
 *   
 *   // *** class implementation ***
 *
 *   void message_button_clicked(GtkWidget*, void* data) {
 *     static_cast<Message*>(data)->close();
 *   }
 *
 *   Message::Message(const char* text): Cgu::WinBase("Message", 0, true) {
 *     GtkWidget* box = gtk_vbox_new(false, 2);
 *     gtk_container_add(GTK_CONTAINER(get_win()), box);
 *     GtkWidget* label = gtk_label_new(text);
 *     gtk_box_pack_start(GTK_BOX(box), label,
 *                        true, false, 0);
 *     GtkWidget* button_box = gtk_hbutton_box_new();
 *     gtk_box_pack_start(GTK_BOX(box), button_box,
 *                        false, false, 0);
 *     GtkWidget* button = gtk_button_new_from_stock(GTK_STOCK_OK);
 *     gtk_container_add(GTK_CONTAINER(button_box), button);
 *     g_signal_connect(G_OBJECT(button), "clicked",
 *                      G_CALLBACK(message_button_clicked), this);
 *     gtk_widget_set_can_default(button, true);
 *
 *     gtk_widget_show_all(GTK_WIDGET(get_win()));
 *   }
 *
 *
 *   // *** user code ***
 *
 *   int main(int argc, char* argv[]) {
 *     gtk_init(&argc, &argv);
 *     Message dialog("This is a message");
 *     dialog.exec();
 *     return 0;
 *   }
 * @endcode
 *
 * See @ref Linkage for further discussion of the
 * message_button_clicked() callback.
 *
 * @note The WinBase class will work fine if a GtkDialog object is
 * passed to the last argument of the class's constructor formed from
 * a call to gtk_dialog_new()/gtk_dialog_new_with_buttons(), but a few
 * points should be noted:
 *
 * @note 1. If the response signal has been connected to, a delete
 * event will cause a response signal to be emitted with the
 * GTK_RESPONSE_DELETE_EVENT id, as well as causing the normal
 * WinBase::on_delete_event() method to be called.  If the response
 * handler acts on the GTK_RESPONSE_DELETE_EVENT id by calling
 * close(), then WinBase::on_delete_event() should be overridden to do
 * nothing so that a double call to close() is not made (although
 * version 1.0.3 onwards of c++-gtk-utils will check against and
 * prevent such a double call of close() from a single delete event).
 *
 * @note 2. It is usually best not to call gtk_dialog_run().  Instead,
 * call WinBase::exec() for a blocking dialog, and override
 * WinBase::get_exec_val() to return any required button response id.
 *
 * @note 3. If creating a GtkDialog object with
 * gtk_dialog_new_with_buttons(), do not pass a GtkDialogFlags
 * argument of GTK_DIALOG_DESTROY_WITH_PARENT.  A GtkDialogFlags
 * argument of GTK_DIALOG_MODAL can be passed, but alternatively to
 * make the dialog modal the user can pass 'true' as the modal
 * argument of the WinBase constructor and pass the parent widget to
 * that constructor rather than to gtk_dialog_new_with_buttons() (so
 * that GtkDialogFlags(0) is passed as the third argument of
 * gtk_dialog_new_with_buttons()).  Either approach will work, but the
 * second has the advantage of consistency with the WinBase interface.
 *
 * @note 4. Likewise, if using gtk_dialog_new_with_buttons(), any
 * dialog caption can be passed either to the title argument of that
 * method or to the caption argument of the WinBase constructor.
 *
 * @note Similar principles apply to the management of objects derived from
 * GtkDialog.
 */

namespace Cgu {

class WinBase {
  // main class object
  GtkWindow* g_window_p;

  bool in_exec_loop;
  bool is_modal;
  bool close_guard;
  GtkWindow* parent_p;

  // WinBase cannot be copied
  WinBase(const WinBase&);
  WinBase& operator=(const WinBase&);
protected:
  /**
   * A convenience function for the use of derived classes which will
   * cause the window to unblock if exec() has been called.  If it is
   * not a blocking window (ie exec() has not been called so it has
   * been constructed on freestore), the window will delete itself.
   * By default it is called by on_delete_event().  This method will
   * not throw (unless a derived class's destructor throws, which it
   * should not do).
   */
  void close();

  /**
   * Provides the value to be returned by exec().  This method will
   * not throw (unless it is overridden by a derived class's method
   * which throws).
   * @return By default returns 0.  It is intended to be overridden by
   * derived classes where relevant to provide a more meaningful
   * value.
   */
  virtual int get_exec_val() const;

  /**
   * Called when there is a delete event on the managed window.
   * Unless overridden by derived classes it just calls close().
   */
  virtual void on_delete_event();
public:
#ifndef DOXYGEN_PARSING
  // this helper class avoids exposing GObject callbacks with C
  // linkage to the global namespace
  class CB;
  friend class CB;
#endif
  /**
   * Returns the GtkWindow object managed by this class.  This method
   * will not throw.
   * @return The managed GtkWindow object.
   */
  GtkWindow* get_win() const {return g_window_p;}

  /**
   * Makes the window block. Calls gtk_main().  It is usually a bad
   * idea to call this method (and so have nested loops) on a
   * non-modal dialog.  This method will not throw.
   * @return The value returned by get_exec_val().
   */
  int exec();

  /**
   * A convenience function which calls
   * gtk_widget_show_all(GTK_WIDGET(get_win())).  For classes which
   * are intended to be derived from, it may be undesirable to call
   * gtk_widget_show_all() in the class's constructor (a derived class
   * may add widgets of its own in a way which requires the base
   * class's widgets not to be realized).  In such a case, calling
   * gtk_widget_show_all() should be left to the user code.  This is a
   * function which makes that less verbose.
   *
   * Since 1.2.5
   */
   void show_all() {gtk_widget_show_all(GTK_WIDGET(get_win()));}

  /**
   * The constructor will not throw.
   * @param caption Window caption (optional).
   * @param icon A pixbuf which will comprise the window icon (optional).
   * @param modal Whether the window is to be modal.  If this argument
   * is false, the user may want to consider calling
   * gtk_window_set_type_hint() if the window is a dialog before it
   * becomes visible.
   * @param parent The parent of a modal dialog or NULL (this argument
   * is only relevant and acted on where the modal argument is true).
   * The parent will be made insensitive while the modal dialog is in
   * existence, and automatically made sensitive again when the modal
   * dialog is finished with.  gtk_window_set_transient_for() is also
   * called, which means that the dialog will normally be kept on top
   * of its parent and/or centered over it by the window manager.  If
   * the user does not want the parent to appear as insensitive, pass
   * NULL to this argument and call gtk_window_set_transient_for() in
   * the derived class's constructor.
   * @param window A preformed GtkWindow object (such as a GtkDialog
   * object), or NULL.  If NULL, gtk_window_new(GTK_WINDOW_TOPLEVEL)
   * will be called.
   */
  WinBase(const char* caption = 0, GdkPixbuf* icon = 0, bool modal = false,
	  GtkWindow* parent = 0, GtkWindow* window = 0);

  /**
   * The destructor will not throw.
   */
  virtual ~WinBase();

#ifdef CGU_USE_GLIB_MEMORY_SLICES_NO_COMPAT
  CGU_GLIB_MEMORY_SLICES_FUNCS
#endif
};

} // namespace Cgu

#endif
