# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-

# Copyright (C) 2010-2012 Stephane Graber
# Author: Stephane Graber <stgraber@ubuntu.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import debconf
import os

from ubiquity.plugin import Plugin, PluginUI

NAME = 'edu-target'
BEFORE = 'partman'
WEIGHT = 10
OEM = False

PATH_WORKSTATION = "/cdrom/casper/filesystem.squashfs"
PATH_TERMSERVER = "/cdrom/casper/ltsp.squashfs"
PATH_SERVER = "/cdrom/casper/server.squashfs"


class PageGtk(PluginUI):
    plugin_title = 'ubiquity/text/edu-target_heading_label'

    def __init__(self, controller, *args, **kwargs):
        self.controller = controller

        # Don't show at all if both ltsp and server squashfs files
        # aren't on the media
        if not os.path.exists(PATH_TERMSERVER) and \
            not os.path.exists(PATH_SERVER):
            self.page = None
            return

        try:
            from gi.repository import Gtk as gtk
            builder = gtk.Builder()
            builder.add_from_file(os.path.join(os.environ['UBIQUITY_GLADE'],
                'edu-target.ui'))
            builder.connect_signals(self)
            self.controller.add_builder(builder)
            self.page = builder.get_object('edu-target_window')

            # Load required objects
            self.description = builder.get_object('description')

            self.rb_workstation = builder.get_object('rb_workstation')
            self.lbl_workstation = builder.get_object('lbl_workstation')
            self.eb_workstation = builder.get_object('eb_workstation')
            self.eb_workstation.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_workstation)

            self.bx_termserver = builder.get_object('bx_termserver')
            self.rb_termserver = builder.get_object('rb_termserver')
            self.lbl_termserver = builder.get_object('lbl_termserver')
            self.eb_termserver = builder.get_object('eb_termserver')
            self.eb_termserver.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_termserver)
            self.sp_termserver = builder.get_object('sp_termserver')

            self.bx_server = builder.get_object('bx_server')
            self.rb_server = builder.get_object('rb_server')
            self.lbl_server = builder.get_object('lbl_server')
            self.eb_server = builder.get_object('eb_server')
            self.eb_server.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_server)
            self.sp_server = builder.get_object('sp_server')

            # Hide LTSP option when ltsp.squashfs doesn't exist
            if os.path.exists(PATH_TERMSERVER):
                self.sp_termserver.set_visible(True)
                self.bx_termserver.set_visible(True)

            # Hide server option when server.squashfs doesn't exist
            if os.path.exists(PATH_SERVER):
                self.sp_server.set_visible(True)
                self.bx_server.set_visible(True)

        except Exception as e:
            self.debug('Could not create edu-target page: %s', e)
            self.page = None
        self.plugin_widgets = self.page

    def on_lbl_clicked(self, widget, event, target):
        target.set_active(True)
        target.grab_focus()

    def plugin_translate(self, lang):
        self.description.set_markup(self.controller.get_string(
                            'edu-target_description_label', lang))

        self.lbl_workstation.set_markup(self.controller.get_string(
                                'edu-target_workstation_label', lang))

        self.lbl_termserver.set_markup(self.controller.get_string(
                                'edu-target_termserver_label', lang))

        self.lbl_server.set_markup(self.controller.get_string(
                                'edu-target_server_label', lang))


class Page(Plugin):
    def prepare(self):
        try:
            target = self.db.get('edubuntu-live/edu-target_target')
        except debconf.DebconfError:
            target = "workstation"

        if target == "termserver" and os.path.exists(PATH_TERMSERVER):
            self.ui.rb_termserver.set_active(True)
        elif target == "server" and os.path.exists(PATH_SERVER):
            self.ui.rb_server.set_active(True)
        else:
            self.ui.rb_workstation.set_active(True)

    def ok_handler(self):
        target = "workstation"
        if self.ui.rb_termserver.get_active():
            target = "termserver"
            self.preseed('edubuntu-live/edu-packages_desktop_index', "1")
        elif self.ui.rb_server.get_active():
            target = "server"

        self.preseed('edubuntu-live/edu-target_target', target)
        Plugin.ok_handler(self)
