# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-

# Copyright (C) 2011-2012 Stephane Graber
# Author: Stephane Graber <stgraber@ubuntu.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import apt
import apt_pkg
import os
from gi.repository import Gtk as gtk

from ubiquity.plugin import Plugin, PluginUI
from ubiquity import misc
from ubiquity import install_misc
import debconf

NAME = 'edu-packages'
AFTER = 'edu-domain'
WEIGHT = 10
OEM = False


class PackagesModel(gtk.TreeStore):
    (
        COLUMN_INSTALLED,
        COLUMN_NAME,
        COLUMN_DESCRIPTION
    ) = list(range(3))

    def __init__(self):
        gtk.TreeStore.__init__(self, bool, str, str)
        with misc.raised_privileges():
            self.populate_tree()

    def populate_tree(self):
        src_pkg = "edubuntu-meta"
        meta_blacklist = ['edubuntu-desktop-kde', 'edubuntu-fonts']
        package_blacklist = []
        relation_blacklist = ['Depends']

        # Initialize APT
        apt_pkg.init_config()
        apt_pkg.init_system()

        cache = apt_pkg.Cache(apt.progress.base.OpProgress())
        try:
            srcpkgrecords = apt_pkg.SourceRecords()
            srcpkgrecords.lookup(src_pkg)
        except SystemError:
            # Probably on a system without deb-src entries
            binaries = []
            for p in apt.Cache():
                if p.candidate:
                    if p.candidate.source_name == src_pkg:
                        binaries.append(p.name)

            srcpkgrecords = type("SourceRecords", (object, ),
                                    {'binaries': binaries})()

        pkgrecords = apt_pkg.PackageRecords(cache)

        ubiquity_blacklist = install_misc.query_recorded_removed()[1]
        ubiquity_whitelist = install_misc.query_recorded_installed()

        processed = []
        for pkg in sorted(cache.packages, key=lambda package: package.name):
            if pkg.name in srcpkgrecords.binaries and \
                not pkg.name in meta_blacklist and not pkg.name in processed:

                pkgrecords.lookup(pkg.version_list[0].file_list[0])
                piter = self.append(None, (True,
                            pkg.name, pkgrecords.short_desc))
                for dep_type in pkg.version_list[0].depends_list:
                    if dep_type in relation_blacklist:
                        continue

                    for bin_pkg in sorted(
                                    pkg.version_list[0].depends_list[dep_type],
                                    key=lambda package:
                                    package[0].target_pkg.name):
                        if bin_pkg[0].target_pkg.name in package_blacklist:
                            continue

                        # Don't list inter meta package dependencies
                        # unless it's a meta package we don't list
                        if bin_pkg[0].target_pkg.name in (
                            set(srcpkgrecords.binaries) - set(meta_blacklist)):

                            continue

                        try:
                            pkgrecords.lookup(
                            bin_pkg[0].target_pkg.version_list[0].file_list[0])
                        except IndexError:
                            continue

                        if (bin_pkg[0].target_pkg.current_state ==
                            apt_pkg.CURSTATE_INSTALLED and
                            bin_pkg[0].target_pkg.name not in
                            ubiquity_blacklist) or \
                            bin_pkg[0].target_pkg.name in ubiquity_whitelist:

                            self.append(piter, (True,
                                bin_pkg[0].target_pkg.name,
                                pkgrecords.short_desc))
                        else:
                            self.append(piter, (False,
                                bin_pkg[0].target_pkg.name,
                                pkgrecords.short_desc))
                processed.append(pkg.name)


class PageGtk(PluginUI):
    plugin_title = 'ubiquity/text/edu-packages_heading_label'
    removed_packages = set()
    removed_packages_recursive = set()
    installed_packages = set()
    lang = None

    def __init__(self, controller, *args, **kwargs):
        self.controller = controller

        try:
            builder = gtk.Builder()
            builder.add_from_file(os.path.join(os.environ['UBIQUITY_GLADE'],
                'edu-packages.ui'))
            builder.connect_signals(self)
            self.controller.add_builder(builder)
            self.page = builder.get_object('edu-packages_window')

            # Load required objects
            self.lbl_desktop = builder.get_object('lbl_desktop')
            self.bx_desktopselect = builder.get_object('bx_desktopselect')
            self.lbl_desktopselect = builder.get_object('lbl_desktopselect')
            self.cb_desktopselect = builder.get_object('cb_desktopselect')
            self.lbl_pkgselect = builder.get_object('lbl_pkgselect')
            self.sp_pkgselect = builder.get_object('sp_pkgselect')
            self.sc_pkgselect = builder.get_object('sc_pkgselect')
            self.tv_pkgselect = builder.get_object('tv_pkgselect')

            self.lbl_roleselect = builder.get_object('lbl_roleselect')
            self.lbl_mandatory_roleselect = builder.get_object(
                                                    'lbl_mandatory_roleselect')
            self.sc_mandatory_roleselect = builder.get_object(
                                                    'sc_mandatory_roleselect')
            self.tv_mandatory_roleselect = builder.get_object(
                                                    'tv_mandatory_roleselect')

            self.lbl_optional_roleselect = builder.get_object(
                                                    'lbl_optional_roleselect')
            self.sc_optional_roleselect = builder.get_object(
                                                    'sc_optional_roleselect')
            self.tv_optional_roleselect = builder.get_object(
                                                    'tv_optional_roleselect')

            # Populate drop-down
            self.model_desktop = gtk.ListStore(str, str)
            self.model_desktop.append(['ubuntu', 'Unity'])
            self.model_desktop.append(['gnome-classic', 'Gnome "classic"'])
            self.cb_desktopselect.set_model(self.model_desktop)

            cell = gtk.CellRendererText()
            self.cb_desktopselect.pack_start(cell, False)
            self.cb_desktopselect.add_attribute(cell, 'text', 1)
            self.cb_desktopselect.set_active(0)

            # Configure packages treeview
            packages_model = PackagesModel()
            self.tv_pkgselect.set_model(packages_model)
            self.tv_pkgselect.set_headers_visible(True)

            column = gtk.TreeViewColumn("Installed")
            cell = gtk.CellRendererToggle()
            cell.connect("toggled", self.on_toggled, packages_model)
            column.pack_start(cell, False)
            column.add_attribute(cell, "active",
                                    packages_model.COLUMN_INSTALLED)
            self.tv_pkgselect.append_column(column)

            column = gtk.TreeViewColumn("Name")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text",
                                    packages_model.COLUMN_NAME)
            self.tv_pkgselect.append_column(column)

            column = gtk.TreeViewColumn("Description")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text",
                                    packages_model.COLUMN_DESCRIPTION)
            self.tv_pkgselect.append_column(column)

            # Configure mandatory roles
            self.mandatory_role_model = gtk.ListStore(str, str)
            self.tv_mandatory_roleselect.set_model(self.mandatory_role_model)
            self.tv_mandatory_roleselect.set_headers_visible(True)

            column = gtk.TreeViewColumn("Name")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text", 0)
            self.tv_mandatory_roleselect.append_column(column)

            column = gtk.TreeViewColumn("Description")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text", 1)
            self.tv_mandatory_roleselect.append_column(column)


            # Configure optional roles
            self.optional_role_model = gtk.ListStore(bool, str, str)
            self.tv_optional_roleselect.set_model(self.optional_role_model)
            self.tv_optional_roleselect.set_headers_visible(True)

            column = gtk.TreeViewColumn("Installed")
            cell = gtk.CellRendererToggle()
            cell.connect("toggled", self.on_toggled, packages_model)
            column.pack_start(cell, False)
            column.add_attribute(cell, "active", 0)
            self.tv_optional_roleselect.append_column(column)

            column = gtk.TreeViewColumn("Name")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text", 1)
            self.tv_optional_roleselect.append_column(column)

            column = gtk.TreeViewColumn("Description")
            cell = gtk.CellRendererText()
            column.pack_start(cell, False)
            column.add_attribute(cell, "text", 2)
            self.tv_optional_roleselect.append_column(column)


        except Exception as e:
            self.debug('Could not create edu-packages page: %s', e)
            self.page = None
        self.plugin_widgets = self.page

    def on_toggled(self, toggle, path, model):
        piter = model.get_iter_from_string(path)
        install_state = not model.get_value(piter, model.COLUMN_INSTALLED)
        self.toggle(model, piter, install_state)

    def toggle(self, model, piter, install_state):
        if model.get_value(piter, model.COLUMN_INSTALLED) != install_state:
            model.set_value(piter, model.COLUMN_INSTALLED, install_state)
            self.set_state(model, piter, install_state)

        # Check if we are a meta package
        if model.iter_has_child(piter):
            if install_state:
                # If we select a meta, select all children
                for i in range(model.iter_n_children(piter)):
                    child = model.iter_nth_child(piter, i)
                    if model.get_value(child, model.COLUMN_INSTALLED) != \
                        install_state:

                        self.toggle(model, child, install_state)
            else:
                # If we unselect a meta, unselect all children unless
                # they appear multiple times
                for i in range(model.iter_n_children(piter)):
                    child = model.iter_nth_child(piter, i)
                    if self.get_number_occurrence(piter, model,
                            model.get_value(child, model.COLUMN_NAME)) == 1:
                        if model.get_value(child, model.COLUMN_INSTALLED) != \
                            install_state:

                            self.toggle(model, child, install_state)
        else:
            # Check if all siblings share the same state,
            # if so, apply to parent
            parent = model.iter_parent(piter)
            name = model.get_value(piter, model.COLUMN_NAME)

            score = 0
            for i in range(model.iter_n_children(parent)):
                child = model.iter_nth_child(parent, i)
                if model.get_value(child, model.COLUMN_INSTALLED) == \
                    install_state:

                    score += 1

            if score == model.iter_n_children(parent):
                # All siblings share the same state, apply to parent
                if model.get_value(parent, model.COLUMN_INSTALLED) != \
                    install_state:

                    model.set_value(parent, model.COLUMN_INSTALLED,
                                        install_state)
                    self.set_state(model, parent, install_state)
            else:
                # Some siblings have a different state, unselect parent
                if model.get_value(parent, model.COLUMN_INSTALLED):
                    model.set_value(parent, model.COLUMN_INSTALLED, False)
                    self.set_state(model, parent, False)

            # Get all identical package in other meta and switch state
            root = model.iter_parent(parent)
            for meta_index in range(model.iter_n_children(root)):
                meta = model.iter_nth_child(root, meta_index)
                for package_index in range(model.iter_n_children(meta)):
                    package = model.iter_nth_child(meta, package_index)
                    if model.get_value(package, model.COLUMN_NAME) == name:
                        if model.get_value(package, model.COLUMN_INSTALLED) !=\
                            install_state:

                            self.toggle(model, package, install_state)

    def get_number_occurrence(self, piter, model, name):
        count = 0
        root = model.iter_parent(piter)
        for meta_index in range(model.iter_n_children(root)):
            meta = model.iter_nth_child(root, meta_index)
            for package_index in range(model.iter_n_children(meta)):
                package = model.iter_nth_child(meta, package_index)
                if model.get_value(package, model.COLUMN_NAME) == name:
                    count += 1
        return count

    def set_state(self, model, piter, install_state):
        name = model.get_value(piter, model.COLUMN_NAME)
        if not install_state:
            if name in self.installed_packages:
                self.installed_packages.remove(name)
            else:
                self.removed_packages_recursive.add(name)
        else:
            if name in self.removed_packages_recursive:
                self.removed_packages_recursive.remove(name)
            else:
                self.installed_packages.add(name)

    def plugin_translate(self, lang):
        self.lang = lang
        self.lbl_desktop.set_markup(self.controller.get_string(
                                        'edu-packages_desktop_label',
                                        lang))
        self.lbl_desktopselect.set_markup(self.controller.get_string(
                                        'edu-packages_desktopselect_label',
                                        lang))
        self.lbl_pkgselect.set_markup(self.controller.get_string(
                                        'edu-packages_pkgselect_label',
                                        lang))

        self.tv_pkgselect.get_columns()[0].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_installed_title',
                                    lang))
        self.tv_optional_roleselect.get_columns()[0].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_installed_title',
                                    lang))
        self.tv_pkgselect.get_columns()[1].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_name_title', lang))
        self.tv_mandatory_roleselect.get_columns()[0].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_name_title', lang))
        self.tv_optional_roleselect.get_columns()[1].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_name_title', lang))
        self.tv_pkgselect.get_columns()[2].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_description_title',
                                    lang))
        self.tv_mandatory_roleselect.get_columns()[1].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_description_title',
                                    lang))
        self.tv_optional_roleselect.get_columns()[2].set_title(
                                self.controller.get_string(
                                    'edu-packages_column_description_title',
                                    lang))

        self.lbl_roleselect.set_markup(self.controller.get_string(
                                    'edu-packages_roleselect_label',
                                    lang))
        self.lbl_mandatory_roleselect.set_markup(self.controller.get_string(
                                    'edu-packages_mandatory_roleselect_label',
                                    lang))
        self.lbl_optional_roleselect.set_markup(self.controller.get_string(
                                    'edu-packages_optional_roleselect_label',
                                    lang))


class Page(Plugin):
    def prepare(self):
        try:
            target = self.db.get('edubuntu-live/edu-target_target')
        except debconf.DebconfError:
            target = "workstation"
        try:
            auth = self.db.get('edubuntu-live/edu-domain_select')
        except debconf.DebconfError:
            auth = "server"

        if target == "server":
            self.ui.lbl_desktop.set_visible(False)
            self.ui.bx_desktopselect.set_visible(False)
            self.ui.lbl_pkgselect.set_visible(False)
            self.ui.sp_pkgselect.set_visible(False)
            self.ui.sc_pkgselect.set_visible(False)

            self.ui.lbl_roleselect.set_visible(True)
            if auth == "server":
                self.ui.lbl_mandatory_roleselect.set_visible(True)
                self.ui.sc_mandatory_roleselect.set_visible(True)
            else:
                self.ui.lbl_mandatory_roleselect.set_visible(False)
                self.ui.sc_mandatory_roleselect.set_visible(False)
            self.ui.lbl_optional_roleselect.set_visible(True)
            self.ui.sc_optional_roleselect.set_visible(True)

            roles = [
                        (True, "domain_controler", "domain-controler"),
                        (True, "dhcp_server", "dhcp-server"),
                        (True, "dnsr_server", "dnsr-server"),
                        (True, "reverse_proxy", "reverse-proxy"),
                        (True, "management_interface", "management-interface"),
                        (False, "file_server", "file-server"),
                        (False, "web_server", "web-server"),
                    ]

            self.ui.mandatory_role_model.clear()
            self.ui.optional_role_model.clear()
            for role in roles:
                role_title = self.ui.controller.get_string(
                                "edu-packages_role_%s_title" % role[1],
                                self.ui.lang)
                role_description = self.ui.controller.get_string(
                                "edu-packages_role_%s_description" % role[1],
                                self.ui.lang)
                if role[0] and auth == "server":
                    self.ui.mandatory_role_model.append((role_title,
                                                        role_description))
                else:
                    self.ui.optional_role_model.append((False, role_title,
                                                        role_description))

        else:
            # Restore desktop package selection UI
            self.ui.lbl_desktop.set_visible(True)
            self.ui.bx_desktopselect.set_visible(True)
            self.ui.lbl_pkgselect.set_visible(True)
            self.ui.sp_pkgselect.set_visible(True)
            self.ui.sc_pkgselect.set_visible(True)

            self.ui.lbl_roleselect.set_visible(False)
            self.ui.lbl_mandatory_roleselect.set_visible(False)
            self.ui.sc_mandatory_roleselect.set_visible(False)
            self.ui.lbl_optional_roleselect.set_visible(False)
            self.ui.sc_optional_roleselect.set_visible(False)

            with misc.raised_privileges():
                self.ui.removed_packages = \
                    install_misc.query_recorded_removed()[0]
                self.ui.removed_packages_recursive = \
                    install_misc.query_recorded_removed()[1]
                self.ui.installed_packages = \
                    install_misc.query_recorded_installed()
            try:
                self.ui.cb_desktopselect.set_active(
                        int(self.db.get(
                            'edubuntu-live/edu-packages_desktop_index')))
            except debconf.DebconfError:
                self.ui.cb_desktopselect.set_active(0)

    def ok_handler(self):
        try:
            target = self.db.get('edubuntu-live/edu-target_target')
        except debconf.DebconfError:
            target = "workstation"

        if target == "server":
            # Flush any past package selection
            # (as we'll be using a custom squashfs anyway)
            with misc.raised_privileges():
                if os.path.exists('/var/lib/ubiquity/apt-removed'):
                    os.remove('/var/lib/ubiquity/apt-removed')
                if os.path.exists('/var/lib/ubiquity/apt-installed'):
                    os.remove('/var/lib/ubiquity/apt-installed')
        else:
            with misc.raised_privileges():
                if os.path.exists('/var/lib/ubiquity/apt-removed'):
                    os.remove('/var/lib/ubiquity/apt-removed')
                if os.path.exists('/var/lib/ubiquity/apt-installed'):
                    os.remove('/var/lib/ubiquity/apt-installed')

                try:
                    if self.db.get('oem-config/enable') == 'true':
                        # Ensure that we always have the required OEM packages
                        self.ui.installed_packages.add('oem-config-gtk')
                        self.ui.installed_packages.add(
                                                'oem-config-slideshow-ubuntu')
                except debconf.DebconfError:
                    pass

                try:
                    # Install necessary tools to join the domain
                    if self.db.get('edubuntu-live/edu-domain_select') \
                        == 'member':
                        for pkg in ('libsasl2-modules-gssapi-heimdal',
                                    'libnss-sss', 'libpam-sss',
                                    'libsss-sudo1'):
                            self.ui.installed_packages.add(pkg)
                except debconf.DebconfError:
                    pass

                install_misc.record_removed(self.ui.removed_packages, False)
                install_misc.record_removed(
                                    self.ui.removed_packages_recursive, True)
                install_misc.record_installed(self.ui.installed_packages)
            self.preseed('edubuntu-live/edu-packages_desktop',
                            self.ui.model_desktop[
                                self.ui.cb_desktopselect.get_active()][0])
            self.preseed('edubuntu-live/edu-packages_desktop_index',
                            str(self.ui.cb_desktopselect.get_active()))

        Plugin.ok_handler(self)
