/*============================================================================
 *  Définitions des fonctions de base associées à une liste chaînée de
 *  structures `ecs_champ_t' décrivant un champ
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strcmp() */

#include <bft_file.h>
#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"
#include "ecs_param_perio_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_vec_int.h"
#include "ecs_champ_att.h"
#include "ecs_champ.h"
#include "ecs_vec_def_perio.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_chaine.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_priv.h"


/*============================================================================
 *                       Prototypes de fonctions privées
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction libérant la portion d'une liste chaînée de structures `ecs_champ_t'
 *   à partir d'un noeud dont le pointeur est donné en argument.
 *  Le noeud est à NULL au retour de la fonction
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__detruit
(
 ecs_champ_t * * this_champ_noeud
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (*this_champ_noeud != NULL) {

    ecs_champ_chaine__detruit(&(*this_champ_noeud)->l_champ_sui) ;

    *this_champ_noeud = ecs_champ__detruit(*this_champ_noeud) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant à partir d'un noeud `ecs_champ_t' donné
 *   une liste chaînée de champs
 *   sur le flux décrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__imprime
(
 const ecs_champ_t *const this_champ_noeud ,
       ecs_int_t          imp_col          ,
 const ecs_int_t          nbr_imp          ,
       bft_file_t  *const fic_imp
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


#define ECS_FCT_IMP_CHAMP_NOEUD  "champ"


  if (this_champ_noeud != NULL) {

    ecs_champ_chaine__imprime(this_champ_noeud->l_champ_sui,
                              imp_col,
                              nbr_imp,
                              fic_imp) ;

    ecs_fic__imprime_ptr(fic_imp, imp_col, ECS_FCT_IMP_CHAMP_NOEUD,
                         (const void *)this_champ_noeud) ;

    ecs_champ__imprime(this_champ_noeud,
                       imp_col,
                       nbr_imp,
                       fic_imp) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets
 *   d'une chaîne de structures `ecs_champ_t' dont la tête est donnée
 *----------------------------------------------------------------------------*/

float ecs_champ_chaine__ret_taille
(
 const ecs_champ_t *const this_champ_noeud
)
{

  float  taille       ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  taille       = 0. ;


  if (this_champ_noeud != NULL) {

    taille += ecs_champ_chaine__ret_taille(this_champ_noeud->l_champ_sui) ;

    taille += ecs_champ__ret_taille(this_champ_noeud) ;

  }


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne dans une liste chaînée de champs
 *   dont un noeud est donné la référence au champ suivant.
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__ret_suivant
(
 ecs_champ_t *const this_champ
)
{

  return this_champ->l_champ_sui ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui recherche dans une liste chaînée de champs
 *   dont la tête de la liste est donnée,
 *   le champ correspondant au nom fourni en argument
 *
 *  La fonction retourne :
 *   -    la référence du champ trouvé,
 *   - ou NULL si aucun champ n'a été trouvé
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__trouve_nom
(
       ecs_champ_t *const this_champ_tete , /* --> Tête de liste des champs   */
                                            /*     ou est recherche le champ  */
 const char        *const nom_champ         /* --> Nom à rechercher           */
)
{


  ecs_champ_t *ptr_champ ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (this_champ_tete == NULL || nom_champ == NULL)
    return NULL ;


  ptr_champ = this_champ_tete ;


  while ((ptr_champ != NULL)          &&
         (strcmp(ptr_champ->nom,
                 nom_champ))     != 0    ) {

    ptr_champ = ptr_champ->l_champ_sui ;

  }


  return ptr_champ ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui recherche dans une liste chaînée de champs
 *   dont la tête de la liste est donnée,
 *   le premier champ correspondant au statut fourni en argument
 *
 *  La fonction retourne :
 *   -    la référence du champ trouvé,
 *   - ou NULL si aucun champ n'a été trouvé
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__trouve_statut
(
 ecs_champ_t         *const this_champ_tete , /* --> Tête de liste des champs */
                                              /*     où on recherche le champ */
 ECS_CHAMP_STATUT_E         statut            /* --> Nom à rechercher         */
)
{


  ecs_champ_t *ptr_champ ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ_tete != NULL) ;


  ptr_champ = this_champ_tete ;


  while (   (ptr_champ != NULL)
         && (ptr_champ->statut_e != statut)) {

    ptr_champ = ptr_champ->l_champ_sui ;

  }

  return ptr_champ ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui supprime un noeud (c-a-d la référence au champ donné)
 *   dans une liste chaînée de champs dont la tête est donnée
 *
 *  La tête de la liste n'est pas modifiée
 *   sauf si le noeud à supprimer est aussi la tête de la liste !
 *
 *  Attention! Le champ dont la référence est supprimée de la liste,
 *              n'est pas détruit et
 *              son lien sur un champ suivant est ré-initialisé à `NULL'
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__supprime
(
 ecs_champ_t * *const this_champ_noeud ,  /* --> Adresse du champ à supprimer */
 ecs_champ_t * *const champ_tete          /* --> tête de la liste contenant   */
                                          /*      le noeud à supprimer        */
)
{

  ecs_champ_t *loc_champ_sui  ;
  ecs_champ_t *ptr_champ      ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (*this_champ_noeud == NULL)
    return ;


  assert(*champ_tete != NULL) ;


  if ((*this_champ_noeud) == *champ_tete) {

    /* Le champ noeud à supprimer est la tête de liste des champs */
    /* Le champ suivant est la nouvelle tête                      */

    *champ_tete = (*champ_tete)->l_champ_sui ;

  }
  else {

    /* On recherche le noeud qui précède le noeud à supprimer */

    loc_champ_sui = (*this_champ_noeud)->l_champ_sui ;


    for (ptr_champ = *champ_tete ;
         ptr_champ != NULL && ptr_champ->l_champ_sui != (*this_champ_noeud) ;
         ptr_champ = ptr_champ->l_champ_sui)
      ;


    /* Le noeud à supprimer doit être contenu dans la liste */
    /* dont `*champ_tete' est la tête                       */
    assert(ptr_champ != NULL);

    ptr_champ->l_champ_sui = loc_champ_sui ;


  } /* Fin else : le champ à supprimer n'est pas le champ de tête */


  /* Re-initialisation à `NULL' du lien sur un champ suivant */
  /*  pour le champ déconnecté de la chaîne                  */

  (*this_champ_noeud)->l_champ_sui = NULL ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui ajoute à la fin d'une liste chaînée de champs réceptrice
 *   une liste chaînée de champs à concaténer.
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__ajoute
(
 ecs_champ_t * *const this_champ_tete   ,
 ecs_champ_t   *const champ_concat_tete
)
{

  ecs_champ_t *loc_champ_prec ;
  ecs_champ_t *ptr_champ      ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ_tete != NULL) ;


  if (*this_champ_tete != NULL) {

    /* On va à la fin de la chaîne réceptrice */

    for (ptr_champ = *this_champ_tete       ;
         ptr_champ != NULL                  ;
         ptr_champ = ptr_champ->l_champ_sui  )
      loc_champ_prec = ptr_champ ;


    /* On ajoute le lien avec le début de la chaîne à concaténer */

    loc_champ_prec->l_champ_sui = champ_concat_tete ;


  }
  else {


    *this_champ_tete = champ_concat_tete ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui concatène les champs identiques
 *   de deux listes chaînées de champs
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__concatene
(
 ecs_champ_t * * this_champ_tete    ,  /* <-> Tête de la chaîne réceptrice    */
 ecs_champ_t * * concat_champ_tete  ,  /* <-> Tête de la chaîne contenant     */
                                       /*         les champs à concaténer     */
 size_t          nbr_elt_init_recep ,  /* --> Nbr d'elts de l'entité recept.  */
 size_t          nbr_elt_ent_concat    /* --> Nbr d'elts ent. à concaténer.   */
)
{

  ecs_champ_t *this_champ_ptr    ;
  ecs_champ_t *concat_champ_ptr  ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ_tete != NULL) ;
  assert(concat_champ_tete != NULL) ;


  /*---------------------------------------------------------*/
  /* Boucle sur les champs de la chaîne de champs réceptrice */
  /*---------------------------------------------------------*/


  for (this_champ_ptr  = *this_champ_tete ;
       this_champ_ptr != NULL  ;
       this_champ_ptr  = this_champ_ptr->l_champ_sui) {


    /* Le champ existe-t-il dans l'entité de maillage à concaténer ? */
    /*---------------------------------------------------------------*/

    if (concat_champ_tete != NULL)
      concat_champ_ptr = *concat_champ_tete ;
    else
      concat_champ_ptr = NULL ;


    while ((concat_champ_ptr != NULL)           &&
           (strcmp(this_champ_ptr->nom,
                   concat_champ_ptr->nom)) != 0    ) {

      concat_champ_ptr = concat_champ_ptr->l_champ_sui ;

    }


    /* ------------------------------------------------------------------ */
    /* PARTIE REALISEE PENDANT LA FUSION                                  */
    /* ------------------------------------------------------------------ */
    /*    Si le champ n'existe pas dans l'entité de maillage à concaténer */
    /* et si c'est un champ de type "attribut"                            */
    /* (pour lequel tous les éléments doivent être renseignes)            */
    /* on initialise les valeurs de l'attribut pour le champ à concaténer */
    /* ------------------------------------------------------------------ */

    if (concat_champ_ptr != NULL) {

      /* Le champ existe dans l'entité de maillage à concaténer */

      if ((*concat_champ_tete)->statut_e == ECS_CHAMP_STATUT_REF_ELT)

        /* Pour certains champs (de type filiation, ou l'on a une
           'auto-référence'), il faut d'abord décaler les références
           aux éléments du champ à concaténer */

        ecs_champ__incremente_val(*concat_champ_tete,
                                  nbr_elt_init_recep) ;

      /* On le concatène */

      ecs_champ__concatene(this_champ_ptr,
                           concat_champ_ptr) ;

      /* On le détruit de la liste des champs */

      ecs_champ_chaine__supprime(&concat_champ_ptr,
                                 concat_champ_tete) ;

      concat_champ_ptr = ecs_champ__detruit(concat_champ_ptr) ;

    }

    else {

      /* Le champ n'existe pas dans l'entité de maillage à concaténer */

      ecs_champ__prolonge(this_champ_ptr,
                          0,
                          nbr_elt_ent_concat) ;

    }

  } /* Fin : boucle sur les champs de l'entité de maillage receptrice */


  /*----------------------------------------------------------------------*/
  /* La liste chaînée des champs à concaténer RESTANTS                    */
  /*  (qui n'étaient pas initialement présents dans la chaîne réceptrice) */
  /*  est rajoute à la liste chaînée des champs réceptrice                */
  /*  (si elle n'est pas vide !)                                           */
  /*----------------------------------------------------------------------*/

  if (concat_champ_tete != NULL) {

    while (*concat_champ_tete != NULL) {

      concat_champ_ptr = (*concat_champ_tete) ;

      if (concat_champ_ptr->statut_e == ECS_CHAMP_STATUT_REF_ELT)

        /* Pour certains champs (de type filiation, ou l'on a une
           'auto-référence'), il faut d'abord décaler les références
           aux éléments du champ à concaténer */

        ecs_champ__incremente_val(concat_champ_ptr,
                                  nbr_elt_init_recep) ;

      /* Le champ n'existe pas dans l'entité de maillage réceptrice */

      ecs_champ__prolonge(concat_champ_ptr,
                          nbr_elt_init_recep,
                          0) ;


      /* On le détruit de la liste des champs du maillage à concaténer */

      ecs_champ_chaine__supprime(&concat_champ_ptr,
                                 concat_champ_tete) ;

      /* On le transfère à l'entité de maillage réceptrice */

      ecs_champ_chaine__ajoute(this_champ_tete,
                               concat_champ_ptr) ;


    } /* Fin : boucle sur les champs de l'entité de maillage à concaténer */

  }

}


/*----------------------------------------------------------------------------
 *  Fonction appliquant un vecteur de transformation
 *   à une liste chaînée de champs donnée
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__transforme
(
       ecs_champ_t         * this_champ_chaine_tete ,
       size_t                nbr_elt_new            ,
 const ecs_tab_int_t         vect_transf            ,
 ECS_PTR_FCT_TRANSF_CHAMP    fct_transformation
)
{

  ecs_champ_t *ptr_champ ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/



  for (ptr_champ = this_champ_chaine_tete ;
       ptr_champ != NULL ;
       ptr_champ = ptr_champ->l_champ_sui) {


    (* fct_transformation)(ptr_champ  ,
                           nbr_elt_new,
                           vect_transf ) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les champs de la chaîne de champs des sous-éléments
 *                à partir des champs de la chaîne de champs des éléments
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__herite
(
 ecs_champ_t *champ_elt_tete,
 ecs_champ_t *champ_elt_def_sselt,
 ecs_champ_t *champ_def_sselt
)
{

  ecs_champ_t  *ptr_champ_elt ;
  ecs_champ_t  *champ_sselt   ;
  ecs_champ_t  *this_champ_sselt_tete ;

  const ecs_int_t nbr_sselt = ecs_champ__ret_pos_nbr(champ_def_sselt) - 1 ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  this_champ_sselt_tete = NULL ;


  for (ptr_champ_elt = champ_elt_tete ;
       ptr_champ_elt != NULL ;
       ptr_champ_elt = ptr_champ_elt->l_champ_sui) {


    if (ptr_champ_elt->statut_e == ECS_CHAMP_STATUT_HERITABLE) {


      champ_sselt = ecs_champ_att__herite(ptr_champ_elt,
                                          champ_elt_def_sselt,
                                          nbr_sselt) ;


      if (champ_sselt != NULL) {


        /* Ajout du nouveau champ dans la liste chaînée */
        /*  des champs des sous-éléments                */
        /*----------------------------------------------*/

        ecs_champ_chaine__ajoute(&this_champ_sselt_tete,
                                 champ_sselt) ;
      }


    }


  }  /* Fin de la boucle sur les champs des éléments */


  return this_champ_sselt_tete ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renumérote les labels et les références
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__renum
(
 ecs_champ_t    *champ_elt_tete,
 ecs_champ_t    *champ_elt_old_new,
 size_t          nbr_elt_new
)
{

  ecs_champ_t  *ptr_champ_elt ;
  ecs_champ_t  *champ_elt_aux ;
  ecs_champ_t  *this_champ_elt_tete ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  this_champ_elt_tete = NULL ;


  for (ptr_champ_elt = champ_elt_tete ;
       ptr_champ_elt != NULL ;
       ptr_champ_elt = ptr_champ_elt->l_champ_sui) {


    if (ptr_champ_elt->statut_e == ECS_CHAMP_STATUT_REF_ELT) {


      champ_elt_aux = ecs_champ__copie(ptr_champ_elt) ;

      ecs_champ_att__renum_perio(champ_elt_aux,
                                 champ_elt_old_new,
                                 nbr_elt_new) ;


      /* Ajout du nouveau champ dans la liste chaînée */
      /* des champs des sous-éléments                 */
      /*----------------------------------------------*/

      ecs_champ_chaine__ajoute(&this_champ_elt_tete,
                               champ_elt_aux) ;


    }


  } /* Fin de la boucle sur les champs des éléments */


  return this_champ_elt_tete ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de périodicités.
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_champ_chaine__nbr_perio
(
 ecs_champ_t  * champ_att_tete
)
{

  ecs_int_t     nbr_per ;

  ecs_champ_t  *ptr_champ_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  nbr_per = 0 ;


  for (ptr_champ_att = champ_att_tete ;
       ptr_champ_att != NULL ;
       ptr_champ_att = ptr_champ_att->l_champ_sui) {


    if (   strncmp(ptr_champ_att->nom, ECS_CHAMP_NOM_FAC_PERIO,
                   strlen(ECS_CHAMP_NOM_FAC_PERIO))== 0
        && ptr_champ_att->statut_e == ECS_CHAMP_STATUT_REF_ELT) {

      nbr_per++ ;

    }


  }


  return nbr_per ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui met à jour le tableau renvoyé par
 * ecs_champ_def__typ_fac_cel() en fonction des connectivités ajoutées par
 * les périodicités éventuelles.
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__typ_fac_perio
(
 ecs_champ_t    * champ_att_tete,
 ecs_tab_int_t  * typ_fac
)
{

  ecs_int_t     nbr_per ;

  ecs_champ_t  *ptr_champ_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  for (ptr_champ_att = champ_att_tete ;
       ptr_champ_att != NULL ;
       ptr_champ_att = ptr_champ_att->l_champ_sui) {


    if (   strncmp(ptr_champ_att->nom, ECS_CHAMP_NOM_FAC_PERIO,
                   strlen(ECS_CHAMP_NOM_FAC_PERIO))== 0
        && ptr_champ_att->statut_e == ECS_CHAMP_STATUT_REF_ELT) {

      ecs_champ_att__typ_fac_perio(ptr_champ_att,
                                   typ_fac) ;

      nbr_per++ ;

    }


  } /* Fin de la boucle sur les champs attributs des faces */

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de faces périodiques associées à chaque
 * périodicité.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ_chaine__nbr_fac_perio
(
 ecs_champ_t  * champ_att_tete
)
{

  size_t         ind_per ;

  ecs_tab_int_t  nbr_fac_per ;

  ecs_champ_t  *ptr_champ_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  nbr_fac_per.nbr = ecs_champ_chaine__nbr_perio(champ_att_tete) ;

  BFT_MALLOC(nbr_fac_per.val, nbr_fac_per.nbr, ecs_int_t) ;

  for (ind_per = 0 ; ind_per < nbr_fac_per.nbr ; ind_per++)
    nbr_fac_per.val[ind_per] = 0 ;

  ind_per =  0 ;


  for (ptr_champ_att = champ_att_tete ;
       ptr_champ_att != NULL ;
       ptr_champ_att = ptr_champ_att->l_champ_sui) {

    if (   strncmp(ptr_champ_att->nom, ECS_CHAMP_NOM_FAC_PERIO,
                   strlen(ECS_CHAMP_NOM_FAC_PERIO))== 0
        && ptr_champ_att->statut_e == ECS_CHAMP_STATUT_REF_ELT) {

      nbr_fac_per.val[ind_per] =  (2 * ecs_champ__ret_val_nbr(ptr_champ_att)) ;

      ind_per++ ;

    }

  }

  return nbr_fac_per ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui construit la des faces périodiques associées à la
 *  périodicité num_per de la manière suivante :
 *  =>  [i,j; ...] avec i->j
 *  On ne compte pas les faces se voyant elles-mêmes.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ_chaine__liste_fac_perio
(
 ecs_champ_t  * champ_att_tete,
 ecs_int_t      num_per
)
{

  ecs_int_t        ind_per ;

  ecs_tab_int_t    liste_fac_perio ;

  ecs_champ_t    * ptr_champ_att ;
  ecs_vec_int_t  * vec_fac_perio ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Récupération des vecteurs de correspondances périodiques */

  ind_per =  0 ;

  liste_fac_perio.nbr = 0 ;
  liste_fac_perio.val = NULL ;

  for (ptr_champ_att = champ_att_tete ;
       ptr_champ_att != NULL ;
       ptr_champ_att = ptr_champ_att->l_champ_sui) {

    if (   strncmp(ptr_champ_att->nom, ECS_CHAMP_NOM_FAC_PERIO,
                   strlen(ECS_CHAMP_NOM_FAC_PERIO))== 0
        && ptr_champ_att->statut_e == ECS_CHAMP_STATUT_REF_ELT) {

      if (ind_per == num_per - 1) {

        vec_fac_perio = ecs_champ__initialise_vec_int(ptr_champ_att) ;

        liste_fac_perio = ecs_vec_def__liste_fac_perio(vec_fac_perio) ;

        ecs_champ__libere_vec_int(ptr_champ_att,
                                  vec_fac_perio) ;

        break ;
      }
      else
        ind_per++ ;

    }

  }

  return liste_fac_perio ;

}


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

