/* Copyright (C) 2008 by Intevation GmbH
 * Authors:
 * Bernhard Herzog <bh@intevation.de>
 *
 * This program is free software under the LGPL (>=v2.1)
 * Read the file COPYING coming with the software for details.
 */

#include "lib.h"
#include "str.h"
#include "common.h"
#include "imap-quote.h"

#include <string.h>

#include "metadata-plugin.h"
#include "imap-annotatemore-plugin.h"

enum attribute_properties {
	ATTR_INVALID = 0x0001,
	ATTR_PRIVATE = 0x0002,
	ATTR_PUBLIC  = 0x0004,
	ATTR_BOTH    = ATTR_PUBLIC | ATTR_PRIVATE,
};

static bool validate_entry_name(struct client_command_context *cmd,
				const char *entry)
{
	if (entry == NULL) {
		client_send_tagline(cmd, "BAD Missing entry name.");
		return FALSE;
	}

	if (entry[0] != '/') {
		client_send_tagline(cmd,
				    "BAD entry name must start with slash.");
		return FALSE;
	}

	return TRUE;
}

static enum attribute_properties
validate_attribute_name(struct client_command_context *cmd,
			const char *attribute)
{
	if (attribute == NULL) {
		client_send_tagline(cmd,
				    "BAD Missing or NIL attribute name.");
		return ATTR_INVALID;
	}

	if (strcmp(attribute, "value.shared") == 0) {
		return ATTR_PUBLIC;
	} else if (strcmp(attribute, "value.priv") == 0) {
		return ATTR_PRIVATE;
	} else if (strcmp(attribute, "value") == 0) {
		return ATTR_BOTH;
	} else {
		client_send_tagline(cmd,
				    "BAD only 'value.shared' and"
				    " 'value.priv' attributes are"
				    " supported '.'");
		return ATTR_INVALID;
	}
}


static void send_annotation_line(struct client_command_context *cmd,
				 const char *mailbox,
				 const char *entry,
				 const char *value,
				 bool private)
{
	if (value != NULL) {
		string_t *str = t_str_new(128);
		str_append(str, "* ANNOTATION ");
		imap_quote_append_string(str, mailbox, FALSE);
		str_append(str, " ");
		imap_quote_append_string(str, entry, FALSE);
		str_printfa(str, " (\"value.%s\" ",
			    private ? "priv" : "shared");
		imap_quote_append_string(str, value, FALSE);
		str_append(str, ")");
		client_send_line(cmd->client, str_c(str));
	}
}


static bool get_and_send_annotation(struct client_command_context *cmd,
				    const char *mailbox,
				    const char *entry,
				    enum attribute_properties scope)
{
	const char *value;
	bool success = TRUE;

	if ((scope & ATTR_PUBLIC) != 0) {
		value = NULL;
		success = metadata_get_metadata_entry(cmd, mailbox, entry,
						       &value, FALSE);
		send_annotation_line(cmd, mailbox, entry, value, FALSE);
	}

	if (!success) {
		return FALSE;
	}

	if ((scope & ATTR_PRIVATE) != 0) {
		value = NULL;
		success = metadata_get_metadata_entry(cmd, mailbox, entry,
						      &value, TRUE);
		send_annotation_line(cmd, mailbox, entry, value, TRUE);
	}

	return success;
}


static bool extract_single_value(struct client_command_context *cmd,
				 const struct imap_arg *attribute,
				 const char **value_r)
{
	const struct imap_arg *attrlist;

	if (IMAP_ARG_LIST_COUNT(attribute) == 1) {
		attrlist = IMAP_ARG_LIST_ARGS(attribute);
		*value_r = IMAP_ARG_STR(&attrlist[0]);
		return TRUE;
	} else {
		client_send_tagline(cmd,
				    "NO Lists of entries not yet implemented.");
		return FALSE;
	}
}


static bool cmd_getannotation(struct client_command_context *cmd)
{
	const struct imap_arg *args;
	const char *mailbox;
	const char *entry;
	const char *attribute;
	enum attribute_properties attribute_properties;

	if (!client_read_args(cmd, 3, 0, &args))
		return FALSE;

	mailbox = IMAP_ARG_STR(&args[0]);
	if (mailbox == NULL) {
		client_send_tagline(cmd,
				    "BAD Missing mailbox name.");
		return TRUE;
	}

	if (*mailbox == '\0') {
		client_send_tagline(cmd,
				    "NO Server annotations not yet"
				    " implemented.");
		return TRUE;
	}

	if (args[1].type == IMAP_ARG_LIST) {
		if (!extract_single_value(cmd, &args[1], &entry))
			return TRUE;
	} else
		entry = IMAP_ARG_STR(&args[1]);

	if (!validate_entry_name(cmd, entry))
		return TRUE;

	if (args[2].type == IMAP_ARG_LIST) {
		if (!extract_single_value(cmd, &args[2], &attribute))
			return TRUE;
	} else
		attribute = IMAP_ARG_STR(&args[2]);

	attribute_properties = validate_attribute_name(cmd, attribute);
	if (attribute_properties & ATTR_INVALID)
		return TRUE;

	if (get_and_send_annotation(cmd, mailbox, entry, attribute_properties))
		client_send_tagline(cmd, "OK Completed.");

	return TRUE;
}


static bool pair_extract_value(struct client_command_context *cmd,
			       const struct imap_arg *attributes,
			       const char **value_r,
			       bool *private_r)
{
	const struct imap_arg *pairs;
	unsigned int count;

	if (attributes->type != IMAP_ARG_LIST) {
		client_send_tagline(cmd,
				    "BAD attributes parameter must be a list"
				    " of attribute value pairs.");
		return FALSE;
	}

	count = IMAP_ARG_LIST_COUNT(attributes);
	pairs = IMAP_ARG_LIST_ARGS(attributes);

	if (count % 2 != 0) {
		client_send_tagline(cmd,
				    "BAD list of attribute value pairs"
				    " must have an even number of elements");
		return FALSE;
	}

	if (count == 0) {
		client_send_tagline(cmd,
				    "BAD list of attribute value pairs"
				    " is empty");
		return FALSE;
	}

	if (count == 2) {
		enum attribute_properties properties;
		properties = validate_attribute_name(cmd,
						     IMAP_ARG_STR(&pairs[0]));
		if ((properties & ATTR_INVALID) != 0)
			return FALSE;
		if ((properties & ATTR_BOTH) == ATTR_BOTH) {
			client_send_tagline(cmd,
					    "BAD attribute must end in .priv"
					    " or .shared for SETANNOTATION");
			return FALSE;
		}

		*value_r = IMAP_ARG_STR(&pairs[1]);
		*private_r = ((properties & ATTR_PRIVATE) != 0);
		return TRUE;
	}

	client_send_tagline(cmd, "NO only setting single attributes supported");
	return FALSE;
}


static bool cmd_setannotation(struct client_command_context *cmd)
{
	const struct imap_arg *args;
	const char *mailbox;
	const char *entry;
	const char *value;
	bool private;
	bool success;

	if (!client_read_args(cmd, 3, 0, &args))
		return FALSE;

	mailbox = IMAP_ARG_STR(&args[0]);
	if (mailbox == NULL) {
		client_send_tagline(cmd,
				    "BAD Missing mailbox name.");
		return TRUE;
	}
	if (*mailbox == '\0') {
		client_send_tagline(cmd,
				    "NO Server annotations not yet"
				    " implemented.");
		return TRUE;
	}

	if (args[1].type == IMAP_ARG_LIST) {
		client_send_tagline(cmd,
				    "NO Lists of entries not yet implemented.");
		return TRUE;
	}
	entry = IMAP_ARG_STR(&args[1]);
	if (!validate_entry_name(cmd, entry))
		return TRUE;

	if (!pair_extract_value(cmd, &args[2], &value, &private))
		return TRUE;

	if (private && !metadata_private_allowed()) {
		client_send_tagline(cmd,
				    "NO private annotations not supported.");
		return TRUE;
	}

	success = metadata_set_metadata_entry(cmd, mailbox, entry, value,
					      private);
	if (success) {
		client_send_tagline(cmd, "OK Completed.");
	}

	return TRUE;
}


void imap_annotatemore_plugin_init(void)
{
	command_register("GETANNOTATION", cmd_getannotation, 0);
	command_register("SETANNOTATION", cmd_setannotation, 0);
	str_append(capability_string, " ANNOTATEMORE");
}

void imap_annotatemore_plugin_deinit(void)
{
	command_unregister("SETANNOTATION");
	command_unregister("GETANNOTATION");
}
