/*
 *  Remote Network application system for DNAS -- GTK+ interface
 *  Copyright (C) 2002 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: rnas-monitor-gtk.c,v 1.12 2002/12/01 13:27:24 dancer Exp $
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <locale.h>
#include <dlfcn.h>
#include "config.h"
#include "dmachinemon/dmachinemon-debug.h"
#include "dmachinemon/dmachinemon.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
#include "dmachinemon/dmachinemon-commandlineparser.h"
#include "rnas-monitor-gtk.h"

/** GTK+ lock */
pthread_mutex_t gtk_lock = PTHREAD_MUTEX_INITIALIZER;

/** timeout routine lock */
pthread_mutex_t timeout_lock = PTHREAD_MUTEX_INITIALIZER;

dm_commandoption cdat;

/** the vbox, main displaying window. */
static GtkWidget * vbox = NULL;	

/**
 * The callback function used in creating application data.
 *
 * @return the pointer to application data
 */
void *
callback_createapplicationdata(void * data, const char * stringdata)
{
  rnas_applicationdata* m = NULL;
  rnas_applicationdata* start = NULL;
  rnasmonitors * r = data;
  GtkWidget * horz = gtk_hbox_new(TRUE, FALSE);

  gtk_box_pack_start (GTK_BOX(vbox), horz, FALSE, FALSE, FALSE);
  CreateApplicationDataTitle(horz, stringdata); /* add hostname */

  for (; r; r=r->next)
    {
      if (m)
	m = m->next = malloc (sizeof(rnas_applicationdata));
      else			/* m==NULL */
	start = m = malloc (sizeof(rnas_applicationdata));
      m->r = r;
      if (!(m->w = r->create_window_new(stringdata, horz, r, m)))
	{
	  fprintf(stderr, "Error: fatal. w=NULL");
	  exit(1);
	}
    }
  m->next = NULL;
  gtk_widget_show(horz);
  return start;
}

static void
access_master_with_callback(dm_commandoption * cdat, const char * hostname, const char* portnumber, rnasmonitors * r)
{
  int fd = DNAS_connect_to_host(cdat, hostname, atoi(portnumber), 1);
  FILE* f=(fd!=-1)?fdopen (fd, "r"):NULL;
  
  if (!f)
    {
      fprintf (stderr, PROGRAMNAME ": cannot open server socket\n");
      exit(1);
    }

  if (dm_process_nodes_input (f, &maininfo, callback_createapplicationdata, r))
    {
      fprintf (stderr, PROGRAMNAME ": could not handle input...\n");
      fclose(f);
      exit (1);
    }  

  fclose(f);
}

static void
rnas_dump_printhelp(void)
{
  fprintf(stderr, 
	  "RNAS-GTK+ application, Copyright (c)2002 Junichi Uekawa.\n\n"
	  "\trnas-monitor-gtk [DNAS options] -- \n"
	  "\t  [(module-name title tag other-options*)]*\n"
	  "Please specify dump parameter\n\n\n"
	  );
  dmachinemon_print_commandline_help();      
}

/**
 * Database iteration routine, scanning each to execute command on them.
 *
 * Calls displayfunction for individual items.
 *
 * @return nothing
 */
static void
iterate_database(dm_machinelist * a, rnasmonitors *r )
{
  rnas_applicationdata *app;
  
  if (!a)
    {
      fprintf (stderr, "zero info found\n");
      return ;      
    }
  
  for ( ; a; a=a->next)  
    {
      for (app = a->applicationdata; app ; app = app -> next)
	{
	  if (!app->r->displayfunction)
	    {
	      fprintf (stderr, "displayfunction func not found\n");
	      return ;      
	    }
	  app->r->displayfunction (app, a);
	}
    }
}

/**
 *  Routine to do the timeout thing.
 * This function is static, so not in the public namespace.
 * Calls database iteration routines, and does other things.
 *
 * @return 1 to continue.
 */
static gint
timeout_command (gpointer data)
{
  rnasmonitors * r = data;
  
  DEBUGPRINT("timeout one");
  pthread_mutex_lock(&timeout_lock);
  pthread_mutex_lock(&gtk_lock);
  DEBUGPRINT("timeout locked");
  DEBUGPRINT("master access");

  access_master_with_callback (&cdat, cdat.parenthostname, cdat.port_client, r);
  DEBUGPRINT("iterate db");
  iterate_database(maininfo.machinedb, r);

  pthread_mutex_unlock(&gtk_lock);
  pthread_mutex_unlock(&timeout_lock);

  return 1;			/* continue the clock. */
}

/**
 * place to do the main looping.
 */
static void
main_loop(dm_commandoption * cdat, rnasmonitors*r)
{
  int hTimeout;

  DEBUGPRINT("start of loop");
  
  hTimeout = gtk_timeout_add(1000 * cdat->sleeptime, timeout_command, r);
  timeout_command(r);
  gtk_main();  
  gtk_timeout_remove(hTimeout);
}

int
main(int ac, char ** av)
{
  rnasmonitors * r;
  
  dmachinemon_parse_options(ac, av, &cdat);
  if (!cdat.parenthostname)
    {
      rnas_dump_printhelp();
      exit(1);
    }
  
  if (!av[optind])
    {
      rnas_dump_printhelp();
      exit(1);
    }
  
  if (!(r = load_plugins(&av[optind])))
    {
      fprintf(stderr, "load_plugins failed\n");
      exit (1);
    }

  gtk_init(&ac, &av);
  DEBUGPRINT("CREATEMAIN");
  
  vbox = CreateMainWindow(av[0], r);

  DEBUGPRINT("mloop");
  main_loop(&cdat, r);
  exit(0);
}

/*
What follows is a m4 script!
MANUAL([.TH "rnas-monitor-gtk" 1 "2002 Sep 27" "DNAS" "RNAS"
.SH NAME
rnas-monitor-gtk \- RNAS GTK+ interface
.SH SYNOPSIS
.BI "rnas-monitor-gtk [" "DNAS-options" "] -- [(" "display-module title tag other-options)" ")]*"
.SH DESCRIPTION
GTK+ frontend for RNAS monitoring system for DNAS.
.SH "DNAS OPTIONS"
] m4_include([dmachinemon-commandlineparser.inc]) COMMAND [
.SH "EXAMPLES"
.B "rnas-monitor-gtk -pcambria -c3007 -- \"(singlepoint Load-average load1 0 0 10 10)\""

.SH "AUTHOR"
Junichi Uekawa <dancer@mikilab.doshisha.ac.jp> <dancer@debian.org> <dancer@netfort.gr.jp>

This manual page is autogenerated from rnas-monitor-gtk.c and 
dmachinemon-commandlineparser.c
.SH "SEE ALSO"
.RI "rnas-dump(" 1 "), "
.RI "rnas-dump-xml(" 1 "), "
.RI "dmachinemon-master(" 1 ")"])
*/
