--  dh_ada_library, helper for Ada libraries Debian maintainers
--
--  Copyright (C) 2012 Nicolas Boulenguez <nicolas.boulenguez@free.fr>
--
--  This program is free software: you can redistribute it and/or
--  modify it under the terms of the GNU General Public License as
--  published by the Free Software Foundation, either version 3 of the
--  License, or (at your option) any later version.
--  This program is distributed in the hope that it will be useful, but
--  WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
--  General Public License for more details.
--  You should have received a copy of the GNU General Public License
--  along with this program. If not, see <http://www.gnu.org/licenses/>.
--

private with Ada.Finalization;
with Ada.Containers.Indefinite_Vectors;
private with Prj;

package Projects is

   --  Encapsulate all GNAT project stuff handling.

   pragma Elaborate_Body;

   type Project_Record is limited private;

   type Assignment
     (Name_Length  : Natural;
      Value_Length : Natural) is record
      Name         : String (1 .. Name_Length);
      Value        : String (1 .. Value_Length);
   end record;

   pragma Warnings (Off,
                    "pragma Warnings Off for ""*"" unused, could be omitted");
   package Assignment_Vectors is
      new Ada.Containers.Indefinite_Vectors (Positive, Assignment);
   pragma Warnings (On,
                    "pragma Warnings Off for ""*"" unused, could be omitted");

   procedure Parse
     (Project     :    out Project_Record;
      Assignments : in     Assignment_Vectors.Vector;
      File_Name   : in     String);
   Parse_Error : exception;

   function Library_Name (Project : in Project_Record) return String;
   pragma Inline (Library_Name);

   function Dynamic (Project : in Project_Record) return Boolean;
   pragma Inline (Dynamic);

   function Is_Library (Project : in Project_Record) return Boolean;
   pragma Inline (Is_Library);

   function Externally_Built (Project : in Project_Record) return Boolean;
   pragma Inline (Externally_Built);

   function Library_Dir (Project : in Project_Record) return String;
   pragma Inline (Library_Dir);
   --  Absolute path ending with a slash.

   procedure Iterate_On_Imported_Projects
     (Project : in Project_Record;
      Process : not null access procedure
        (Imported_Full_File_Name : in String;
         Imported_Library_Name   : in String));
   --  Imported_Library_Name is "" if the imported project is not a
   --  library project.

   procedure Iterate_On_Ali_Files
     (Project : in Project_Record;
      Process : not null access procedure (Path : in String));
   No_Ada_Sources : exception;

   procedure Iterate_On_Library_Options
     (Project : in Project_Record;
      Process : not null access procedure (Option : in String));
   --  Leading_Library_options are processed first, if any.

   procedure Iterate_On_Sources
     (Project : in Project_Record;
      Process : not null access procedure (Path : in String));

private

   type Project_Record is new Ada.Finalization.Limited_Controlled with record
      Project_Id       : Prj.Project_Id;
      Project_Tree_Ref : Prj.Project_Tree_Ref;
   end record;

   pragma Warnings (Off, "declaration of ""*"" hides one at line *");
   overriding procedure Initialize (Object : in out Project_Record);
   overriding procedure Finalize (Object : in out Project_Record);
   pragma Warnings (On, "declaration of ""*"" hides one at line *");

end Projects;
