/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtCore/QDebug>
#include <QtCore/QCoreApplication>

#include "policyengine.h"
#include "componentmanager.h"
#include "protocolmanager.h"
#include "channelhandler.h"

#include <Decibel/Types>
#include <Decibel/DBusNames>

#include <QtTapioca/Connection>
#include <QtTapioca/Channel>
#include <QtTapioca/Handle>

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private data class for the PolicyEngine.
 * @author Tobias Hunger <info@basyskom.de>
 */
class PolicyEnginePrivate
{
public:
    /** @brief Constructor. */
    PolicyEnginePrivate(const ComponentManager * component_mgr) :
        m_componentMgr(component_mgr)
    {
        Q_ASSERT(component_mgr != 0);
    }

    /** @brief Destructor. */
    ~PolicyEnginePrivate()
    { }

    /**
     * @brief Select a ChannelHandler and activate it.
     * @param connection A pointer to the Connection of the new Channel.
     * @param channel A pointer to the new Channel.
     *
     * Select a ChannelHandler and activate it.
     */
    void activateHandler(const QtTapioca::Connection * connection,
                         const QtTapioca::Channel * channel,
                         const bool incoming) const
    {
        Q_ASSERT(connection != 0);
        Q_ASSERT(channel != 0);

        qDebug() << "PE: Searching for a handler...";

        Decibel::ChannelInfo channel_info(connection, channel, incoming);

        QList<uint> components = findComponents(connection->protocol(),
                                                channel->type(),
                                                channel->target()->handle()->type(),
                                                /* FIXME: Use proper list of IDs (from groupinterface!) */
                                                0);

        if (components.isEmpty())
        {
            qWarning() << "Don't know what to do!";
            return;
        }

        uint component;
        foreach (component, components)
        {
            Decibel::ComponentInfo handler(m_componentMgr->component(component));
            if (runHandler(handler.service_name,
                           handler.object_path.path(),
                           channel_info))
            { return; }
        }

        qWarning() << "All handlers rejected the channel.";
    }

private:
    /**
     * @brief Activate a ChannelHandler.
     * @param service The service name of the ChannelHandler to activate.
     * @param path The object path of the ChannelHandler to activate.
     * @param channel_info The channel to handle.
     * @param incoming true if this is a incoming channel.
     * @return true if the ChannelHandler was activated successfully and
     * agreed to handle the Channel and false otherwise.
     *
     * Activate a ChannelHandler and check whether it agrees to handle the
     * Channel for us.
     */
    bool runHandler(const QString service, const QString path,
                    const Decibel::ChannelInfo & channel_info) const
    {
        qDebug() << "PE: Running handler" << service << "(" << path << ")";
        de::basyskom::Decibel::ChannelHandler
        channel_handler(service, path, QDBusConnection::sessionBus(),
                        QCoreApplication::instance());
        QDBusReply<bool> reply =
                channel_handler.handleChannel(channel_info);

        if (!reply.isValid())
        {
            QDBusError error = reply.error();
            qWarning() << "PE: Failed to activate Handler:" << service << path;
            qWarning() << "    DBus Error:" << error.type() << ":" << error.message();
            return false;
        }

        qWarning() << "PE: Handler" << service << path
                   << "replied with:" << reply.value();

        return reply.value();
    }

    QList<uint> findComponents(const QString & protocol,
                               const uint type,
                               const uint target_type,
                               const uint target_id) const
    {
        Q_UNUSED(target_id)

        QString profile_name = m_componentMgr->activeProfile();
        QList<Decibel::Component> components =
            m_componentMgr->profile(profile_name);

        QList<uint> results;

        qDebug() << "PE: Searching for components in profile"
                 << profile_name << "...";
        Decibel::Component component;
        foreach (component, components)
        {
            Decibel::ComponentInfo info =
                m_componentMgr->component(component.component_handle);

            // Filter out components:
            if (!component.protocol_list.isEmpty() &&
                !component.protocol_list.contains(protocol))
            { continue; }
            if (!info.possible_protocol_list.isEmpty() &&
                !info.possible_protocol_list.contains(protocol))
            { continue; }

            if (!component.type_list.isEmpty() &&
                !component.type_list.contains(type))
            { continue; }
            if (!info.possible_type_list.isEmpty() &&
                !info.possible_type_list.contains(type))
            { continue; }

            if (!component.target_type_list.isEmpty() &&
                !component.target_type_list.contains(target_type))
            { continue; }
            if (!info.possible_target_type_list.isEmpty() &&
                !info.possible_target_type_list.contains(target_type))
            { continue; }

            qDebug() << "PE: considering" << info.service_name << "...";

            results.append(component.component_handle);
        }

        return results;
    }

    /** @brief A pointer to the ComponentManager. */
    const ComponentManager * const m_componentMgr;
};

/// @endcond

// **************************************************************************

PolicyEngine::PolicyEngine(const ComponentManager * component_mgr,
                           QObject * parent) :
        QObject(parent),
        d(new PolicyEnginePrivate(component_mgr))
{
    Q_ASSERT(0 != d);
}

PolicyEngine::~PolicyEngine()
{ delete d; }

void PolicyEngine::onChannelOpened(QtTapioca::Connection * connection,
                                   QtTapioca::Channel * channel,
                                   const bool incoming) const
{ d->activateHandler(connection, channel, incoming); }
