# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: DownloaderFeedback.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Gather and display statistics about the download.

@type INIT_STATE: C{tuple} of C{string}
@var INIT_STATE: the state to display, 'L'/'R' = locally/remotely initiated, 
    '+' = encrypted

"""

from cStringIO import StringIO
from urllib import quote
from threading import Event

INIT_STATE = (('R','R+'),('L','L+'))

class DownloaderFeedback:
    """Gather and display statistics about the download.
    
    @type choker: L{Choker.Choker}
    @ivar choker: the Choker instance for the download
    @type httpdl: L{HTTPDownloder.HTTPDownloader}
    @ivar httpdl: the HTTP downloader instance used by the download
    @type add_task: C{method}
    @ivar add_task: the method to call to schedule future tasks
    @type upfunc: C{method}
    @ivar upfunc: the method to call to get the upload rate
    @type downfunc: C{method}
    @ivar downfunc: the method to call to get the download rate
    @type ratemeasure: L{DebTorrent.RateMeasure.RateMeasure}
    @ivar ratemeasure: the download rate measurement for the download
    @type leftfunc: C{method}
    @ivar leftfunc: the method to call to determine the amount of data 
            left to download
    @type file_length: C{long}
    @ivar file_length: the total length of the download
    @type finflag: C{threading.Event}
    @ivar finflag: flag to indicate when the download is complete
    @type statistics: L{Statistics.Statistics}
    @ivar statistics: the statistics gatherer for the download
    @type lastids: C{list} of C{string}
    @ivar lastids: the peer IDs that were found the last time
    @type doneprocessing: C{threading.Event}
    @ivar doneprocessing: flag that is set when the previous display is complete
    @type displayfunc: C{method}
    @ivar displayfunc: the method to call to display the gathered data
    @type interval: C{int}
    @ivar interval: the number of seconds between displays of the gathered data
    
    """
    
    def __init__(self, choker, httpdl, add_task, upfunc, downfunc,
            ratemeasure, leftfunc, file_length, finflag, statistics,
            statusfunc = None, interval = None):
        """Initialize the instance.
        
        @type choker: L{Choker.Choker}
        @param choker: the Choker instance for the download
        @type httpdl: L{HTTPDownloder.HTTPDownloader}
        @param httpdl: the HTTP downloader instance used by the download
        @type add_task: C{method}
        @param add_task: the method to call to schedule future tasks
        @type upfunc: C{method}
        @param upfunc: the method to call to get the upload rate
        @type downfunc: C{method}
        @param downfunc: the method to call to get the download rate
        @type ratemeasure: L{DebTorrent.RateMeasure.RateMeasure}
        @param ratemeasure: the download rate measurement for the download
        @type leftfunc: C{method}
        @param leftfunc: the method to call to determine the amount of data 
            left to download
        @type file_length: C{long}
        @param file_length: the total length of the download
        @type finflag: C{threading.Event}
        @param finflag: flag to indicate when the download is complete
        @type statistics: L{Statistics.Statistics}
        @param statistics: the statistics gatherer for the download
        @type statusfunc: C{method}
        @param statusfunc: the method to call to automatically display the gathered data
            (optional, default is to not automatically display anything)
        @type interval: C{int}
        @param interval: the number of seconds between displays of the gathered data
            (optional, but must be set if the L{statusfunc} is set)
        
        """
        
        self.choker = choker
        self.httpdl = httpdl
        self.add_task = add_task
        self.upfunc = upfunc
        self.downfunc = downfunc
        self.ratemeasure = ratemeasure
        self.leftfunc = leftfunc
        self.file_length = file_length
        self.finflag = finflag
        self.statistics = statistics
        self.lastids = []
        self.doneprocessing = Event()
        self.doneprocessing.set()
        if statusfunc:
            self.autodisplay(statusfunc, interval)
        

    def _rotate(self):
        """Rotate the list of connections so it starts roughly where it did before.
        
        @rtype: C{list} of L{Connecter.Connection}
        @return: the connections from peers to the client
        
        """
        
        cs = self.choker.connections
        for id in self.lastids:
            for i in xrange(len(cs)):
                if cs[i].get_id() == id:
                    return cs[i:] + cs[:i]
        return cs

    def spews(self):
        """Get information about the currently connected peers.
        
        @rtype: C{list} of C{dictionary}
        @return: information about all the current peer connections
        
        """
        
        l = []
        cs = self._rotate()
        self.lastids = [c.get_id() for c in cs]
        for c in cs:
            a = {}
            a['id'] = c.get_readable_id()
            a['ip'] = c.get_ip()
            a['optimistic'] = (c is self.choker.connections[0])
            a['direction'] = INIT_STATE[c.is_locally_initiated()][c.is_encrypted()]
            u = c.get_upload()
            a['uprate'] = int(u.measure.get_rate())
            a['uinterested'] = u.is_interested()
            a['uchoked'] = u.is_choked()
            d = c.get_download()
            a['downrate'] = int(d.measure.get_rate())
            a['dinterested'] = d.is_interested()
            a['dchoked'] = d.is_choked()
            a['snubbed'] = d.is_snubbed()
            a['utotal'] = d.connection.upload.measure.get_total()
            a['dtotal'] = d.connection.download.measure.get_total()
            if len(d.connection.download.have) > 0:
                a['completed'] = float(len(d.connection.download.have)-d.connection.download.have.numfalse)/float(len(d.connection.download.have))
            else:
                a['completed'] = 1.0
            a['speed'] = d.connection.download.peermeasure.get_rate()

            l.append(a)                                               

        for dl in self.httpdl.get_downloads():
            if dl.goodseed:
                a = {}
                a['id'] = 'http seed'
                a['ip'] = dl.baseurl
                a['optimistic'] = False
                a['direction'] = 'L'
                a['uprate'] = 0
                a['uinterested'] = False
                a['uchoked'] = False
                a['downrate'] = int(dl.measure.get_rate())
                a['dinterested'] = True
                a['dchoked'] = not dl.active
                a['snubbed'] = not dl.active
                a['utotal'] = None
                a['dtotal'] = dl.measure.get_total()
                a['completed'] = 1.0
                a['speed'] = None

                l.append(a)

        return l


    def gather(self, displayfunc = None):
        """Gather the information about the download.
        
        @type displayfunc: C{method}
        @param displayfunc: not used (optional)
        @rtype: C{dictionary}
        @return: various information about the download
        
        """
        
        s = {'stats': self.statistics.update()}
        s['up'] = self.upfunc()
        if self.finflag.isSet():
            s['done'] = self.file_length
            return s
        s['down'] = self.downfunc()
        obtained, desired = self.leftfunc()
        s['done'] = obtained
        s['wanted'] = desired
        if desired > 0:
            s['frac'] = float(obtained)/desired
        else:
            s['frac'] = 1.0
        if desired == obtained:
            s['time'] = 0
        else:
            s['time'] = self.ratemeasure.get_time_left(desired-obtained)
        return s        


    def display(self, displayfunc):
        """Gather the information about the download and display it.
        
        @type displayfunc: C{method}
        @param displayfunc: the method to call with the gathered information
        
        """
        
        if not self.doneprocessing.isSet():
            return
        self.doneprocessing.clear()
        stats = self.gather()
        if self.finflag.isSet():
            displayfunc(dpflag = self.doneprocessing,
                upRate = stats['up'],
                statistics = stats['stats'])
        elif stats['time'] is not None:
            displayfunc(dpflag = self.doneprocessing,
                fractionDone = stats['frac'], sizeDone = stats['done'],
                downRate = stats['down'], upRate = stats['up'],
                statistics = stats['stats'],
                timeEst = stats['time'])
        else:
            displayfunc(dpflag = self.doneprocessing,
                fractionDone = stats['frac'], sizeDone = stats['done'],
                downRate = stats['down'], upRate = stats['up'],
                statistics = stats['stats'])


    def autodisplay(self, displayfunc, interval):
        """Initialize the automatic displayer.
        
        @type displayfunc: C{method}
        @param displayfunc: the method to call to display the gathered data
        @type interval: C{int}
        @param interval: the number of seconds between displays of the gathered data
        
        """
        
        self.displayfunc = displayfunc
        self.interval = interval
        self._autodisplay()

    def _autodisplay(self):
        """Call the displayer and reschedule."""
        self.add_task(self._autodisplay, self.interval)
        self.display(self.displayfunc)
