# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: __init__.py 327 2008-01-10 04:39:34Z camrdale-guest $

"""The main package to implement the DebTorrent protocol.

This package, and it's subpackage L{BT1}, contains all the modules needed
to implement the DebTorrent protocol.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type product_name: C{string}
@var product_name: the name given for the package
@type version_short: C{string}
@var version_short: the short version number
@type protocol_name: C{string}
@var protocol_name: the protocol name to use in handshaking
@type mapbase64: C{string}
@var mapbase64: the mapping from 64 bit numbers to string characters

"""

product_name = 'DebTorrent'
version_short = 'T-0.1.5'

version = version_short+' ('+product_name+')'
report_email = 'debtorrent-devel@lists.alioth.debian.org'

from types import StringType
from sha import sha
from time import time, clock
from binascii import b2a_hex
from urllib import quote
import logging
try:
    from os import getpid
except ImportError:
    def getpid():
        return 1

logger = logging.getLogger('DebTorrent')

protocol_name = product_name+'/'+'.'.join(version_short[2:].split('.')[:2])

mapbase64 = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz.-'

_idprefix = version_short[0]
for subver in version_short[2:].split('.'):
    try:
        subver = int(subver)
    except:
        subver = 0
    _idprefix += mapbase64[subver]
_idprefix += ('-' * (6-len(_idprefix)))
_idrandom = [None]

def resetPeerIDs():
    """Reset the generation of peer IDs before generating a new random one."""
    try:
        f = open('/dev/urandom','rb')
        x = f.read(20)
        f.close()
    except:
        x = ''

    logger.info('Resetting the peer ID generation')
    l1 = 0
    t = clock()
    while t == clock():
        l1 += 1
    l2 = 0
    t = long(time()*100)
    while t == long(time()*100):
        l2 += 1
    l3 = 0
    if l2 < 1000:
        t = long(time()*10)
        while t == long(clock()*10):
            l3 += 1
    x += ( repr(time()) + '/' + str(time()) + '/'
           + str(l1) + '/' + str(l2) + '/' + str(l3) + '/'
           + str(getpid()) )

    s = ''
    for i in sha(x).digest()[-11:]:
        s += mapbase64[ord(i) & 0x3F]
    _idrandom[0] = s
        
resetPeerIDs()

def make_readable(s):
    """Convert a string peer ID to be human-readable.
    
    @type s: C{string}
    @param s: the string to convert
    @rtype: C{string}
    @return: the resulting hex string, or the original string if it was already
        readable
    
    """
    
    if not s:
        return ''
    if quote(s).find('%') >= 0:
        return b2a_hex(s)
    return '"'+s+'"'

def createPeerID(ins = '---'):
    """Generate a somewhat random peer ID
    
    @type ins: C{string}
    @param ins: the length 3 string to insert in the middle of the peer ID
        between the prefix and the random part of the ID
        (optional, defaults to '---')
    @rtype: C{string}
    @return: the peer ID to use
    
    """
    
    assert type(ins) is StringType
    assert len(ins) == 3
    logger.info('New peer ID: '+make_readable(_idprefix + ins + _idrandom[0]))
    return _idprefix + ins + _idrandom[0]
