#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "control_file.h"

#include "architecture.h"
#include "cross_compiling.h"

#include "unknown_argument.h"

#include "silent_progress.h"

#include <apt-pkg/init.h>
#include <apt-pkg/cachefile.h>

#include <boost/lexical_cast.hpp>

#include <iostream>
#include <exception>

int main(int const argc, char const *const *const argv) throw()
try
{
	typedef char const *const *arg_iterator;
	arg_iterator const args_begin = &argv[1];
	arg_iterator const args_end = &argv[argc];

	bool archdep_only = false;

	for(arg_iterator a = args_begin; a != args_end; ++a)
	{
		std::string arg(*a);
		if(arg == "-B")
			archdep_only = true;
		else
			throw unknown_argument(arg);
	}

	control_file control;
	try
	{
		control.merge("debian/xcontrol");
	}
	catch(std::exception &)
	{
		control.merge("debian/control");
	}
	dependency_list dependencies = control.source().build_depends_tools;

	if(!archdep_only)
		dependencies += control.source().build_depends_indep;

	if(architecture::host != architecture::build)
		dependencies += cross_compiling::to_cross(architecture::host, control.source().build_depends);
	else
		dependencies += control.source().build_depends;

	bool successful = true;

	pkgInitConfig(*_config);
	pkgInitSystem(*_config, _system);

	silent_progress progress;

	pkgCacheFile cache;
	cache.Open(progress, false);
	
	for(dependency_list::const_iterator i = dependencies.begin(); i != dependencies.end(); ++i)
	{
		pkgCache::PkgIterator pkg = cache->FindPkg(i->package);
		if(pkg.end())
		{
			std::cerr << "E: dependency on " << *i
				<< " not fulfilled (and package not available)"
				<< std::endl;
			successful = false;
		}
		else
		{
			if(pkg->ProvidesList)
			{
				std::cerr << "I: " << i->package << " is a virtual package." << std::endl;
				for(pkgCache::PrvIterator p = pkg.ProvidesList(); !p.end(); ++p)
				{
					pkgCache::PkgIterator pp = p.OwnerPkg();
					if(cache[pp].Status == 2)
						continue;
					std::cerr << "I: " << pp.Name() << " "
						<< pp.CurrentVer().VerStr()
						<< " provides it." << std::endl;
				}
			}
			else
			{
				if(pkg->CurrentVer)
					std::cerr << "I: " << *i << " / " << pkg.CurrentVer().VerStr() << std::endl;
				else
				{
					std::cerr << "E: dependency on " << *i
						<< " not fulfilled."
						<< std::endl;
					successful = false;
				}
			}
		}
	}

	cache.Close();

	return successful?0:1;
}
catch(std::exception &e)
{
	std::cerr << "E: " << e.what() << std::endl;
	return 1;
}
