#ifndef DBA_MSG_VARS_H
#define DBA_MSG_VARS_H

#ifdef  __cplusplus
extern "C" {
#endif

/** @file
 * @ingroup msg
 * Shortcut IDs and functions to quickly refer to commonly used values inside a
 * ::dba_msg.
 *
 * The shortcuts are defined in the file vars.csv, which maps a name to the tuple
 * (preferred C type, BLocal, Level type, L1, L2, P indicator, P1, P2).  The file
 * vars.csv is used to generate code (vars.h and vars.c) that provides a set of
 * getters and setters for every named value.
 */

#include <dballe/msg/msg.h>

/**
 * Full data corresponding to a shortcut ID
 */
struct _dba_msg_var {
    /** Variable description code */
    dba_varcode code;
    /** Type of the first level.  See @ref level_table. */
    int ltype1;
    /** L1 value of the level.  See @ref level_table. */
    int l1;
    /** Type of the second level.  See @ref level_table. */
    int ltype2;
    /** L2 value of the level.  See @ref level_table. */
    int l2;
    /** Time range type indicator.  See @ref trange_table. */
    int pind;
    /** Time range P1 indicator.  See @ref trange_table. */
    int p1;
    /** Time range P2 indicator.  See @ref trange_table. */
    int p2;
};
/** @copydoc _dba_msg_var */
typedef struct _dba_msg_var* dba_msg_var;

/** Array with all the full expansions of the shortcut IDs, indexed by shortcut
  * ID
  */
extern struct _dba_msg_var dba_msg_vartable[];


/** Total precipitation in the last hour */
#define DBA_MSG_TOT_PREC1 0
/** Total precipitation in the last 3 hours */
#define DBA_MSG_TOT_PREC3 1
/** Total precipitation in the last 6 hours */
#define DBA_MSG_TOT_PREC6 2
/** Total precipitation in the last 12 hours */
#define DBA_MSG_TOT_PREC12 3
/** Total precipitation in the last 24 hours */
#define DBA_MSG_TOT_PREC24 4
/** 3 hour pressure change at ground level */
#define DBA_MSG_PRESS_3H 5
/** Characteristic of pressure tendency */
#define DBA_MSG_PRESS_TEND 6
/** Past weather (1) */
#define DBA_MSG_PAST_WTR1 7
/** Past weather (2) */
#define DBA_MSG_PAST_WTR2 8
/** Geopotential */
#define DBA_MSG_GEOPOTENTIAL 9
/** Pressure at ground level */
#define DBA_MSG_PRESS 10
/** Anemometer height */
#define DBA_MSG_HEIGHT_ANEM 11
/** Total show depth */
#define DBA_MSG_TOT_SNOW 12
/** Visibility */
#define DBA_MSG_VISIBILITY 13
/** Present weather */
#define DBA_MSG_PRES_WTR 14
/** General weather indicator (TAF/METAR) */
#define DBA_MSG_METAR_WTR 15
/** Sea/water surface temperature */
#define DBA_MSG_WATER_TEMP 16
/** Pressure reduced to mean sea level */
#define DBA_MSG_PRESS_MSL 17
/** Altimeter setting (QNH) */
#define DBA_MSG_QNH 18
/** Temperature at 2 metres above ground */
#define DBA_MSG_TEMP_2M 19
/** Wet bulb temperature at 2 metres above ground */
#define DBA_MSG_WET_TEMP_2M 20
/** Dew point at 2 metres above ground */
#define DBA_MSG_DEWPOINT_2M 21
/** Humidity at 2 metres above ground */
#define DBA_MSG_HUMIDITY 22
/** Wind direction at 10 metres above ground */
#define DBA_MSG_WIND_DIR 23
/** Wind speed at 10 metres above ground */
#define DBA_MSG_WIND_SPEED 24
/** Extreme counterclockwise wind direction of a variable wind at 10 metres above ground */
#define DBA_MSG_EX_CCW_WIND 25
/** Extreme clockwise wind direction of a variable wind at 10 metres above ground */
#define DBA_MSG_EX_CW_WIND 26
/** Maximum wind gust speed at 10 metres above ground */
#define DBA_MSG_WIND_MAX 27
/** Total cloud cover (N) */
#define DBA_MSG_CLOUD_N 28
/** Cloud amount (NH) */
#define DBA_MSG_CLOUD_NH 29
/** Height of base of cloud (HH) */
#define DBA_MSG_CLOUD_HH 30
/** Cloud type (CL) */
#define DBA_MSG_CLOUD_CL 31
/** Cloud type (CM) */
#define DBA_MSG_CLOUD_CM 32
/** Cloud type (CH) */
#define DBA_MSG_CLOUD_CH 33
/** WMO block number */
#define DBA_MSG_BLOCK 34
/** WMO station number */
#define DBA_MSG_STATION 35
/** Aircraft registration number or other identification */
#define DBA_MSG_FLIGHT_REG_NO 36
/** Mobile station identifier */
#define DBA_MSG_IDENT 37
/** Direction of motion of moving observing platform */
#define DBA_MSG_ST_DIR 38
/** Speed of motion of moving observing platform */
#define DBA_MSG_ST_SPEED 39
/** Station or site name */
#define DBA_MSG_ST_NAME 40
/** ICAO location indicator */
#define DBA_MSG_ST_NAME_ICAO 41
/** Report code */
#define DBA_MSG_REP_COD 42
/** Air quality observing station local code */
#define DBA_MSG_POLL_LCODE 43
/** Airbase air quality observing station code */
#define DBA_MSG_POLL_SCODE 44
/** GEMS air quality observing station code */
#define DBA_MSG_POLL_GEMSCODE 45
/** Air quality observing station dominant emission source */
#define DBA_MSG_POLL_SOURCE 46
/** Air quality observing station area type */
#define DBA_MSG_POLL_ATYPE 47
/** Air quality observing station terrain type */
#define DBA_MSG_POLL_TTYPE 48
/** Type of station */
#define DBA_MSG_ST_TYPE 49
/** Type of instrumentation for wind measurement */
#define DBA_MSG_WIND_INST 50
/** Precision of temperature observation */
#define DBA_MSG_TEMP_PRECISION 51
/** Radiosonde type */
#define DBA_MSG_SONDE_TYPE 52
/** Radiosonde computational method */
#define DBA_MSG_SONDE_METHOD 53
/** Aircraft navigational system */
#define DBA_MSG_NAVSYS 54
/** Aircraft data relay system */
#define DBA_MSG_DATA_RELAY 55
/** Aircraft roll angle */
#define DBA_MSG_FLIGHT_ROLL 56
/** Original specification of latitude/longitude */
#define DBA_MSG_LATLON_SPEC 57
/** Year of the observation */
#define DBA_MSG_YEAR 58
/** Month of the observation */
#define DBA_MSG_MONTH 59
/** Day of the observation */
#define DBA_MSG_DAY 60
/** Hour of the observation */
#define DBA_MSG_HOUR 61
/** Minute of the observation */
#define DBA_MSG_MINUTE 62
/** Second of the observation */
#define DBA_MSG_SECOND 63
/** Latitude of the station */
#define DBA_MSG_LATITUDE 64
/** Longiture of the station */
#define DBA_MSG_LONGITUDE 65
/** Height of station */
#define DBA_MSG_HEIGHT 66
/** Isobaric surface */
#define DBA_MSG_ISOBARIC_SURFACE 67
/** Height of barometer above mean sea level */
#define DBA_MSG_HEIGHT_BARO 68
/** Phase of aircraft flight */
#define DBA_MSG_FLIGHT_PHASE 69
/** Time significance */
#define DBA_MSG_TIMESIG 70
/** Cloud amount (N1) */
#define DBA_MSG_CLOUD_N1 71
/** Cloud amount (C1) */
#define DBA_MSG_CLOUD_C1 72
/** Height of base of cloud (H1) */
#define DBA_MSG_CLOUD_H1 73
/** Cloud amount (N2) */
#define DBA_MSG_CLOUD_N2 74
/** Cloud amount (C2) */
#define DBA_MSG_CLOUD_C2 75
/** Height of base of cloud (H2) */
#define DBA_MSG_CLOUD_H2 76
/** Cloud amount (N3) */
#define DBA_MSG_CLOUD_N3 77
/** Cloud amount (C3) */
#define DBA_MSG_CLOUD_C3 78
/** Height of base of cloud (H3) */
#define DBA_MSG_CLOUD_H3 79
/** Cloud amount (N4) */
#define DBA_MSG_CLOUD_N4 80
/** Cloud amount (C4) */
#define DBA_MSG_CLOUD_C4 81
/** Height of base of cloud (H4) */
#define DBA_MSG_CLOUD_H4 82
/** Number of items in dba_msg_vartable */
#define DBA_MSG_VARTABLE_SIZE 82
/** Set the value of "Total precipitation in the last hour" from a variable of type double */
inline static dba_err dba_msg_set_tot_prec1(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 11), val, conf, 1, 0, 0, 0, 1, 0, 3600); }
/** Set the value of "Total precipitation in the last hour" from a ::dba_var */
inline static dba_err dba_msg_set_tot_prec1_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 11), 1, 0, 0, 0, 1, 0, 3600); }
/** Get the "Total precipitation in the last hour" physical value stored in the message */
inline static dba_var dba_msg_get_tot_prec1_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_PREC1); return d == NULL ? NULL : d->var; }
/** Set the value of "Total precipitation in the last 3 hours" from a variable of type double */
inline static dba_err dba_msg_set_tot_prec3(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 11), val, conf, 1, 0, 0, 0, 1, 0, 10800); }
/** Set the value of "Total precipitation in the last 3 hours" from a ::dba_var */
inline static dba_err dba_msg_set_tot_prec3_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 11), 1, 0, 0, 0, 1, 0, 10800); }
/** Get the "Total precipitation in the last 3 hours" physical value stored in the message */
inline static dba_var dba_msg_get_tot_prec3_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_PREC3); return d == NULL ? NULL : d->var; }
/** Set the value of "Total precipitation in the last 6 hours" from a variable of type double */
inline static dba_err dba_msg_set_tot_prec6(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 11), val, conf, 1, 0, 0, 0, 1, 0, 21600); }
/** Set the value of "Total precipitation in the last 6 hours" from a ::dba_var */
inline static dba_err dba_msg_set_tot_prec6_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 11), 1, 0, 0, 0, 1, 0, 21600); }
/** Get the "Total precipitation in the last 6 hours" physical value stored in the message */
inline static dba_var dba_msg_get_tot_prec6_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_PREC6); return d == NULL ? NULL : d->var; }
/** Set the value of "Total precipitation in the last 12 hours" from a variable of type double */
inline static dba_err dba_msg_set_tot_prec12(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 11), val, conf, 1, 0, 0, 0, 1, 0, 43200); }
/** Set the value of "Total precipitation in the last 12 hours" from a ::dba_var */
inline static dba_err dba_msg_set_tot_prec12_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 11), 1, 0, 0, 0, 1, 0, 43200); }
/** Get the "Total precipitation in the last 12 hours" physical value stored in the message */
inline static dba_var dba_msg_get_tot_prec12_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_PREC12); return d == NULL ? NULL : d->var; }
/** Set the value of "Total precipitation in the last 24 hours" from a variable of type double */
inline static dba_err dba_msg_set_tot_prec24(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 11), val, conf, 1, 0, 0, 0, 1, 0, 86400); }
/** Set the value of "Total precipitation in the last 24 hours" from a ::dba_var */
inline static dba_err dba_msg_set_tot_prec24_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 11), 1, 0, 0, 0, 1, 0, 86400); }
/** Get the "Total precipitation in the last 24 hours" physical value stored in the message */
inline static dba_var dba_msg_get_tot_prec24_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_PREC24); return d == NULL ? NULL : d->var; }
/** Set the value of "3 hour pressure change at ground level" from a variable of type double */
inline static dba_err dba_msg_set_press_3h(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 61), val, conf, 1, 0, 0, 0, 4, 0, 10800); }
/** Set the value of "3 hour pressure change at ground level" from a ::dba_var */
inline static dba_err dba_msg_set_press_3h_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 61), 1, 0, 0, 0, 4, 0, 10800); }
/** Get the "3 hour pressure change at ground level" physical value stored in the message */
inline static dba_var dba_msg_get_press_3h_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PRESS_3H); return d == NULL ? NULL : d->var; }
/** Set the value of "Characteristic of pressure tendency" from a variable of type double */
inline static dba_err dba_msg_set_press_tend(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 63), val, conf, 1, 0, 0, 0, 205, 0, 10800); }
/** Set the value of "Characteristic of pressure tendency" from a ::dba_var */
inline static dba_err dba_msg_set_press_tend_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 63), 1, 0, 0, 0, 205, 0, 10800); }
/** Get the "Characteristic of pressure tendency" physical value stored in the message */
inline static dba_var dba_msg_get_press_tend_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PRESS_TEND); return d == NULL ? NULL : d->var; }
/** Set the value of "Past weather (1)" from a variable of type int */
inline static dba_err dba_msg_set_past_wtr1(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 4), val, conf, 1, 0, 0, 0, 205, 0, 21600); }
/** Set the value of "Past weather (1)" from a ::dba_var */
inline static dba_err dba_msg_set_past_wtr1_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 4), 1, 0, 0, 0, 205, 0, 21600); }
/** Get the "Past weather (1)" physical value stored in the message */
inline static dba_var dba_msg_get_past_wtr1_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PAST_WTR1); return d == NULL ? NULL : d->var; }
/** Set the value of "Past weather (2)" from a variable of type int */
inline static dba_err dba_msg_set_past_wtr2(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 5), val, conf, 1, 0, 0, 0, 205, 0, 21600); }
/** Set the value of "Past weather (2)" from a ::dba_var */
inline static dba_err dba_msg_set_past_wtr2_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 5), 1, 0, 0, 0, 205, 0, 21600); }
/** Get the "Past weather (2)" physical value stored in the message */
inline static dba_var dba_msg_get_past_wtr2_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PAST_WTR2); return d == NULL ? NULL : d->var; }
/** Set the value of "Geopotential" from a variable of type double */
inline static dba_err dba_msg_set_geopotential(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 3), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Geopotential" from a ::dba_var */
inline static dba_err dba_msg_set_geopotential_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 3), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Geopotential" physical value stored in the message */
inline static dba_var dba_msg_get_geopotential_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_GEOPOTENTIAL); return d == NULL ? NULL : d->var; }
/** Set the value of "Pressure at ground level" from a variable of type double */
inline static dba_err dba_msg_set_press(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 4), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Pressure at ground level" from a ::dba_var */
inline static dba_err dba_msg_set_press_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 4), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Pressure at ground level" physical value stored in the message */
inline static dba_var dba_msg_get_press_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PRESS); return d == NULL ? NULL : d->var; }
/** Set the value of "Anemometer height" from a variable of type double */
inline static dba_err dba_msg_set_height_anem(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 197), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Anemometer height" from a ::dba_var */
inline static dba_err dba_msg_set_height_anem_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 197), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Anemometer height" physical value stored in the message */
inline static dba_var dba_msg_get_height_anem_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_HEIGHT_ANEM); return d == NULL ? NULL : d->var; }
/** Set the value of "Total show depth" from a variable of type double */
inline static dba_err dba_msg_set_tot_snow(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 13), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Total show depth" from a ::dba_var */
inline static dba_err dba_msg_set_tot_snow_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 13), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Total show depth" physical value stored in the message */
inline static dba_var dba_msg_get_tot_snow_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TOT_SNOW); return d == NULL ? NULL : d->var; }
/** Set the value of "Visibility" from a variable of type double */
inline static dba_err dba_msg_set_visibility(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 1), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Visibility" from a ::dba_var */
inline static dba_err dba_msg_set_visibility_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 1), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Visibility" physical value stored in the message */
inline static dba_var dba_msg_get_visibility_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_VISIBILITY); return d == NULL ? NULL : d->var; }
/** Set the value of "Present weather" from a variable of type int */
inline static dba_err dba_msg_set_pres_wtr(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 3), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Present weather" from a ::dba_var */
inline static dba_err dba_msg_set_pres_wtr_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 3), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Present weather" physical value stored in the message */
inline static dba_var dba_msg_get_pres_wtr_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PRES_WTR); return d == NULL ? NULL : d->var; }
/** Set the value of "General weather indicator (TAF/METAR)" from a variable of type int */
inline static dba_err dba_msg_set_metar_wtr(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 9), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "General weather indicator (TAF/METAR)" from a ::dba_var */
inline static dba_err dba_msg_set_metar_wtr_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 9), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "General weather indicator (TAF/METAR)" physical value stored in the message */
inline static dba_var dba_msg_get_metar_wtr_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_METAR_WTR); return d == NULL ? NULL : d->var; }
/** Set the value of "Sea/water surface temperature" from a variable of type double */
inline static dba_err dba_msg_set_water_temp(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 22, 42), val, conf, 1, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Sea/water surface temperature" from a ::dba_var */
inline static dba_err dba_msg_set_water_temp_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 22, 42), 1, 0, 0, 0, 254, 0, 0); }
/** Get the "Sea/water surface temperature" physical value stored in the message */
inline static dba_var dba_msg_get_water_temp_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WATER_TEMP); return d == NULL ? NULL : d->var; }
/** Set the value of "Pressure reduced to mean sea level" from a variable of type double */
inline static dba_err dba_msg_set_press_msl(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 51), val, conf, 101, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Pressure reduced to mean sea level" from a ::dba_var */
inline static dba_err dba_msg_set_press_msl_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 51), 101, 0, 0, 0, 254, 0, 0); }
/** Get the "Pressure reduced to mean sea level" physical value stored in the message */
inline static dba_var dba_msg_get_press_msl_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_PRESS_MSL); return d == NULL ? NULL : d->var; }
/** Set the value of "Altimeter setting (QNH)" from a variable of type double */
inline static dba_err dba_msg_set_qnh(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 10, 52), val, conf, 103, 2000, 0, 0, 254, 0, 0); }
/** Set the value of "Altimeter setting (QNH)" from a ::dba_var */
inline static dba_err dba_msg_set_qnh_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 10, 52), 103, 2000, 0, 0, 254, 0, 0); }
/** Get the "Altimeter setting (QNH)" physical value stored in the message */
inline static dba_var dba_msg_get_qnh_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_QNH); return d == NULL ? NULL : d->var; }
/** Set the value of "Temperature at 2 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_temp_2m(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 12, 1), val, conf, 103, 2000, 0, 0, 254, 0, 0); }
/** Set the value of "Temperature at 2 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_temp_2m_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 12, 1), 103, 2000, 0, 0, 254, 0, 0); }
/** Get the "Temperature at 2 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_temp_2m_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TEMP_2M); return d == NULL ? NULL : d->var; }
/** Set the value of "Wet bulb temperature at 2 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_wet_temp_2m(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 12, 2), val, conf, 103, 2000, 0, 0, 254, 0, 0); }
/** Set the value of "Wet bulb temperature at 2 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_wet_temp_2m_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 12, 2), 103, 2000, 0, 0, 254, 0, 0); }
/** Get the "Wet bulb temperature at 2 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_wet_temp_2m_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WET_TEMP_2M); return d == NULL ? NULL : d->var; }
/** Set the value of "Dew point at 2 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_dewpoint_2m(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 12, 3), val, conf, 103, 2000, 0, 0, 254, 0, 0); }
/** Set the value of "Dew point at 2 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_dewpoint_2m_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 12, 3), 103, 2000, 0, 0, 254, 0, 0); }
/** Get the "Dew point at 2 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_dewpoint_2m_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_DEWPOINT_2M); return d == NULL ? NULL : d->var; }
/** Set the value of "Humidity at 2 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_humidity(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 13, 3), val, conf, 103, 2000, 0, 0, 254, 0, 0); }
/** Set the value of "Humidity at 2 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_humidity_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 13, 3), 103, 2000, 0, 0, 254, 0, 0); }
/** Get the "Humidity at 2 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_humidity_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_HUMIDITY); return d == NULL ? NULL : d->var; }
/** Set the value of "Wind direction at 10 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_wind_dir(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 11, 1), val, conf, 103, 10000, 0, 0, 254, 0, 0); }
/** Set the value of "Wind direction at 10 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_wind_dir_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 11, 1), 103, 10000, 0, 0, 254, 0, 0); }
/** Get the "Wind direction at 10 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_wind_dir_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WIND_DIR); return d == NULL ? NULL : d->var; }
/** Set the value of "Wind speed at 10 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_wind_speed(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 11, 2), val, conf, 103, 10000, 0, 0, 254, 0, 0); }
/** Set the value of "Wind speed at 10 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_wind_speed_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 11, 2), 103, 10000, 0, 0, 254, 0, 0); }
/** Get the "Wind speed at 10 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_wind_speed_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WIND_SPEED); return d == NULL ? NULL : d->var; }
/** Set the value of "Extreme counterclockwise wind direction of a variable wind at 10 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_ex_ccw_wind(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 11, 16), val, conf, 103, 10000, 0, 0, 254, 0, 0); }
/** Set the value of "Extreme counterclockwise wind direction of a variable wind at 10 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_ex_ccw_wind_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 11, 16), 103, 10000, 0, 0, 254, 0, 0); }
/** Get the "Extreme counterclockwise wind direction of a variable wind at 10 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_ex_ccw_wind_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_EX_CCW_WIND); return d == NULL ? NULL : d->var; }
/** Set the value of "Extreme clockwise wind direction of a variable wind at 10 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_ex_cw_wind(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 11, 17), val, conf, 103, 10000, 0, 0, 254, 0, 0); }
/** Set the value of "Extreme clockwise wind direction of a variable wind at 10 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_ex_cw_wind_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 11, 17), 103, 10000, 0, 0, 254, 0, 0); }
/** Get the "Extreme clockwise wind direction of a variable wind at 10 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_ex_cw_wind_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_EX_CW_WIND); return d == NULL ? NULL : d->var; }
/** Set the value of "Maximum wind gust speed at 10 metres above ground" from a variable of type double */
inline static dba_err dba_msg_set_wind_max(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 11, 41), val, conf, 103, 10000, 0, 0, 254, 0, 0); }
/** Set the value of "Maximum wind gust speed at 10 metres above ground" from a ::dba_var */
inline static dba_err dba_msg_set_wind_max_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 11, 41), 103, 10000, 0, 0, 254, 0, 0); }
/** Get the "Maximum wind gust speed at 10 metres above ground" physical value stored in the message */
inline static dba_var dba_msg_get_wind_max_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WIND_MAX); return d == NULL ? NULL : d->var; }
/** Set the value of "Total cloud cover (N)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_n(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 10), val, conf, 256, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Total cloud cover (N)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_n_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 10), 256, 0, 0, 0, 254, 0, 0); }
/** Get the "Total cloud cover (N)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_n_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_N); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (NH)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_nh(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 11), val, conf, 256, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (NH)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_nh_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 11), 256, 0, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (NH)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_nh_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_NH); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of base of cloud (HH)" from a variable of type double */
inline static dba_err dba_msg_set_cloud_hh(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 13), val, conf, 256, 0, 0, 0, 254, 0, 0); }
/** Set the value of "Height of base of cloud (HH)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_hh_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 13), 256, 0, 0, 0, 254, 0, 0); }
/** Get the "Height of base of cloud (HH)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_hh_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_HH); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud type (CL)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_cl(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 256, 1, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud type (CL)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_cl_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 256, 1, 0, 0, 254, 0, 0); }
/** Get the "Cloud type (CL)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_cl_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_CL); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud type (CM)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_cm(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 256, 2, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud type (CM)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_cm_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 256, 2, 0, 0, 254, 0, 0); }
/** Get the "Cloud type (CM)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_cm_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_CM); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud type (CH)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_ch(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 256, 3, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud type (CH)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_ch_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 256, 3, 0, 0, 254, 0, 0); }
/** Get the "Cloud type (CH)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_ch_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_CH); return d == NULL ? NULL : d->var; }
/** Set the value of "WMO block number" from a variable of type int */
inline static dba_err dba_msg_set_block(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "WMO block number" from a ::dba_var */
inline static dba_err dba_msg_set_block_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "WMO block number" physical value stored in the message */
inline static dba_var dba_msg_get_block_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_BLOCK); return d == NULL ? NULL : d->var; }
/** Set the value of "WMO station number" from a variable of type int */
inline static dba_err dba_msg_set_station(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 2), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "WMO station number" from a ::dba_var */
inline static dba_err dba_msg_set_station_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 2), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "WMO station number" physical value stored in the message */
inline static dba_var dba_msg_get_station_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_STATION); return d == NULL ? NULL : d->var; }
/** Set the value of "Aircraft registration number or other identification" from a variable of type const char* */
inline static dba_err dba_msg_set_flight_reg_no(dba_msg msg, const char* val, int conf)  { return dba_msg_setc(msg, DBA_VAR(0, 1, 8), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Aircraft registration number or other identification" from a ::dba_var */
inline static dba_err dba_msg_set_flight_reg_no_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 8), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Aircraft registration number or other identification" physical value stored in the message */
inline static dba_var dba_msg_get_flight_reg_no_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_FLIGHT_REG_NO); return d == NULL ? NULL : d->var; }
/** Set the value of "Mobile station identifier" from a variable of type const char* */
inline static dba_err dba_msg_set_ident(dba_msg msg, const char* val, int conf)  { return dba_msg_setc(msg, DBA_VAR(0, 1, 11), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Mobile station identifier" from a ::dba_var */
inline static dba_err dba_msg_set_ident_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 11), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Mobile station identifier" physical value stored in the message */
inline static dba_var dba_msg_get_ident_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_IDENT); return d == NULL ? NULL : d->var; }
/** Set the value of "Direction of motion of moving observing platform" from a variable of type int */
inline static dba_err dba_msg_set_st_dir(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 12), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Direction of motion of moving observing platform" from a ::dba_var */
inline static dba_err dba_msg_set_st_dir_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 12), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Direction of motion of moving observing platform" physical value stored in the message */
inline static dba_var dba_msg_get_st_dir_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ST_DIR); return d == NULL ? NULL : d->var; }
/** Set the value of "Speed of motion of moving observing platform" from a variable of type int */
inline static dba_err dba_msg_set_st_speed(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 13), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Speed of motion of moving observing platform" from a ::dba_var */
inline static dba_err dba_msg_set_st_speed_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 13), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Speed of motion of moving observing platform" physical value stored in the message */
inline static dba_var dba_msg_get_st_speed_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ST_SPEED); return d == NULL ? NULL : d->var; }
/** Set the value of "Station or site name" from a variable of type const char* */
inline static dba_err dba_msg_set_st_name(dba_msg msg, const char* val, int conf)  { return dba_msg_setc(msg, DBA_VAR(0, 1, 19), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Station or site name" from a ::dba_var */
inline static dba_err dba_msg_set_st_name_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 19), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Station or site name" physical value stored in the message */
inline static dba_var dba_msg_get_st_name_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ST_NAME); return d == NULL ? NULL : d->var; }
/** Set the value of "ICAO location indicator" from a variable of type const char* */
inline static dba_err dba_msg_set_st_name_icao(dba_msg msg, const char* val, int conf)  { return dba_msg_setc(msg, DBA_VAR(0, 1, 63), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "ICAO location indicator" from a ::dba_var */
inline static dba_err dba_msg_set_st_name_icao_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 63), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "ICAO location indicator" physical value stored in the message */
inline static dba_var dba_msg_get_st_name_icao_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ST_NAME_ICAO); return d == NULL ? NULL : d->var; }
/** Set the value of "Report code" from a variable of type int */
inline static dba_err dba_msg_set_rep_cod(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 193), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Report code" from a ::dba_var */
inline static dba_err dba_msg_set_rep_cod_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 193), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Report code" physical value stored in the message */
inline static dba_var dba_msg_get_rep_cod_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_REP_COD); return d == NULL ? NULL : d->var; }
/** Set the value of "Air quality observing station local code" from a variable of type int */
inline static dba_err dba_msg_set_poll_lcode(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 212), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Air quality observing station local code" from a ::dba_var */
inline static dba_err dba_msg_set_poll_lcode_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 212), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Air quality observing station local code" physical value stored in the message */
inline static dba_var dba_msg_get_poll_lcode_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_LCODE); return d == NULL ? NULL : d->var; }
/** Set the value of "Airbase air quality observing station code" from a variable of type int */
inline static dba_err dba_msg_set_poll_scode(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 213), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Airbase air quality observing station code" from a ::dba_var */
inline static dba_err dba_msg_set_poll_scode_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 213), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Airbase air quality observing station code" physical value stored in the message */
inline static dba_var dba_msg_get_poll_scode_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_SCODE); return d == NULL ? NULL : d->var; }
/** Set the value of "GEMS air quality observing station code" from a variable of type int */
inline static dba_err dba_msg_set_poll_gemscode(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 214), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "GEMS air quality observing station code" from a ::dba_var */
inline static dba_err dba_msg_set_poll_gemscode_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 214), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "GEMS air quality observing station code" physical value stored in the message */
inline static dba_var dba_msg_get_poll_gemscode_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_GEMSCODE); return d == NULL ? NULL : d->var; }
/** Set the value of "Air quality observing station dominant emission source" from a variable of type int */
inline static dba_err dba_msg_set_poll_source(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 215), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Air quality observing station dominant emission source" from a ::dba_var */
inline static dba_err dba_msg_set_poll_source_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 215), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Air quality observing station dominant emission source" physical value stored in the message */
inline static dba_var dba_msg_get_poll_source_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_SOURCE); return d == NULL ? NULL : d->var; }
/** Set the value of "Air quality observing station area type" from a variable of type int */
inline static dba_err dba_msg_set_poll_atype(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 216), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Air quality observing station area type" from a ::dba_var */
inline static dba_err dba_msg_set_poll_atype_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 216), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Air quality observing station area type" physical value stored in the message */
inline static dba_var dba_msg_get_poll_atype_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_ATYPE); return d == NULL ? NULL : d->var; }
/** Set the value of "Air quality observing station terrain type" from a variable of type int */
inline static dba_err dba_msg_set_poll_ttype(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 1, 217), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Air quality observing station terrain type" from a ::dba_var */
inline static dba_err dba_msg_set_poll_ttype_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 1, 217), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Air quality observing station terrain type" physical value stored in the message */
inline static dba_var dba_msg_get_poll_ttype_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_POLL_TTYPE); return d == NULL ? NULL : d->var; }
/** Set the value of "Type of station" from a variable of type int */
inline static dba_err dba_msg_set_st_type(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Type of station" from a ::dba_var */
inline static dba_err dba_msg_set_st_type_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Type of station" physical value stored in the message */
inline static dba_var dba_msg_get_st_type_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ST_TYPE); return d == NULL ? NULL : d->var; }
/** Set the value of "Type of instrumentation for wind measurement" from a variable of type int */
inline static dba_err dba_msg_set_wind_inst(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 2), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Type of instrumentation for wind measurement" from a ::dba_var */
inline static dba_err dba_msg_set_wind_inst_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 2), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Type of instrumentation for wind measurement" physical value stored in the message */
inline static dba_var dba_msg_get_wind_inst_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_WIND_INST); return d == NULL ? NULL : d->var; }
/** Set the value of "Precision of temperature observation" from a variable of type double */
inline static dba_err dba_msg_set_temp_precision(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 2, 5), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Precision of temperature observation" from a ::dba_var */
inline static dba_err dba_msg_set_temp_precision_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 5), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Precision of temperature observation" physical value stored in the message */
inline static dba_var dba_msg_get_temp_precision_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TEMP_PRECISION); return d == NULL ? NULL : d->var; }
/** Set the value of "Radiosonde type" from a variable of type int */
inline static dba_err dba_msg_set_sonde_type(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 11), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Radiosonde type" from a ::dba_var */
inline static dba_err dba_msg_set_sonde_type_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 11), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Radiosonde type" physical value stored in the message */
inline static dba_var dba_msg_get_sonde_type_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_SONDE_TYPE); return d == NULL ? NULL : d->var; }
/** Set the value of "Radiosonde computational method" from a variable of type int */
inline static dba_err dba_msg_set_sonde_method(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 12), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Radiosonde computational method" from a ::dba_var */
inline static dba_err dba_msg_set_sonde_method_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 12), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Radiosonde computational method" physical value stored in the message */
inline static dba_var dba_msg_get_sonde_method_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_SONDE_METHOD); return d == NULL ? NULL : d->var; }
/** Set the value of "Aircraft navigational system" from a variable of type int */
inline static dba_err dba_msg_set_navsys(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 61), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Aircraft navigational system" from a ::dba_var */
inline static dba_err dba_msg_set_navsys_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 61), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Aircraft navigational system" physical value stored in the message */
inline static dba_var dba_msg_get_navsys_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_NAVSYS); return d == NULL ? NULL : d->var; }
/** Set the value of "Aircraft data relay system" from a variable of type int */
inline static dba_err dba_msg_set_data_relay(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 62), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Aircraft data relay system" from a ::dba_var */
inline static dba_err dba_msg_set_data_relay_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 62), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Aircraft data relay system" physical value stored in the message */
inline static dba_var dba_msg_get_data_relay_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_DATA_RELAY); return d == NULL ? NULL : d->var; }
/** Set the value of "Aircraft roll angle" from a variable of type double */
inline static dba_err dba_msg_set_flight_roll(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 2, 63), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Aircraft roll angle" from a ::dba_var */
inline static dba_err dba_msg_set_flight_roll_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 63), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Aircraft roll angle" physical value stored in the message */
inline static dba_var dba_msg_get_flight_roll_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_FLIGHT_ROLL); return d == NULL ? NULL : d->var; }
/** Set the value of "Original specification of latitude/longitude" from a variable of type int */
inline static dba_err dba_msg_set_latlon_spec(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 2, 70), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Original specification of latitude/longitude" from a ::dba_var */
inline static dba_err dba_msg_set_latlon_spec_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 2, 70), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Original specification of latitude/longitude" physical value stored in the message */
inline static dba_var dba_msg_get_latlon_spec_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_LATLON_SPEC); return d == NULL ? NULL : d->var; }
/** Set the value of "Year of the observation" from a variable of type int */
inline static dba_err dba_msg_set_year(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Year of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_year_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Year of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_year_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_YEAR); return d == NULL ? NULL : d->var; }
/** Set the value of "Month of the observation" from a variable of type int */
inline static dba_err dba_msg_set_month(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 2), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Month of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_month_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 2), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Month of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_month_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_MONTH); return d == NULL ? NULL : d->var; }
/** Set the value of "Day of the observation" from a variable of type int */
inline static dba_err dba_msg_set_day(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 3), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Day of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_day_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 3), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Day of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_day_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_DAY); return d == NULL ? NULL : d->var; }
/** Set the value of "Hour of the observation" from a variable of type int */
inline static dba_err dba_msg_set_hour(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 4), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Hour of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_hour_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 4), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Hour of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_hour_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_HOUR); return d == NULL ? NULL : d->var; }
/** Set the value of "Minute of the observation" from a variable of type int */
inline static dba_err dba_msg_set_minute(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 5), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Minute of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_minute_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 5), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Minute of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_minute_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_MINUTE); return d == NULL ? NULL : d->var; }
/** Set the value of "Second of the observation" from a variable of type int */
inline static dba_err dba_msg_set_second(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 4, 6), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Second of the observation" from a ::dba_var */
inline static dba_err dba_msg_set_second_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 4, 6), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Second of the observation" physical value stored in the message */
inline static dba_var dba_msg_get_second_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_SECOND); return d == NULL ? NULL : d->var; }
/** Set the value of "Latitude of the station" from a variable of type double */
inline static dba_err dba_msg_set_latitude(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 5, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Latitude of the station" from a ::dba_var */
inline static dba_err dba_msg_set_latitude_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 5, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Latitude of the station" physical value stored in the message */
inline static dba_var dba_msg_get_latitude_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_LATITUDE); return d == NULL ? NULL : d->var; }
/** Set the value of "Longiture of the station" from a variable of type double */
inline static dba_err dba_msg_set_longitude(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 6, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Longiture of the station" from a ::dba_var */
inline static dba_err dba_msg_set_longitude_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 6, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Longiture of the station" physical value stored in the message */
inline static dba_var dba_msg_get_longitude_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_LONGITUDE); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of station" from a variable of type int */
inline static dba_err dba_msg_set_height(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 7, 1), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Height of station" from a ::dba_var */
inline static dba_err dba_msg_set_height_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 7, 1), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Height of station" physical value stored in the message */
inline static dba_var dba_msg_get_height_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_HEIGHT); return d == NULL ? NULL : d->var; }
/** Set the value of "Isobaric surface" from a variable of type double */
inline static dba_err dba_msg_set_isobaric_surface(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 7, 4), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Isobaric surface" from a ::dba_var */
inline static dba_err dba_msg_set_isobaric_surface_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 7, 4), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Isobaric surface" physical value stored in the message */
inline static dba_var dba_msg_get_isobaric_surface_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_ISOBARIC_SURFACE); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of barometer above mean sea level" from a variable of type int */
inline static dba_err dba_msg_set_height_baro(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 7, 31), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Height of barometer above mean sea level" from a ::dba_var */
inline static dba_err dba_msg_set_height_baro_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 7, 31), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Height of barometer above mean sea level" physical value stored in the message */
inline static dba_var dba_msg_get_height_baro_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_HEIGHT_BARO); return d == NULL ? NULL : d->var; }
/** Set the value of "Phase of aircraft flight" from a variable of type int */
inline static dba_err dba_msg_set_flight_phase(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 8, 4), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Phase of aircraft flight" from a ::dba_var */
inline static dba_err dba_msg_set_flight_phase_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 8, 4), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Phase of aircraft flight" physical value stored in the message */
inline static dba_var dba_msg_get_flight_phase_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_FLIGHT_PHASE); return d == NULL ? NULL : d->var; }
/** Set the value of "Time significance" from a variable of type int */
inline static dba_err dba_msg_set_timesig(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 8, 21), val, conf, 257, 0, 0, 0, 0, 0, 0); }
/** Set the value of "Time significance" from a ::dba_var */
inline static dba_err dba_msg_set_timesig_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 8, 21), 257, 0, 0, 0, 0, 0, 0); }
/** Get the "Time significance" physical value stored in the message */
inline static dba_var dba_msg_get_timesig_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_TIMESIG); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (N1)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_n1(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 11), val, conf, 258, 1, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (N1)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_n1_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 11), 258, 1, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (N1)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_n1_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_N1); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (C1)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_c1(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 258, 1, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (C1)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_c1_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 258, 1, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (C1)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_c1_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_C1); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of base of cloud (H1)" from a variable of type double */
inline static dba_err dba_msg_set_cloud_h1(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 13), val, conf, 258, 1, 0, 0, 254, 0, 0); }
/** Set the value of "Height of base of cloud (H1)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_h1_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 13), 258, 1, 0, 0, 254, 0, 0); }
/** Get the "Height of base of cloud (H1)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_h1_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_H1); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (N2)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_n2(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 11), val, conf, 258, 2, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (N2)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_n2_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 11), 258, 2, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (N2)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_n2_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_N2); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (C2)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_c2(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 258, 2, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (C2)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_c2_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 258, 2, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (C2)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_c2_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_C2); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of base of cloud (H2)" from a variable of type double */
inline static dba_err dba_msg_set_cloud_h2(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 13), val, conf, 258, 2, 0, 0, 254, 0, 0); }
/** Set the value of "Height of base of cloud (H2)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_h2_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 13), 258, 2, 0, 0, 254, 0, 0); }
/** Get the "Height of base of cloud (H2)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_h2_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_H2); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (N3)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_n3(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 11), val, conf, 258, 3, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (N3)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_n3_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 11), 258, 3, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (N3)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_n3_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_N3); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (C3)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_c3(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 258, 3, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (C3)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_c3_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 258, 3, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (C3)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_c3_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_C3); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of base of cloud (H3)" from a variable of type double */
inline static dba_err dba_msg_set_cloud_h3(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 13), val, conf, 258, 3, 0, 0, 254, 0, 0); }
/** Set the value of "Height of base of cloud (H3)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_h3_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 13), 258, 3, 0, 0, 254, 0, 0); }
/** Get the "Height of base of cloud (H3)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_h3_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_H3); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (N4)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_n4(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 11), val, conf, 258, 4, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (N4)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_n4_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 11), 258, 4, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (N4)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_n4_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_N4); return d == NULL ? NULL : d->var; }
/** Set the value of "Cloud amount (C4)" from a variable of type int */
inline static dba_err dba_msg_set_cloud_c4(dba_msg msg, int val, int conf)  { return dba_msg_seti(msg, DBA_VAR(0, 20, 12), val, conf, 258, 4, 0, 0, 254, 0, 0); }
/** Set the value of "Cloud amount (C4)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_c4_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 12), 258, 4, 0, 0, 254, 0, 0); }
/** Get the "Cloud amount (C4)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_c4_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_C4); return d == NULL ? NULL : d->var; }
/** Set the value of "Height of base of cloud (H4)" from a variable of type double */
inline static dba_err dba_msg_set_cloud_h4(dba_msg msg, double val, int conf)  { return dba_msg_setd(msg, DBA_VAR(0, 20, 13), val, conf, 258, 4, 0, 0, 254, 0, 0); }
/** Set the value of "Height of base of cloud (H4)" from a ::dba_var */
inline static dba_err dba_msg_set_cloud_h4_var(dba_msg msg, dba_var val)  { return dba_msg_set(msg, val, DBA_VAR(0, 20, 13), 258, 4, 0, 0, 254, 0, 0); }
/** Get the "Height of base of cloud (H4)" physical value stored in the message */
inline static dba_var dba_msg_get_cloud_h4_var(dba_msg msg) 
{ dba_msg_datum d = dba_msg_find_by_id(msg, DBA_MSG_CLOUD_H4); return d == NULL ? NULL : d->var; }

#ifdef  __cplusplus
}
#endif

#endif

