/*
 * Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "LibRawInterface.h"

#include <QFileInfo>

#include <klocale.h>

#include "libraw/libraw.h"

#include "MessagesModel.h"
#include "ProcessingOptions.h"

#define rawParams _raw.imgdata.params

void LibRawInterface::prepareRaw( LibRaw& _raw, const ProcessingOptions& _processingOptions ) {
  
  rawParams.no_auto_bright = 1;
  rawParams.gamma_16bit = 0;
  rawParams.output_bps = 16;
  rawParams.four_color_rgb = 0;
  rawParams.use_fuji_rotate = 0;
  int highlight = _processingOptions.asInteger("HighlightsType");
  if( highlight <= 2)
  {
    rawParams.highlight = highlight;
  } else {
    rawParams.highlight = _processingOptions.asInteger("LevelValue");
  }
  switch(_processingOptions.asInteger("WhiteBalance")) {
    case 0:
      break;
    case 1:
      rawParams.use_camera_wb = 1;
      break;
    case 2:
      rawParams.use_auto_wb = 1;
      break;
    case 3:
    {
      double T = _processingOptions.asInteger("Temperature");
      double green = _processingOptions.asDouble("Tint");
      double RGB[3];
      double xD, yD, X, Y, Z;
      
      
      /* Here starts the code picked and adapted from ufraw (0.12.1)
          to convert Temperature + green multiplier to RGB multipliers
      */
      /* Convert between Temperature and RGB.
        * Base on information from http://www.brucelindbloom.com/
        * The fit for D-illuminant between 4000K and 12000K are from CIE
        * The generalization to 2000K < T < 4000K and the blackbody fits
        * are my own and should be taken with a grain of salt.
        */
      const double XYZ_to_RGB[3][3] = {
          { 3.24071,  -0.969258,  0.0556352 },
          {-1.53726,  1.87599,    -0.203996 },
          {-0.498571, 0.0415557,  1.05707 } };
      // Fit for CIE Daylight illuminant
      if (T <= 4000)
      {
          xD = 0.27475e9/(T*T*T) - 0.98598e6/(T*T) + 1.17444e3/T + 0.145986;
      }
      else if (T <= 7000)
      {
          xD = -4.6070e9/(T*T*T) + 2.9678e6/(T*T) + 0.09911e3/T + 0.244063;
      }
      else
      {
          xD = -2.0064e9/(T*T*T) + 1.9018e6/(T*T) + 0.24748e3/T + 0.237040;
      }

      yD     = -3*xD*xD + 2.87*xD - 0.275;
      X      = xD/yD;
      Y      = 1;
      Z      = (1-xD-yD)/yD;
      RGB[0] = X*XYZ_to_RGB[0][0] + Y*XYZ_to_RGB[1][0] + Z*XYZ_to_RGB[2][0];
      RGB[1] = X*XYZ_to_RGB[0][1] + Y*XYZ_to_RGB[1][1] + Z*XYZ_to_RGB[2][1];
      RGB[2] = X*XYZ_to_RGB[0][2] + Y*XYZ_to_RGB[1][2] + Z*XYZ_to_RGB[2][2];
      /* End of the code picked to ufraw
      */

      RGB[1] = RGB[1] / green;
      RGB[0] = 1.0 / RGB[0];
      RGB[1] = 1.0 / RGB[1];
      RGB[2] = 1.0 / RGB[2];

      // (-r) set Raw Color Balance Multipliers.
      rawParams.user_mul[0] = RGB[0];
      rawParams.user_mul[1] = RGB[1];
      rawParams.user_mul[2] = RGB[2];
      rawParams.user_mul[3] = RGB[1];
    } 
  }
  rawParams.user_qual = _processingOptions.asInteger("DemosaicQuality");
  if( _processingOptions.asBool("EnableNoiseReduction") ) {
    rawParams.threshold = _processingOptions.asInteger("ThresholdNoiseReduction");
  }
  if( _processingOptions.asBool("ChromaticAberrationCorrection") ) {
    rawParams.aber[0] = _processingOptions.asDouble("RedMultiplier");
    rawParams.aber[2] = _processingOptions.asDouble("GreenMultiplier");
  }
  rawParams.output_color = 0;
}

bool LibRawInterface::decodeRaw( LibRaw& _raw, RawImageInfoSP _rawImageInfo, QByteArray& _imageData, int& width, int& height, const QString& _who) {
  
  int ret = _raw.open_file(QFile::encodeName(_rawImageInfo->fileInfo().absoluteFilePath()));
  if (ret != LIBRAW_SUCCESS)
  {
    _rawImageInfo->setStatus( RawImageInfo::ERROR );
    MessagesModel::instance()->tellError(_who, i18n("Can't open file: %1.", _rawImageInfo->fileInfo().absoluteFilePath()));
    return false;
  }
  
  ret = _raw.unpack();  
  if(ret != LIBRAW_SUCCESS)
  {
    _rawImageInfo->setStatus( RawImageInfo::ERROR );
    MessagesModel::instance()->tellError(_who, i18n("Decoding of %1 failed.", _rawImageInfo->fileInfo().absoluteFilePath()));
    return false;
  }
  
  ret = _raw.dcraw_process();
  if (ret != LIBRAW_SUCCESS)
  {
    _rawImageInfo->setStatus( RawImageInfo::ERROR );
    MessagesModel::instance()->tellError(_who, i18n("Processing of %1 failed.", _rawImageInfo->fileInfo().absoluteFilePath()));
    return false;
  }
  libraw_processed_image_t *img = _raw.dcraw_make_mem_image(&ret);
  Q_ASSERT(img);  
  _imageData = QByteArray((const char*)img->data, (int)img->data_size);
  width  = img->width;
  height = img->height;
  free(img);
  return true;
}
