/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* utils.c */

#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "dialog.h"
#include "typedefs.h"
#include "utils.h"
#include "version.h"

/* Static functions declaration. */

/* Function definitions. */
/*
 * @utils_gint_to_gchar
 *
 *  Convert the value specified by the argument to the specified digits 
 *  chars. If failed, set the result NULL and return.
 *
 *  ex) 123   to 5chars => cut_spc=FALSE:pointer to the strings '  123\0'
 *                         cut_spc=TRUE :pointer to the strings '123\0'
 *      12345 to 3chars => pointer to the strings '345\0'
 *      12345 to 0chars => NULL
 *
 *  For using 'gint', the argument 'digit' must be under 9.
 *
 */
void utils_gint_to_gchar (gint     src     ,
                          gint     digit   ,
                          gboolean cut_spc ,
                          gchar    **result  ) {
  gboolean num_set;
  gint     i, num, pow_ret;

  /* '9' is the restriction for using the 'gint' type. */
  /*  cf. 9 digits = 100,000,000                       */
  if ((digit > 9) || (digit < 1)) { result = NULL; return; }

  *result = (gchar*)malloc(sizeof(gchar) * (digit + 1));
  if (*result == NULL) { return; }

  i = 0;
  num_set = FALSE;
  while (digit > 0) {
    num = 0;
    pow_ret = (gint)pow(10, digit - 1);
    while (src >= pow_ret) { 
      if (num < 9) { num++; } else { num = 0; } 
      src -= pow_ret; 
    }

    if ((num == 0) && (num_set == FALSE) && (digit > 1))
      (*result)[i] = ' ';
    else {
      (*result)[i] = '0' + num;
      num_set = TRUE;
    }

    digit--; 
    if ((cut_spc == FALSE)  ||
        ((cut_spc == TRUE)  && ((*result)[i] != ' '))) { i++; }
  }

  (*result)[i] = '\0';

  return;
}

/*
 * @utils_copy_file
 *
 *  Copy the 'src' to 'dest'.
 *  If fialed, return the value defined as AppErrorNum.
 *  -- arggument 'which' is return value and indicates which 
 *     file had errors.  0..src 1..dist
 *
 */
AppErrorNum utils_copy_file (gchar *dest  ,
                             gchar *src   ,
                             gint  *which   ) {
  int         fd1, fd2;
  struct stat stat_buf;
  mode_t      mode;
  int         nread, nwrite;
  char        buf[16384];
  AppErrorNum err_num;

  /* Initialize the local variables. */
  fd1     = fd2    = -1;
  nread   = nwrite = 0;
  err_num = APP_ERROR_NONE;

  /* Get the src file permission. */
  if ((stat(src, &stat_buf)) != 0) {
    *which = 0;
    return OPEN_FILE_ERROR; 
  }
  else {
    mode = stat_buf.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO);
  }

  fd1 = open(src, O_RDONLY, S_IROTH | S_IRGRP | S_IRUSR);
  if (fd1 == -1) {
    *which = 0;
    switch (errno) {
      case ENOENT:
        err_num = FILE_NOT_EXISTS;
        break;
      case EACCES:
      case EPERM :
        err_num = PERMISSION_DENIED;
        break;
      case ENAMETOOLONG:
        err_num = TOO_LONG_FILE_NAME;
        break;
      default:
        err_num = OPEN_FILE_ERROR;
        break;
    }
    *which = 0;
    return err_num;
  }

  fd2 = open(dest, O_WRONLY | O_CREAT, mode);
  if (fd2 == -1) {
    *which = 1;
    switch (errno) {
      case EACCES:
      case EPERM :
        err_num = PERMISSION_DENIED;
        break;
      case ENAMETOOLONG:
        err_num = TOO_LONG_FILE_NAME;
        break;
      case EROFS:
        err_num = READ_ONLY_FS;
        break;
      case ENOSPC:
        err_num = NO_SPACE_LEFT;
        break;
      default:
        err_num = OPEN_FILE_ERROR;
        break;
    }
    *which = 1;
    close(fd1);
    return err_num;
  }

  /* Copy src file to dest file. */
  while((nread = read(fd1, buf, 16384)) != 0) {
    if (nread == -1) {
      /* Read failed. */
      close (fd1); close (fd2); 
      return FILE_READ_ERROR;
    }

    nwrite = write(fd2, buf, nread);
    if (nwrite == -1) {
      /* Write failed. */
      switch (errno) {
        case ENOENT:
        case ENOSPC:
          err_num = NO_SPACE_LEFT;
          break;
        default:
          err_num = FILE_WRITE_ERROR;
          break;
      }
      close (fd1); close (fd2); 
      return err_num;
    }
    if (nwrite < nread) {
      switch (errno) {
        case ENOENT:
        case ENOSPC:
          err_num = NO_SPACE_LEFT;
          break;
        default:
          err_num = FILE_WRITE_ERROR;
          break;
      }
      close (fd1); close (fd2); 
      return err_num;
    }
  }

  close(fd1);
  close(fd2);

  return APP_ERROR_NONE;
}

/* Static function definitions. */


