/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#ifndef _CRL_HH
#define _CRL_HH

#include <stdio.h>

#include "Certificate.hh"
#include "OpenSSL++.hh"
#include "env.hh"
#include "Dn.hh"

namespace Cryptonit {

    class Certificate;
    class User;

    /** This class represents a CRL
     * a CRL is a list of certificate revoked
     */
    class Crl
    {
    private :
	/** pointer on the crl
	 */
	X509_CRL *crl;
    
    
    public:
	/**
	 * simple constructor
	 */
	Crl();


	/**
	 * constructor loading a crl in the X509 standard format
	 */
	Crl(X509_CRL *_crl);


	Crl(const Crl &c);
	/**
	 * simple destructor
	 */
	~Crl();
    

	/** load a crl from a file
	 *	@param filename :  a filename containing a crl in different format:
	 *                             - pem_format
	 *                             - der_format
	 *                             - pkcs7_der_format
	 *                             - pkcs7_pem_format
	 *
	 *	@param format :  format of the file
	 *	@return an error code (see error.hh)
	 */
	int load(const char *filename, FileFormat format);

	/** load a crl from a file
	 * @param filename :  a filename containing a crl
	 * @return an error code (SUCCESS if crl is successfully loaded)
	 */
	int load(const char *filename);


	/** load a crl from a bio
	 *	@param bio  : a bio pointing on a crl
	 *	@param format :  format of the bio (pem_format, der_format or pkcs7_pem_format))
	 *	@return an error code (see error.hh)
	 */
	int load(BIO *bio, FileFormat format);
    
	/** load a crl using an HTTP request
	 * @note Not yet implemented
	 * @return an error code (see error.hh)
	 */
	//int load(const char* url, FileFormat format, int port=80);


	/** load a crl using an LDAP request
	 * @note Not yet implemented
	 * @return an error code (see error.hh)
	 */
	int load();
    

	int save(const char* filename, FileFormat format=der_format);


	/** clean the things up
	 */
	void free();
    
	/** return a copy of the crl
	 * @return a X509_CRL*
	 */
	X509_CRL *getCRL();


	/** Display the attributes of this CRL
	 * @return a string containing the attributes.
	 */
	std::string getAttributes();
      
    
	/** Get the CRL version 
	 * @return the version number.
	 */
	int getVersion();
    
	/** get the issuer name of this CRL
	 * @return the dn of the issuer name
	 */
	dn getIssuerName();

	/** 
	 * The date of the last update
	 * 
	 * @return the date of the last update
	 */
	std::string getLastUpdate();

	/** 
	 * The date for the next update
	 * 
	 * 
	 * @return the date for the next update
	 */
	std::string getNextUpdate();

	/** 
	 * Give the extension of the loaded crl
	 * 
	 * @return 
	 */
	std::vector<std::string> getExtensions();


	std::vector<std::string> getRevokedSerials();

	/** 
	 * Indicate if the crl has been issued by a certificate
	 * 
	 * @param cert the certificate we want to know if it has issued the crl
	 * 
	 * @return true if certificate has issued the crl, else false
	 */
	bool isIssued(Certificate &cert);

	int isUpToDate();
	
	std::string getHash();

	/**  static method
	 * Check if a Crl (given by its URI) is outdated.
	 * @param uri: URI of a CRL
	 * @param user: User object (to get options, directory ...)
	 * @param c : Certificate requesting the CRL.
	 */
	static bool outDated(std::string uri , User *user , Certificate *c);
	
    };
}
#endif
