/* common.monoalphabetic.c - common monoalphabetic key manipulation routines
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"
#include "common.monoalphabetic.h"

/* Makes a display of two rows of letters, to illustrate a key */
GtkWidget *make_key_display(key *global_key_ptr, GtkWidget **global_key_labels) {
    int i;
    GtkWidget *table;
    GtkWidget *label;
    char buf[2];

    table = gtk_table_new(2, 26, TRUE);
    for (i=0; i < 26; i++) {
	sprintf(buf, "%c", i + 'A');
	label = gtk_label_new(buf);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 1.0);
        gtk_table_attach(GTK_TABLE(table), label, i, i + 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
        gtk_widget_show(label);
        
        sprintf(buf, "%c", (*global_key_ptr)[i + 'A']);
	label = gtk_label_new(buf);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.0);
        gtk_table_attach(GTK_TABLE(table), label, i, i + 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
        gtk_widget_show(label);
        global_key_labels[i] = label;
    }
    return table;
}

/* Match the user interface display with the contents of the global key */
void update_key_labels(key *global_key_ptr, GtkWidget **global_key_labels) {
    int i;
    char buf[2];
    for (i = 0; i < 26; i++) {
        if ((*global_key_ptr)[i + 'A'])
	    sprintf(buf, "%c", (*global_key_ptr)[i + 'A']);
	else
	    sprintf(buf, "*");
	gtk_label_set(GTK_LABEL(global_key_labels[i]), buf);
    }
}

char *apply_key_text(key *k, char *text) {
    char c, new_c;
    int i;
    char *newtext;
    int length = strlen(text);
    newtext = malloc((length + 1)  * sizeof(char));
    for (i = 0; i < length; i++) {
	c = text[i];
        if (!isalpha(c)) {
	    newtext[i] = c;
            continue;
        }
	c = toupper(c);
	new_c = (*k)[(int) c];
	if (new_c)
	    newtext[i] = new_c;
	else
	    newtext[i] = c;
    }
    newtext[i] = '\0';
    return newtext;
}


/* Return the identity permutation */
void key_identity(key *k) {
    char letter;
    for (letter = 'A'; letter <= 'Z'; letter++)
	(*k)[(int)letter] =  tolower(letter);
}

/* From - to */
void key_copy(key *k1, key *k2) {
    int i, j;
    for (i = 'A', j = 'A'; i <= 'Z'; (*k2)[i++] = (*k1)[j++]) ; 
}

void key_swap(key *k, int index1, int index2) {
    char temp = (*k)[index1];
    (*k)[index1] = (*k)[index2];
    (*k)[index2] = temp;
}

void key_shift_L(key *k) {
    int i;
    char wrapchar = (*k)['A'];
    for (i = 1; i < 26; i++)
	(*k)[i + 'A' - 1] = (*k)[i + 'A'];
    (*k)['Z'] = wrapchar;
}

void key_shift_R(key *k) {
    int i;
    char wrapchar = (*k)['Z'];
    for (i = 24; i >=0 ; i--)
	(*k)[i + 'A' + 1] = (*k)[i + 'A'];
    (*k)['A'] = wrapchar;
}

void key_reverse(key *k) {
    int i;
    for (i = 0; i < 13; i++)
	key_swap(k, i + 'A', 25 - i + 'A');
}

void key_invert(key *k) {
    int i;
    key k2;
    int to_upper = 'A' - 'a';
    /* initialise k2 */
    for (i = 0; i < 26; i++)
	k2[i + 'A'] = 0;
    /* reverse pairs across k into k2 */
    for (i = 0; i < 26; i++) 
        if ((*k)[i + 'A'])
	    k2[((*k)[i + 'A']) + to_upper] = i + 'a';
    /* copy k2 into k */
    for (i = 0; i < 26; i++)
	(*k)[i + 'A'] = k2[i + 'A'];
}

void key_clear (key *k) {
    int i;
    for (i = 0; i < 26; i++)
	(*k)[i + 'A'] = 0;
}

/* Make the key bijective */
void key_complete (key *k) {

    key to_count_list;
    int i;
    key from, to;
    int from_count = 0, to_count = 0;
    char fc, tc;
    
    /* Two inversions will make the key injective*/
    key_invert(k);
    key_invert(k);

    /* Now need to make it surjective */

    /* Blank to_count_list */
    for (i = 0; i < 26; i++)
	to_count_list[i] = 0;
    
    /* Construct from[], which lists all the unmapped 'from' letters */
    for (i = 0; i < 26; i++) {
	fc = i + 'A';
	tc = (*k)[(int)fc];
	if (tc)
	    to_count_list[(int)tc - 'a'] = TRUE;
	else {
	    from[from_count] = fc;
	    from_count++;
	}
    }
    /* From to_count_list, construct the unmapped 'to' letters */
    for (i = 0; i < 26; i ++) 
	if (!to_count_list[i]) {
	    to[to_count] = 'a' + i;
	    to_count++;
	}

    for (i = 0; i < to_count; i++)
	(*k)[(int)from[i]] = to[i];
    
}
