/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.math.stat.inference;

import org.apache.commons.math.MathException;

/**
 * An interface for Chi-Square tests.
 * <p>This interface handles only known distributions. If the distribution is
 * unknown and should be provided by a sample, then the {@link UnknownDistributionChiSquareTest
 * UnknownDistributionChiSquareTest} extended interface should be used instead.</p>
 * @version $Revision: 670469 $ $Date: 2008-06-23 04:01:38 -0400 (Mon, 23 Jun 2008) $ 
 */
public interface ChiSquareTest {
     
     /**
     * Computes the <a href="http://www.itl.nist.gov/div898/handbook/eda/section3/eda35f.htm">
     * Chi-Square statistic</a> comparing <code>observed</code> and <code>expected</code> 
     * frequency counts. 
     * <p>
     * This statistic can be used to perform a Chi-Square test evaluating the null hypothesis that
     *  the observed counts follow the expected distribution.</p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>Expected counts must all be positive.  
     * </li>
     * <li>Observed counts must all be >= 0.   
     * </li>
     * <li>The observed and expected arrays must have the same length and
     * their common length must be at least 2.  
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param observed array of observed frequency counts
     * @param expected array of expected frequency counts
     * @return chiSquare statistic
     * @throws IllegalArgumentException if preconditions are not met
     */
    double chiSquare(double[] expected, long[] observed) 
        throws IllegalArgumentException;
    
    /**
     * Returns the <i>observed significance level</i>, or <a href=
     * "http://www.cas.lancs.ac.uk/glossary_v1.1/hyptest.html#pvalue">
     * p-value</a>, associated with a 
     * <a href="http://www.itl.nist.gov/div898/handbook/eda/section3/eda35f.htm">
     * Chi-square goodness of fit test</a> comparing the <code>observed</code> 
     * frequency counts to those in the <code>expected</code> array.
     * <p>
     * The number returned is the smallest significance level at which one can reject 
     * the null hypothesis that the observed counts conform to the frequency distribution 
     * described by the expected counts.</p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>Expected counts must all be positive.  
     * </li>
     * <li>Observed counts must all be >= 0.   
     * </li>
     * <li>The observed and expected arrays must have the same length and
     * their common length must be at least 2.  
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param observed array of observed frequency counts
     * @param expected array of expected frequency counts
     * @return p-value
     * @throws IllegalArgumentException if preconditions are not met
     * @throws MathException if an error occurs computing the p-value
     */
    double chiSquareTest(double[] expected, long[] observed) 
        throws IllegalArgumentException, MathException;
    
    /**
     * Performs a <a href="http://www.itl.nist.gov/div898/handbook/eda/section3/eda35f.htm">
     * Chi-square goodness of fit test</a> evaluating the null hypothesis that the observed counts 
     * conform to the frequency distribution described by the expected counts, with 
     * significance level <code>alpha</code>.  Returns true iff the null hypothesis can be rejected
     * with 100 * (1 - alpha) percent confidence.
     * <p>
     * <strong>Example:</strong><br>
     * To test the hypothesis that <code>observed</code> follows 
     * <code>expected</code> at the 99% level, use </p><p>
     * <code>chiSquareTest(expected, observed, 0.01) </code></p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>Expected counts must all be positive.  
     * </li>
     * <li>Observed counts must all be >= 0.   
     * </li>
     * <li>The observed and expected arrays must have the same length and
     * their common length must be at least 2.  
     * <li> <code> 0 < alpha < 0.5 </code>
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param observed array of observed frequency counts
     * @param expected array of expected frequency counts
     * @param alpha significance level of the test
     * @return true iff null hypothesis can be rejected with confidence
     * 1 - alpha
     * @throws IllegalArgumentException if preconditions are not met
     * @throws MathException if an error occurs performing the test
     */
    boolean chiSquareTest(double[] expected, long[] observed, double alpha) 
        throws IllegalArgumentException, MathException;
    
    /**
     *  Computes the Chi-Square statistic associated with a 
     * <a href="http://www.itl.nist.gov/div898/handbook/prc/section4/prc45.htm">
     *  chi-square test of independence</a> based on the input <code>counts</code>
     *  array, viewed as a two-way table.  
     * <p>
     * The rows of the 2-way table are 
     * <code>count[0], ... , count[count.length - 1] </code></p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>All counts must be >= 0.  
     * </li>
     * <li>The count array must be rectangular (i.e. all count[i] subarrays
     *  must have the same length). 
     * </li>
     * <li>The 2-way table represented by <code>counts</code> must have at
     *  least 2 columns and at least 2 rows.
     * </li>
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param counts array representation of 2-way table
     * @return chiSquare statistic
     * @throws IllegalArgumentException if preconditions are not met
     */
    double chiSquare(long[][] counts) 
    throws IllegalArgumentException;
    
    /**
     * Returns the <i>observed significance level</i>, or <a href=
     * "http://www.cas.lancs.ac.uk/glossary_v1.1/hyptest.html#pvalue">
     * p-value</a>, associated with a 
     * <a href="http://www.itl.nist.gov/div898/handbook/prc/section4/prc45.htm">
     * chi-square test of independence</a> based on the input <code>counts</code>
     * array, viewed as a two-way table.  
     * <p>
     * The rows of the 2-way table are 
     * <code>count[0], ... , count[count.length - 1] </code></p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>All counts must be >= 0.  
     * </li>
     * <li>The count array must be rectangular (i.e. all count[i] subarrays must have the same length). 
     * </li>
     * <li>The 2-way table represented by <code>counts</code> must have at least 2 columns and
     *        at least 2 rows.
     * </li>
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param counts array representation of 2-way table
     * @return p-value
     * @throws IllegalArgumentException if preconditions are not met
     * @throws MathException if an error occurs computing the p-value
     */
    double chiSquareTest(long[][] counts) 
    throws IllegalArgumentException, MathException;
    
    /**
     * Performs a <a href="http://www.itl.nist.gov/div898/handbook/prc/section4/prc45.htm">
     * chi-square test of independence</a> evaluating the null hypothesis that the classifications 
     * represented by the counts in the columns of the input 2-way table are independent of the rows,
     * with significance level <code>alpha</code>.  Returns true iff the null hypothesis can be rejected
     * with 100 * (1 - alpha) percent confidence.
     * <p>
     * The rows of the 2-way table are 
     * <code>count[0], ... , count[count.length - 1] </code></p>
     * <p>
     * <strong>Example:</strong><br>
     * To test the null hypothesis that the counts in
     * <code>count[0], ... , count[count.length - 1] </code>
     *  all correspond to the same underlying probability distribution at the 99% level, use </p><p>
     * <code>chiSquareTest(counts, 0.01) </code></p>
     * <p>
     * <strong>Preconditions</strong>: <ul>
     * <li>All counts must be >= 0.  
     * </li>
     * <li>The count array must be rectangular (i.e. all count[i] subarrays must have the same length). 
     * </li>
     * <li>The 2-way table represented by <code>counts</code> must have at least 2 columns and
     *        at least 2 rows.
     * </li>
     * </li></ul></p><p>
     * If any of the preconditions are not met, an 
     * <code>IllegalArgumentException</code> is thrown.</p>
     *
     * @param counts array representation of 2-way table
     * @param alpha significance level of the test
     * @return true iff null hypothesis can be rejected with confidence
     * 1 - alpha
     * @throws IllegalArgumentException if preconditions are not met
     * @throws MathException if an error occurs performing the test
     */
    boolean chiSquareTest(long[][] counts, double alpha) 
    throws IllegalArgumentException, MathException;

}
