/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *
 * License: GPL-2+
 *
 * Initial version: 2008-09-02
 *
 * CNModel runner (main, cmdline parser)
 */


#include <unistd.h>
#include <list>

#include "libstilton/exprparser.hh"
#include "runner.hh"

#include "config.h"

using namespace std;
using namespace Stilton;

CLogFacility *Log;

CModel *Model;


SCNRunnerOptions Options;
const char *ScriptFileName = ""; // CNRUN_DEFAULT_SCRIPT;



#define CNRUN_CLPARSE_HELP_REQUEST	-1
#define CNRUN_CLPARSE_ERROR		-2
#define CNRUN_DUMP_PARAMS		-3


static int parse_options( int argc, char **argv,
			  list<SVariable>&);
static void usage( const char *argv0);


#define CNRUN_EARGS		-1
#define CNRUN_ESETUP		-2
#define CNRUN_ETRIALFAIL	-3


static void
LOG( int vrb, const char* fmt, ...)
{
	va_list ap;
	va_start (ap, fmt);

	Log->msgv( vrb, "CNrun", fmt, ap);
	va_end (ap);
}


int
main( int argc, char *argv[])
{
	int retval = 0;

	Log = new CLogFacility( nullptr, 0, Options.verbosely, 0, 0|STILTON_LOG_NOLOCK);

	list<SVariable> Variables;
	switch ( parse_options( argc, argv, Variables) ) {
	case CNRUN_CLPARSE_ERROR:
		LOG( -1, "Problem parsing command line or sanitising values; try -h for help");
		return CNRUN_EARGS;
	case CNRUN_CLPARSE_HELP_REQUEST:
		usage( argv[0]);
		return 0;
	}

	Log->log_threshold = Options.verbosely;

	printf( "CNRun ver. " VERSION " compiled " __DATE__ "  " __TIME__ "\n");

      // purely informational, requires no model
	if ( Options.list_units ) {
		cnmodel_dump_available_units();
		return 0;
	}

      // cd as requested
	char *pwd = nullptr;
	if ( Options.working_dir ) {
		pwd = getcwd( nullptr, 0);
		if ( chdir( Options.working_dir) ) {
			LOG( -1, "Failed to cd to \"%s\"", Options.working_dir);
			return CNRUN_EARGS;
		}
	}

	__cn_verbosely = Options.verbosely;
	__cn_default_unit_precision = Options.precision;

	interpreter_run( ScriptFileName, 0, Options.interp_howstrict,
			 true, true, Variables);

	delete Model;

	if ( pwd )
		if ( chdir( pwd) )
			;

	return retval;
}






static int
parse_options( int argc, char **argv, list<SVariable>& Variables)
{
	int	c;

	while ( (c = getopt( argc, argv, "e:t::L:E:g:k:UsC:n:D:v:h")) != -1 )
		switch ( c ) {

		case 'e':	ScriptFileName = optarg;					break;

		case 't':	switch ( optarg[0] ) {
				case 'T':	if ( sscanf( optarg+1, "%lg", &Options.integration_dt_max) != 1 ) {
							LOG( -1, "-tT takes a double");
							return CNRUN_CLPARSE_ERROR;
						}						break;
				case 't':	if ( sscanf( optarg+1, "%lg", &Options.integration_dt_min) != 1 ) {
							LOG( -1, "-tt takes a double");
							return CNRUN_CLPARSE_ERROR;
						}						break;
				case 'x':	if ( sscanf( optarg+1, "%lg", &Options.integration_dt_max_cap) != 1 ) {
							LOG( -1, "-tx takes a double");
							return CNRUN_CLPARSE_ERROR;
						}						break;
				default:	LOG( -1, "Unrecognised option modifier for -i");
							return CNRUN_CLPARSE_ERROR;
				}								break;

		case 'L':	if ( strchr( optarg, 'd') != nullptr )
					Options.listen_deferwrite = true;
				if ( strchr( optarg, '1') != nullptr )
					Options.listen_1varonly = true;
				if ( strchr( optarg, 'b') != nullptr )
					Options.listen_binary = true;
				if ( strchr( optarg, 'L') != nullptr )
					Options.log_dt = true;				break;

		case 'E':	if ( sscanf( optarg, "%g", &Options.listen_dt) != 1 ) {
					LOG( -1, "-E takes a double");
					return CNRUN_CLPARSE_ERROR;
				}						break;
		case 'g':	if ( sscanf( optarg, "%u", &Options.precision) != 1 ) {
					LOG( -1, "-g takes a short unsigned int");
					return CNRUN_CLPARSE_ERROR;
				}						break;

		case 'n':	if ( optarg && *optarg == 'c' )
					Options.dont_coalesce = true;		break;

		case 'k':	Options.log_spikers = true;
				switch ( *optarg ) {
				case '0':	Options.log_spikers_use_serial_id = true;	break;
				case 'l':	Options.log_spikers_use_serial_id = false;	break;
				case 'S':	if ( sscanf( optarg+1, "%g", &Options.spike_threshold) != 1 ) {
							LOG( -1, "-kS takes a double");
							return CNRUN_CLPARSE_ERROR;
						}
				default:	LOG( -1, "Expecting 0, l, or S<double> after -k");
						return CNRUN_CLPARSE_ERROR;
				}						break;

		case 'U':	Options.list_units = true;			break;


		case 's':	Options.sort_units = true;			break;


		case 'C':	Options.working_dir = optarg;			break;

		case 'D':	{
					double	unused;
					CExpression expr;
					if ( expr( optarg, unused, &Variables) ) {
						LOG( -1, "Malformed variable assignment with -D");
						return CNRUN_CLPARSE_ERROR;
					}
				}

		case 'v':	Options.verbosely = strtol( optarg, nullptr, 10);
				{	char *p;
					if ( (p = strchr( optarg, '%')) )
						Options.display_progress_percent = (*(p+1) != '-');
					if ( (p = strchr( optarg, 't')) )
						Options.display_progress_time    = (*(p+1) != '-');
				}
										break;
		case 'h':
			return CNRUN_CLPARSE_HELP_REQUEST;
		case '?':
		default:
			return CNRUN_CLPARSE_ERROR;
		}

	return 0;
}




static void
usage( const char *argv0)
{
	cout << "Usage: " << argv0 << "\n" <<
		" -e <script_fname>\tExecute script\n"
		" -D \t\t\tDump all unit types in the model and exit\n"
		" -C <dir>\t\tWork in dir\n"
		" -s \t\t\tSort units\n"
		"\n"
		" -L[d1Lx] \t\tLogging & listeners:\n"
		"    d \t\t\tdefer writing to disk until done rather than writing continuously\n"
		"    1\t\t\tonly log the first variable\n"
		"    x\t\t\twrite in native binary form rather than in ASCII\n"
		"    L\t\t\tlog integrator dt\n"
		" -E<double>\t\tListen at this interval (default " << Options.listen_dt << ";\n"
		"\t\t\t   set to 0 to listen every cycle)\n"
		"\n"
		" -kl \t\t\tWrite a model-wide log of spiking neurons, using labels\n"
		" -k0 \t\t\t... use unit id instead\n"
		" -kS<double>\t\tSpike detection threshold (default " << Options.spike_threshold << ")\n"
		"\n"
		" -e <uint>\t\tSet precision for all output (default " << Options.precision << ")\n"
		"\n"
		" -tT <double>\t\tdt_max (default " << Options.integration_dt_max << ")\n"
		" -tt <double>\t\tdt_min (default " << Options.integration_dt_min << ")\n"
		" -tx <double>\t\tCap dt by current dt value x this (default " << Options.integration_dt_max_cap << ")\n"
		"\n"
		" -D EXPR\t\tAny valid expression, will inject whatever variables get assigned in it\n"
		"\n"
		" -v <int>\t\tSet verbosity level (default " << Options.verbosely << ")\n"
		"\t\t\t Use a negative value to show the progress percentage only,\n"
		" -v[%[-]t[-]]\t\tDisplay (with -, suppress) progress percent and/or time\n"
		"\t\t\t indented on the line at 8 x (minus) this value.\n"
		"\n"
		" -U \t\t\tList available unit types with parameter names\n"
		"\t\t\t  and values, and exit\n"
		" -h \t\t\tDisplay this help\n"
		"\n";
}



// EOF
