# -*- coding: ascii -*-

###########################################################################
# clive, video extraction utility
# Copyright (C) 2007-2008 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

## The classes used to represent data transfer progress.

import sys
import time
import math

__all__ = ['Progress', 'ProgressVerify']


## The class used for file transfers
class Progress:

	## Called by urlgrabber when the transfer begins
	def start(self, filename, url, basename, length, text):
		self._length = length
		self._ttime = time.time()
		self._filename = filename
		self._url = url

	## Called by urlgrabber during transfer
	def update(self, read):
		if read == 0: return

		progress = self._percent(read, self._length)

		if progress > 0:
			text = 'transfer: %s / %s (%03d%%)' % (
				self._bytesmb(read),
				self._bytesmb(self._length),
				progress
			)

			elapsed = time.time() - self._ttime
			if elapsed == 0: elapsed=1

			rate = read / elapsed
			left = (self._length - read) / rate
			gmt = time.gmtime(left)

			left = '%02d:%02d:%02d' % (gmt[3], gmt[4], gmt[5])
			text += ' [%s] %12s/s' % (left, self._byteshuman(rate))
		else:
			text = 'transfer: %s' % self._bytesmb(read)

		self._write_bspaced(text)

	## Called by urlgrabber when the transfer ends
	def end(self, total):
		self._total = total
		self._write_stdout('\n')
	
	def _percent(self, i, j):
		if not i or not j or i == 0 or j == 0: return 0
		return int(float(i) / float(j) * 100)
	
	def _byteskb(self, bytes):
		return '%.0fKB' % (float(bytes) / 1024)

	def _bytesmb(self, bytes):
		return '%.3fMB' % (float(bytes) / (1024*1024))
	
	def _byteshuman(self, bytes):
		a = ['bytes','KB','MB','GB','TB']
		try:
			i = int(math.floor(math.log(bytes,1024)))
			b = round(float(bytes/math.pow(1024,i)),2)
		except OverflowError:
			i = 0
			b = 0.0
		return '%.2f%s' % (b,a[i])

	def _write_bspaced(self, text):
		bspaces = '\b' * len(text)
		self._write_stdout(bspaces + text)
	
	def _write_stdout(self, text):
		sys.stdout.write(text)
		sys.stdout.flush()


## The class used for video page (HTML) transfers
class ProgressVerify(Progress):

	## Constructor
	def __init__(self, index):
		self._index = index

	## Called by urlgrabber during transfer
	def update(self, read):
		if read == 0: return

		progress = self._percent(read, self._length)

		if progress > 0:
			text = 'status: %s %03d%% (%d of %d) [%s]' % (
				self._byteskb(read), progress,
				self._index[0]+1, self._index[1], self._url)
		else:
			text = 'status: %s (%d of %d) [%s]' % (
				self._byteskb(read), self._index[0]+1,
				self._index[1], self._url)

		text = text[:70]
		if len(text) == 70:
			text += '...]'

		self._write_bspaced(text)


