# -*- coding: ascii -*-

###########################################################################
# clive, video extraction utility
# Copyright (C) 2007-2008 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

## The classes for parsing runtime configuration and command line options

import os
import os.path
import sys
import random
import time
import gzip
import errno

from optparse import OptionParser, OptionGroup

import urlgrabber as _urlgrabber
import clive as _clive

try:
	import Tkinter
	_tkinter_avail = 1
except:
	_tkinter_avail = 0

try:
	import win32clipboard
	import win32con
	_mswin_clipboard_avail = 1
except ImportError:
	_mswin_clipboard_avail = 0

__all__ = ['Options']


## The class for parsing runtime configuration and command line options
class Options:
	_rcfile = 'config.py'
	_lastfile = 'last.log'
	_historyfile = 'history.log'
	_rcdir = '.clive'

	## Constructor
	def __init__(self):
		self._random_user_agent = 0

		self._version = '%s (%s; python/%s; urlgrabber/%s)' % (
			_clive.__version_str__,
			sys.platform,
			sys.version.split()[0],
			_urlgrabber.__version__
		)

		if sys.platform == 'win32':
			self._rcdir = self._rcdir.strip('.')

		self._full_rcdir = os.path.join(os.path.expanduser('~'), self._rcdir)

		try: # Make sure ~/.clive exists
			os.makedirs(self._full_rcdir)
		except OSError, e:
			if e.errno == errno.EEXIST: pass
			else: raise

		## Path to runtime configuration file
		self._rcfile = os.path.join(self._full_rcdir, self._rcfile)
		self._lastfile = os.path.join(self._full_rcdir, self._lastfile)
		self._historyfile = os.path.join(self._full_rcdir, self._historyfile)

		self._warnings = []

	## Parses runtime configuration and command line options
	def parse(self):
		self._parseopts()
		return (self.opts, self.args)

	## Prints out some of the program options to stdout
	def show(self, say):
		if self.opts.rename: if_exists = 'rename'
		elif self.opts.overwrite: if_exists = 'overwrite'
		else: if_exists = 'skip'

		onoff = ['off','on']

		say('%s\n[log:%s] [play:%s] [encode:%s] [throttle:%s] [exists:%s]' % (
			self._version,
			(onoff[self.opts.log]),
			self.opts.play,
			self.opts.re_encode,
			(['off','%.0fK/s' % (self.opts.throttle/1024)] \
				[self.opts.throttle > 0]),
			if_exists)
		)

		for w in self._warnings:
			say(w)

	## Writes runtime configuration file
	def write_config(self, opts):
		random_user_agent = opts._random_user_agent

		a = ['configure','history','_random_user_agent',
			'emit', 'check_updates','clear_logs', 'last']

		for i in a: del opts.__dict__[i] # Filter out the above keys

		# Mighty messy. #2

		for (name, value) in opts.__dict__.items(): # Filter out
			_del = 0
			if value == None:
				_del = 1
			else:
				value = str(value)
				if len(value) == 0:
					_del = 1
				elif value == 'off':
					if name != 'output_mask':
						_del = 1
			if not _del:
				if name == 'user_agent':
					if random_user_agent:
						_del = 1
				elif name == 'prefix':
					if value == os.getcwd():
						_del = 1
			if _del:
				del opts.__dict__[name]

		#for k in opts.__dict__.items():
		#	print k

		f = open(self._rcfile, 'w')
		f.write('# NOTE: --configure generated\n')
		f.write('{\n')
		for (name, value) in opts.__dict__.items():
			if name.startswith('_'): continue
			if name == 'throttle':
				value = value/1024
			if value == True: value = 1
			elif value == False: value = 0
			f.write('\t%s:%s,\n' % (repr(name), repr(value)))
		f.write('}\n\n')
		f.close()

	## Writes ~/.clive/last.log file
	def write_last_log(self, raw_urls):
		try:
			f = open(self._lastfile, 'w')
		except IOError, e:
			if e.errno == errno.ENOENT: pass
			else: raise

		for u in raw_urls:
			f.write(u + '\n')

		f.close()

	## Writes a new entry to ~/.clive/history.log file
	#
	# The log file is a CSV file and uses the line format:
	# "current-time","url","output-filename","file-length"
	def write_history(self, video):
		if os.path.exists(self._historyfile): # Archive existing log if:
			if os.path.getsize(self._historyfile) >= (1024*1024): # >= 1MB
				i = 1
				while 1:
					fn = '%s-%d.gz' % (self._historyfile, i)
					if not os.path.exists(fn):
						break
					i += 1
				f = open(self._historyfile, 'r')
				gzip.GzipFile(fn, 'w').write(f.read())
				f.close()
				os.remove(self._historyfile)
				
		(url, vurl, hlen, length, fn, reget, exists) = video

		open(self._historyfile, 'a').write(
			'"%s","%s","%s","%s"\n' % (
				time.strftime('%Y-%m-%dT%H:%M:%S'),
				url, os.path.basename(fn), hlen
			)
		)

	## Clears all logs from ~/.clive
	def clear_logs(self):
		for file in [self._lastfile, self._historyfile]:
			try:
				os.remove(file)
			except OSError, e:
				if e.errno == errno.ENOENT: pass
				else: raise

	# Non-public

	def _parseopts(self):
		try:
			self._config = eval(open(self._rcfile).read())
		except IOError, e:
			if e.errno == errno.ENOENT: pass
			else: raise
			self._config = None

		p = OptionParser(
			usage = '%prog [options] URLs',
			description = _clive.__doc__,
			version = self._version + '\n' + _clive.__notice__
		)

		gstdout = OptionGroup(p, 'Stdout Options', None)
		goutput = OptionGroup(p, 'Output Options', None)
		gconn = OptionGroup(p, 'Connection Options', None)
		gplayback = OptionGroup(p, 'Playback Options', None)
		gencode = OptionGroup(p, 'Re-encoding Options', None)
		gclip = OptionGroup(p, 'Clipboard Options', None)
		glog = OptionGroup(p, 'Logging Options', None)

		p.add_option_group(gstdout)
		p.add_option_group(goutput)
		p.add_option_group(gconn)
		p.add_option_group(gplayback)
		p.add_option_group(gencode)
		p.add_option_group(gclip)
		p.add_option_group(glog)

		# optgroup, varname, lopt, sopt, action, const
		# choices, desc, callback, type, metavar, default

		lookup = [
			(p, 'check_updates', '--check-updates', None, 'store_true',
				None, None, 'Check for updates', None, None, None, 0),
			(p, 'configure', '--configure', None, 'store_true',
				None, None,
				'Configure using a graphical user interface',
				None, None, None, 0),

			(gstdout, 'verbose', '--quiet', '-q', 'store_false', None, None,
				'Run quiet', None, None, None, self._getconf('verbose',1,1)),
			(gstdout, 'emit', '--emit', '-e', 'store_true', None, None,
				'List found video extraction URLs and exit',
				None, None, None, 0),

			(goutput, 'overwrite', '--overwrite', '-o', 'store_true', None,
				None, 'Overwrite already existing output files',
				None, None, None, self._getconf('overwrite',0,1)),
			(goutput, 'rename', '--rename', '-r', 'store_true', None, None,
				'Rename output file to be written if it exists already',
				None, None, None, self._getconf('rename',0,1)),
			(goutput, 'prefix', '--prefix', None, 'store', None, None,
				'Write extracted videos to PREFIX',
				None, None, 'PREFIX', self._getconf('prefix',os.getcwd())),
			(goutput, 'output_file', '--output-file', '-O', 'store',
				None, None, 'Write videos to FILE', None, None, 'FILE', None),
			(goutput, 'output_mask', '--output-mask', None, 'store',
				None, None, 'Set output filename re.sub mask (A-Za-z0-9)',
				None, None, 'MASK', self._getconf('output_mask','A-Za-z0-9')),
			(goutput, 'output_mask', '--disable-mask', '-M', 'store_const',
				'off', None, 'Disable output mask explicitly', None, None,
				None, self._getconf('output_mask','A-Za-z0-9')),
			(goutput, 'output_fmt', '--output-fmt', None, 'store',
				None, None, 'Set output filename format (%t.%e)',
				None, None, 'FMT', self._getconf('output_fmt','%t.%e')),

			(gconn, 'proxy', '--proxy', None, 'store', None, None,
				'Set PROXY as HTTP proxy', None, None, 'PROXY',
				self._getconf('proxy')),
			(gconn, 'user_agent', '--user-agent', None, 'store', None, None,
				'Use AGENT as HTTP user-agent string', None, None, 'AGENT',
				self._getconf('user_agent',self._gen_user_agent())),
			(gconn, 'throttle', '--throttle', None, 'store', None, None,
				'Set LIMIT as throttle limit for HTTP transfers (KB/s)',
				None, 'int', 'LIMIT', self._getconf('throttle',0)),
			(gconn, 'gzip', '--disable-gzip', '-Z', 'store_false',
				None, None,	'Disable gzip HTTP stream compression explicitly',
				None, None, None, self._getconf('gzip',1,1)),

			(gplayback, 'player', '--player', None, 'store', None, None,
				'Set PATH to player', None, None, 'PATH',
				self._getconf('player')),
			(gplayback, 'play', '--play', '-p', 'store',
				None, ['src','s','mpg','m','avi','a'],
				'Play video format (src, mpg, avi)', None, None, 'FMT',
				self._getconf('play')),
			(gplayback, 'play', '--disable-play', '-P', 'store_const',
				'off', None, 'Disable playback explicitly',
				None, None, None, self._getconf('play')),

			(gencode, 'ffmpeg', '--ffmpeg', None, 'store', None, None,
				'Set PATH to ffmpeg', None, None, 'PATH',
				self._getconf('ffmpeg')),
			(gencode, 're_encode', '--re-encode', None, 'store',
				None, ['mpg','m', 'avi','a'],
				'Re-encode video to format (mpg, avi)', None, None, 'FMT',
				self._getconf('re_encode')),
			(gencode, 're_encode', '--disable-encode', '-R', 'store_const',
				'off', None, 'Disable re-encoding explicitly',
				None, None, None, self._getconf('re_encode')),

			(gclip, 'xclip', '--xclip', None, 'store', None, None,
				'Set path to xclip', None, None, 'PATH',
				self._getconf('xclip')),
			(gclip, 'from_clipb', '--from-clipb', '-c',
				'store_true', None, None, 'Read URLs from the clipboard',
				None, None, None, self._getconf('from_clipb',0,1)),

			(glog, 'last', '--last', '-l', 'store_true', None, None,
				'Recall URLs from last run time', None, None, None, 0),
			(glog, 'log', '--disable-log', '-L', 'store_false',
				None, None, 'Disable all logging explicitly',
				None, None, None, self._getconf('log',1,1)),
			(glog, 'history', '--history', None, 'store_true', None, None,
				'Browse history using a graphical user interface', None,
				None, None, 0),
			(glog, 'clear_logs', '--clear-logs', None, 'store_true',
				None, None, 'Clear all logs', None, None, None,
				self._getconf('clear_logs',0,1)),
		]

		for (optgroup, varname, lopt, sopt, action, \
				const, choices, desc, callback, \
				_type, metavar, default) in lookup:
			optgroup.add_option(
				lopt, sopt,
				dest=varname,
				action=action,
				const=const,
				choices=choices,
				callback=callback,
				type=_type,
				help=desc,
				metavar=metavar,
				default=default
			)

		#print p.defaults
		(self.opts, self.args) = p.parse_args()
		#print self.opts

		# Exceptions / tweaks

		proxy = os.environ.get('http_proxy')
		if proxy and not self.opts.proxy:
			self.opts.proxy = proxy

		lookup = {
			None:'off',
			's':'src',
			'm':'mpg',
			'a':'avi'
		}

		for l in lookup:
			if self.opts.re_encode == l:
				self.opts.re_encode = lookup[l]
			if self.opts.play == l:
				self.opts.play = lookup[l]

		if not os.path.exists(self.opts.prefix):
			self.opts.prefix = os.getcwd()
			self._warnings.append('warn: prefix path does not exist; ' +
				'using current working dir instead')

		self.opts.throttle *= 1024 # kb -> bytes (urlgrabber)

		# NOTE: Mighty messy.

		if self.opts.player: # Ensure --player was supplied with '%i'
			if self.opts.player.find('%i') == -1 and \
				self.opts.play != 'off':
				self._warnings.append('warn: --player does not ' +
					'specify %i, playback disabled.')
				if self.opts.play != 'src':
					self._warnings.append('warn: re-encoding disabled due ' +
					'to the above.')
					self.opts.re_encode = 'off'
				self.opts.play = 'off'

		if self.opts.ffmpeg: # Ensure --ffmpeg was supplied with '%i' and '%o'
			lookup = ['%i','%o']
			for l in lookup:
				if self.opts.ffmpeg.find(l) == -1 and \
				self.opts.re_encode != 'off':
					self._warnings.append('warn: --ffmpeg does not ' +
						'specify %s, re-encoding disabled.' % l)
					self.opts.re_encode = 'off'
					if self.opts.play != 'off' and self.opts.play != 'src':
						self.opts.re_encode = 'off'

		if self.opts.play != 'off' or self.opts.re_encode != 'off':

			if not self.opts.player and self.opts.play != 'off':
				# Make sure --player was set if --play is being used
				self._warnings.append('warn: --player not set, ' +
					'playback disabled.')
				self.opts.play = 'off'

			if not self.opts.ffmpeg and (self.opts.play != 'src' or \
				self.opts.re_encode != 'off'):
				# Make sure --ffmpeg was set if --play is != 'src' 
				self._warnings.append(
					'warn: --ffmpeg not set, playback and re-encoding ' +
					'disabled.')
				self.opts.play = 'off'
				self.opts.re_encode = 'off'

			else: # Set --re-encode as --play if we get this far
				if self.opts.play != 'off' and self.opts.play != 'src':
					self.opts.re_encode = self.opts.play

		if self.opts.from_clipb: # Check clipboard options
			if sys.platform != 'win32':
				if not self.opts.xclip:
					self._warnings.append('warn: --xclip not used, ' +
						'disabling --from-clipboard')
					self.opts.from_clipb = 0
			else:
				if not _mswin_clipboard_avail:
					self._warnings.append('warn: win32clipboard module ' +
						'not found, disabling --from-clipboard')
					self.opts.from_clipb = 0

		# Set various flags

		setattr(self.opts, '_random_user_agent',
			self._random_user_agent) # --configure

		setattr(self.opts, '_tkinter_avail',
			_tkinter_avail) # --configure et al

		setattr(self.opts, '_mswin_clipboard_avail',
			_mswin_clipboard_avail)

		setattr(self.opts, '_full_rcdir',
			self._full_rcdir)

	def _getconf(self, key, default=None, return_bool=0, choices=None):
		if not self._config:
			if key == 'user_agent':
				self._random_user_agent = 1
			return default

		try:
			value = self._config[key]

			if return_bool:
				value = self._istrue(value)

			elif choices:
				try:
					choices.index(value)
				except ValueError:
					raise SystemExit('%s:%s: invalid choice: "%s", ' \
						'choose from %s' % (self._rcfile, key,
							value, repr(choices)))
			return value

		except KeyError:
			if key == 'user_agent':
				self._random_user_agent = 1
			return default

	def _istrue(self, value):
		if type(value) == str:
			v = value.lower()
			return (v == 'on' or v == 'yes' or v == '1' or v == 'true')
		else:
			return value == 1
	
	def _gen_user_agent(self, max_pairs=4, max_pair_len=2):
		words = ['clive', 'youtube', 'daily',
			'motion', 'stage', 'video',	'google', 'guba']

		a = []

		for w in words:
			i=0; l=max_pair_len
			while 1:
				a.append(w[i:l])
				i=l; l+=max_pair_len
				if i >= len(w):
					break

		t=''; l=-1

		for i in range(max_pairs):
			while 1:
				j = random.randint(0, len(a)-1)
				if j != l: break
			l=j; t+=a[j]

		ma = random.randint(0, 2)
		mi = random.randint(0, 9)
		rv = random.randint(0, 9)

		return '%s/%d.%d.%d' % (t, ma, mi, rv)


