#!/usr/bin/env python

########################################################################
# Chris's Lame Filebrowser 4 (Icon View)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################################################


import commands, os, os.path, shutil, string, sys, textwrap
import gobject, gtk, gtk.gdk, pygtk
import  fileinfo, file_monitor, file_mover, menu, mimetype, properties
import id3, mmpython

IMAGE_DIRECTORY = os.path.abspath(string.replace(sys.argv[0],"claw4.py","") + "Images")

class create:
	def __init__(self, tab):
		'''This function creates the layout and sets the events for the
		icon view'''
		self.tab = tab
		self.select = [0,0]
		self.over_icon = False
		self.tab.file_descriptor = -1
		
		# Create the actual layout widget and set the callbacks
		self.main_layout = gtk.Layout()
		self.main_layout.modify_bg(gtk.STATE_NORMAL, self.tab.bg_color)
		self.main_layout.show()
		
		self.main_layout.set_events(gtk.gdk.ALL_EVENTS_MASK)
		self.main_layout.connect("button-press-event", self.__layoutClicked__)
		self.main_layout.drag_dest_set(gtk.DEST_DEFAULT_ALL, [("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)
		self.main_layout.connect("drag-data-received", self.__dataReceived__)
		self.main_layout.connect("motion-notify-event", self.__mouseMoving__)
		self.main_layout.connect("button-release-event", self.__buttonReleased__)

		# Add the layout to the scrolled window in the notebook page
		self.tab.scroll.add_with_viewport(self.main_layout)

		# Populate the layout with icons
		self.__populate__(os.getcwd())

		# Only the icon view cares if the notebook resizes
		page_signal = self.tab.Notebook.main_notebook.connect("size-allocate", self.__notebookResized__)
		self.tab.Notebook.pages[tab] = page_signal


	############################ POPULATE LAYOUT ###############################

	def __populate__(self, new_location=None):
		'''This function places the icons in the layout'''
		self.__populating__ = True

		# This deletes any one file monitor on the old location
		file_monitor.delete(self.tab.file_descriptor)

		# Update the history if this isn't a "Back" move
		if new_location != None:
			if self.tab.history[-1] != new_location:
				self.tab.history.append(new_location)

			# If this is a new location, update the tab, the location and the tooltip and this history
			os.chdir(new_location)
			self.tab.changeTabLabel(new_location)
			current_location = new_location

		else:
			current_location = os.getcwd()
			
		self.current_location = current_location

		# Create a dialog to show we are loading the directory
		self.tab.__loadingDialog__(True)

		# Get and filter the files
		files = os.listdir(current_location)

		# Get an organized list of the directory contents and their mimetypes
		locations = self.tab.__organizeContents__(current_location)
		mimetypes = mimetype.getMimeTypes(current_location)

		# Create/Clear some dicts to hold the ievent boxes and labels
		self.icons = {}
		self.labels = {}

		# Create an icon for each location
		for location in locations:
			self.tab.__loadingDialog__("Creating Icons")	

			# Create the icon and label
			eventbox, label = self.__createIcon__(location, mimetypes[location])

			# Add the icon to the list
			self.icons[location] = eventbox
			self.labels[location] = label

		# Set the current locations list to global and place the icons in the layout
		self.locations = locations
		self.tab.__loadingDialog__("Placing Icons in Layout")
		self.__placeIcons__()

		# Create a file monitor for the current directory
		self.tab.file_descriptor = file_monitor.create(self, os.getcwd(), "icon")

		# Destroy the loading dialog
		self.tab.__loadingDialog__(False)

		self.counter = 0
		source_id = gobject.timeout_add(100, self.__createImages__)


	def __placeIcons__(self):
		'''This functions places the icons in the layout'''

		x, y, height =.3, 5, 100
		max_x = 0
		row_labels = []
		icons = self.icons
		labels = self.labels
		locations = self.locations
		self.new_layout_width = 1
		self.__populating__ = True

		# Clear the layout if needed
		if (self.main_layout.get_children()) > 1:
			for child in self.main_layout.get_children():
				self.main_layout.remove(child)

		# Get the current width of the layout
		layout_width = self.tab.scroll.allocation.width - 20
		if layout_width == 1:
			layout_width = 420

		for location in locations:
			self.main_layout.put(icons[location], int(x*100), int(y))
			self.main_layout.put(labels[location], int((x*100)-20), int(y+ 48))

			row_labels.append(labels[location])
			if self.main_layout.allocation.width == 1:
				labels[location].set_size_request(90, int(labels[location].size_request()[1]*1.5))			
			else:
				labels[location].set_size_request(90, int(labels[location].size_request()[1]))

			x+=1
			if (x*100) + 90 > layout_width:
				height = max([label.size_request()[1] for label in row_labels])
				max_x = x*100
				x=.3
				y += 60 + height
				row_labels = []

		if max_x == 0:
			max_x = x*100
		if self.tab.scroll.allocation.width > max_x + 120:
			max_x = self.tab.scroll.allocation.width
		self.new_layout_width =  int(max_x)
		self.main_layout.set_size_request(int(max_x), int(y+75)+height)

		# We are done populating
		self.__populating__ = False


	############################ CREATE IMAGES ###############################

	def __createImages__(self):
		'''This function is used to make image previews on the fly so as no to eat up loading time'''

		# First let's check to see if we are done
		if self.__populating__== True or self.counter >= len(self.icons):
			return False

		# Get the icon information
		icon = self.icons[self.locations[self.counter]]
		if icon.get_child() == None:
			return False
		mime_type = icon.get_child().get_name()
		catagory, mime = mime_type.split("/")

		image_types = ["x-xpixmap","x-ms-bmp","bmp","gif", "jpeg", "png", "svg"] 		

		# If we don't want a preview, then don't change a thing
		if catagory != "image" or self.tab.inline_picture_preview == False:
			pass

		elif catagory == "image" and mime in image_types:
			image = icon.get_child()
			new_image = gtk.Image()
			location = icon.get_name()

			try:
				if os.path.getsize(location)/1024 < self.tab.image_max_size*1024:
					pixbuf = gtk.gdk.pixbuf_new_from_file(location)

					width = pixbuf.get_width()
					height = pixbuf.get_height()

					while width > 48 or height > 48:
						width = width*.90
						height = height*.90
					pixbuf = pixbuf.scale_simple(int(width), int(height), gtk.gdk.INTERP_HYPER)
					new_image.set_from_pixbuf(pixbuf)
					icon.remove(image)
					icon.add(new_image)
					new_image.show()
					new_image.set_name(mime_type)
				else:
					# If the pic is too big, then do nothing
					print "INFORMATION:  This image is bigger than the max preview size"

			except:
				self.tab.setInfoBarText("Failed To Load " + os.path.basename(location), "ERROR")
				# If the pic fails to load, then do nothing

		self.counter += 1
		return True


	############################ LAYOUT CALLBACKS ###############################
		
	def __layoutClicked__(self, main_layout, event):
		'''This function handles button presses on the layout'''

		# A left click deselects all the selected icons and then checks out making a drag box
		if event.button == 1 and self.over_icon == False:
			[self.icons[location].set_state(gtk.STATE_NORMAL) for location in self.icons if self.icons[location].state == gtk.STATE_SELECTED]

			x, y = self.main_layout.get_pointer()
			print "STARTING:", x, y
			self.select = [x,y]
			
		# A right click opens a menu
		elif event.button == 3  and self.over_icon == False:
			self.__layoutMenu__(main_layout, event)

	def __mouseMoving__(self, main_layout, event):
		'''This function handles resizing the selection rectangle'''
		if self.select != [0,0]:

			x, y = main_layout.get_pointer()
			start_x, start_y = self.select[1], self.select[0]

			if x >= start_x:
				width = x - start_x
			else:
				width = start_x - x

			if y >= start_y:
				height = y - start_y
			else:
				height = start_y - y

			size = width, height

	def __buttonReleased__(self, main_layout, event):
		'''This function gets rid of the layout selection rectangle'''
		print "DELETED", self.main_layout.get_pointer()	
		self.select = [0,0]


	def __layoutMenu__(self, main_layout, event):
		'''This function handles creating the layout menu for layout right clicks'''
		menu.layout(self, event)


	################################ ICON CALLBACKS ###############################

	def __iconClicked__(self, eventbox, event):
		'''This function is the callback for a clicked icon,'''

		# Left click
		if (event.button == 1 and not event.type == gtk.gdk._2BUTTON_PRESS) and self.tab.single_click == False:
			if eventbox.state == gtk.STATE_NORMAL:
				eventbox.set_state(gtk.STATE_SELECTED)
			else:
				eventbox.set_state(gtk.STATE_NORMAL)	
		# Right click
		elif event.button == 3:
			self.__rightClickMenu__(string.split(eventbox.get_child().get_name(), "/"), eventbox, event)
		# Double-click
		elif event.button == 1:
			if os.path.isdir(eventbox.get_name()):
				if os.access(eventbox.get_name(), os.R_OK):
					self.__populate__(eventbox.get_name())
				else:
					self.tab.setInfoBarText("You Don't Have Permission To Enter " + os.path.basename(eventbox.get_name()) +  "ERROR")
			else:
				menu.__menuClick__(None, self, self.tab, "Open", eventbox.get_name(), string.split(eventbox.get_child().get_name(), "/"))


	def __stateChanged__(self, eventbox, state):
		'''This functions handles changes in an icon's state'''

		if state == gtk.STATE_NORMAL:
			image = eventbox.get_child()
			pixbuf = eventbox.get_child().get_pixbuf().copy()
			pixbuf.saturate_and_pixelate(pixbuf, 0.1,False)
			image.set_from_pixbuf(pixbuf)
		elif state == gtk.STATE_SELECTED:

			image = eventbox.get_child()
			location = eventbox.get_name()
			catagory, mime = string.split(image.get_name(), "/")
			image_pic = "plain.png"
			if catagory == "image" and mime in ["x-xpixmap","x-ms-bmp","bmp","gif", "jpeg", "png", "svg"]:
				try:
					if os.path.getsize(location)/1024 < self.tab.image_max_size*1024:
						pixbuf = gtk.gdk.pixbuf_new_from_file(location)

						width = pixbuf.get_width()
						height = pixbuf.get_height()

						while width > 48 or height > 48:
							width = width*.90
							height = height*.90
						pixbuf = pixbuf.scale_simple(int(width), int(height), gtk.gdk.INTERP_HYPER)
					else:
						image_pic = "image.png"
						pixbuf = gtk.gdk.pixbuf_new_from_file(self.tab.pixbuf_list[image_pic])

					image.set_from_pixbuf(pixbuf)
				except:
					self.tab.setInfoBarText("Making Image Preview Failed", "ERROR")
					image_pic = "image.png"
					image.set_from_pixbuf(self.tab.pixbuf_list[image_pic])

			else:
				image_pic = mime + ".png"
				if image_pic in self.tab.pixbuf_list:
					image.set_from_pixbuf(self.tab.pixbuf_list[image_pic])
				else:
					image.set_from_pixbuf(self.tab.pixbuf_list["plain.png"])


	def __rightClickMenu__(self, mime, eventbox, event):
		'''This functions creates the right click menu for the right click
		on an icon'''

		menu.icon(self, self.tab, event, eventbox.get_name(), mime)


	######################### DRAG AND DROP CALLBACKS ##########################

	def __dataGet__(self, eventbox, drag_context, selection_data, drag_id, timestamp):
		'''This function handles getting the source widget's information'''
		selection_data.set(selection_data.target, 8, self.__getSelectedFiles__())


	def __dataReceived__(self, eventbox, drag_context, x, y, selection_data, drag_id, timestamp):
		'''This function handles the information sent to the destination widget'''

		# Get the selected files
		files = selection_data.data
		if eventbox == self.main_layout:
			drop_location = os.getcwd()
		else:
			drop_location = eventbox.get_name()
		
		# First check to see if valid files have been moved and we have permission and a valid location
		if not os.path.isdir(drop_location):
			self.tab.setInfoBarText("You Can Only Drop Files Into A Directory","ERROR")		
		elif not os.access(drop_location, os.W_OK):
			self.tab.setInfoBarText("You Don't Have Permission To Drop To " + os.path.basename(drop_location),"ERROR")
		elif files == [] or files == [""]:
			self.tab.setInfoBarText("Please Select Files To Drag", "ERROR")
		else:
			menu.drop(self.tab, timestamp, files, drop_location)

	def __getSelectedFiles__(self):
		'''This function get the locations of all the selected eventboxes'''
		
		# Make a string conataing the names of the selected icon locations
		selection_text = ""
		for location in self.icons:
			if self.icons[location].state == gtk.STATE_SELECTED:
				selection_text += location + "\t"

		return selection_text	

	def __dragEnter__(self, eventbox, drag_context, x, y, timestamp):
		'''This function is the callback for when a drag enters a drop target'''
		self.over_icon= True


	def __dragLeave__(self, eventbox, drag_context, timestamp):	
		'''This function is the callback for when a drag leaves a drop target'''	
		self.over_icon=False

	############################# MOUSE CALLBACKS ##############################


	def __mouseEnter__(self, eventbox, event):
		'''This functions creates the pop previews when you are over an icon and
		also handles audio previews'''

		# This variable let's the layout know if the mouse is over an icons
		self.over_icon = True
		
		# Get some vital information
		location = eventbox.get_name()
		mime = string.split(eventbox.get_child().get_name(), "/")

		#print "THIS FILE IS AT:", location
		#print "MIME TYPE:", mime

		# Play an audio preview if requested
		if self.tab.play_audio_previews == True and mime[0] == "audio" and mime[1] in ["mp3","mpeg", "ogg", "x-wav", "basic"]:
		
			# Set the types and players
			types  = ["mp3"      , "mpeg"     , "ogg"	    ,"x-wav" ,"basic"]
			player = ["mpg321","mpg321", "ogg123" , "play"   , "play"] 
			player_commands = ["mpg321 -q","mpg321 -q", "ogg123" , "play"   , "play"]  

			# Find the right player for the type
			number = types.index(mime[1])
			if commands.getstatusoutput("which " + player[number])[0] != 0:
				self.tab.setInfoBarText(player[number] +" Is Not Installed", "ERROR")
				self.audio_preview_playing = False
			else:
				try:
					os.popen2(player_commands[number] + " " + __returnClean__(location))

					# We set this variable to the player so that it is easy to kill (neat thinking, eh?)
					self.audio_preview_playing = player_commands[number]

				# If a valid type can't be played, it's probabaly because the player is not installed
				except:
					self.tab.setInfoBarText(player[number]+" Won't Play "+location, "ERROR")
					self.audio_preview_playing = False
	
		else:
			self.audio_preview_playing = False


		# if this is an mp3 make a popup window with the id3 tag information
		if mime == ["audio", "mp3"] or mime== ["audio","mpeg"]:

			# Get the file information
				
			'''  When this works again, I'll use it.

			# Available types are 'title', 'artist', 'composer', 'album', 'length', 'samplerate'	, 
			# 'size', 'release', 'date', 'copyright', 'genre', 'text', 'language', 'link', 'picture',
			# 'media_type','track_number'
				
			# Here we use the python meta data (for mp3's, gif's, etc.)
			medium = mmpython.parse(location)				
			if hasattr(medium, 'title')  and medium.title not in [None,'']:
				label +=  "<b>" + string.capwords(medium.title) + "</b>"
			if hasattr(medium, 'artist')  and medium.artist not in [None,'']:
				label += "\nBy <b>" + string.capwords(medium.artist) + "</b>"
			if hasattr(medium, 'composer')  and medium.composer not in [None,'']:
				label += "\nWritten by <b>" +  string.capwords(medium.composer) + "</b>"
			if hasattr(medium, 'album')  and medium.album not in [None,'']:
				label += "\nOff of <b>"+  string.capwords(medium.album) + "</b>"
			if hasattr(medium, 'copyright')  and medium.copyright  not in [None,'']:
				label += "\nCopyright " +  string.capwords(medium.copyright)		
			if hasattr(medium, 'release')  and medium.release  not in [None,'']:
				label += "\nReleased " + str(medium.release)
			if hasattr(medium, 'date') and medium.date  not in [None,'']:
				label += "\nReleased " + str(medium.date)
			if hasattr(medium, 'samplerate')  and medium.samplerate  not in [None,'']:
				label += "\n" + str(medium.samplerate) + " bit"
			'''

			# Get the information from the id3 tag
			info = {}
			try:
				idtag = id3.ID3v2(location)
				for frame in id3tag.frames:
					# TIT2 = title, TPE1 = performer, TALB = album, COMM = comments, TRCK = Track,
					# TSSE = encoder used, TPUB = publisher, TCOM = composer, TLEN = length
					# TPE2 = band, PRIV = private, TCON = content type / genre				

					if frame.id == "TIT2":
						info["title"] = unicode(frame).encode('utf-8')
					elif frame.id == "TPE1":
						info["performer"] = unicode(frame).encode('utf-8')
					elif frame.id == "TALB":
						info["album"]  = unicode(frame).encode('utf-8')
					elif frame.id == "TYER":
						info["year"] = unicode(frame).encode('utf-8')
					elif frame.id == "TPUB":
						info["publisher"] = unicode(frame).encode('utf-8')
					elif frame.id == "TCOM":
						info["composer"] = unicode(frame).encode('utf-8')
					elif frame.id == "TLEN":
						info["length"] = unicode(frame).encode('utf-8')
					elif frame.id == "TPE2":
						info["band"] = unicode(frame).encode('utf-8')
					elif frame.id == "PRIV":
						pass
					elif frame.id == "TCON":
						info["content"] = unicode(frame).encode('utf-8')					
					else:
						pass
			except:
				id3tag = id3.ID3v1(location)
				
				# Possilble information: artist, album, year, command, genre, track
				if id3tag.title != '':
					info['title'] = id3tag.title
				if id3tag.artist != '':
					info['performer'] = id3tag.artist					
				if id3tag.album != '':
					info['album'] = id3tag.album
				if id3tag.year != '':
					info['year'] = id3tag.year

			# Make the label
			label = ""
			if "title" in info:
				label += " <b>" + info["title"] + "</b> \n"
			if "performer" in info:
				label += " <b>" + info["performer"] + "</b> \n"					
			if "album" in info:
				label += " <b>\"" + info["album"] + "\"</b> \n"
			if "publisher" in info:
				label += " " + info["publisher"] + " "
			if "year" in info:
				label += " " + info["year"] + " \n"													
			if "length" in info:
				time =  str(int(info["length"])/60)
				label += " " + time[0] +":" + time[1:3]+ " \n"	

			# Clean up the label
			bad_chars 	= ["\'","\"","&"]
			good_chars	= ['','','+']
			for x in range(len(bad_chars)):
				if bad_chars[x] in label:
					label = string.replace(label, bad_chars[x], good_chars[x])
			if label == "":
				label = "No Information Available"
				
			# Create a label for the tag info
			tag = gtk.Label()
			tag.set_use_markup(True)
			tag.set_markup(label)
			tag.show()

			# Make a popup window to hold the label
			tag_window = gtk.Window(gtk.WINDOW_POPUP)
			#tag_window.modify_bg(gtk.gdk.Color(30000, 35000,65000,0))

			# Place the tag at the mouse location
			x, y =self.tab.__getMousePosition__()
			tag_window.move(x +40, y)

			# Add the icon to the window
			tag_window.add(tag)

			#  Show and resize the window
			tag_window.show()
			tag_window.resize(tag_window.allocation.width, tag_window.allocation.height)

			# Set this to the window
			self.tag_preview = tag_window			
		
		else:
			self.tag_preview = None

		# If this is an image file, show a popup preview
		if self.tab.show_picture_preview == True and mime[0] == "image":

			# If we have permission to read it, then make a pixbuf out of it (we don't have svg or gimp capabilities, yet)
			if os.access(location, os.R_OK) and mime[1] not in ["x-xcf", "gimp", "x-3ds"]:

				try:
					medium = mmpython.parse(location)
					label = ""
					
					if hasattr(medium, 'title') and medium.title not in [None,'']:
						label += "<b>" + str(medium.title) + "</b>\n"
					if hasattr(medium, 'artist') and medium.artist not in [None,'']:
						label += "Artist: " + str(medium.artist) + "\n"
					if os.path.getsize(location):
						size = os.path.getsize(location)
						if int(size) > 1048576:
							size = str(size/1048576) + " M \n"
						elif int(size) > 1024:
							size = str(size/1024) + " k\n"
						else:
							size = str(size) + " bytes\n"
						label += "Size: " + size
					if hasattr(medium, 'caption') and medium.caption not in [None,'']:
						label += "Caption: " + str(medium.caption)  + "\n"
					if hasattr(medium, 'comment') and medium.comment not in [None,'']:
						label += "Comment: " + str(medium.comment)  + "\n"
					if hasattr(medium, 'date') and medium.date not in [None,'']:
						label += "Created: " + str(medium.date)  + "\n"
					#if hasattr(medium, 'url') and medium.url not in [None,'']:
					#	label += "url: " + os.path.basename(medium.url)

					# Clean up the label
					bad_chars 	= ["\'","\"","&"]
					good_chars	= ['','','+']
					for x in range(len(bad_chars)):
						if bad_chars[x] in label:
							label = string.replace(label, bad_chars[x], good_chars[x])

					preview_label = gtk.Label(string.strip(label))

					# We create a pixbuf from the picture (make a copy in memory)
					preview_pixbuf = gtk.gdk.pixbuf_new_from_file(location)

					# We get the image's original size
					width = preview_pixbuf.get_width()
					height = preview_pixbuf.get_height()
					wholeWidth = width
					wholeHeight = height

					# Get a scaleable size
					x=1
					while width > 300 or height > 300:
						width = (width*.90)
						height = (height*.90)
						x -= .05

					# Set the height and width of the icon pixbuf
					preview_pixbuf = preview_pixbuf.scale_simple(int(width), int(height), gtk.gdk.INTERP_BILINEAR)

					# Then we add it to the image object
					preview = gtk.Image()
					preview.set_from_pixbuf(preview_pixbuf)
					
					# Make a popup window to hold the image
					preview_window = gtk.Window(gtk.WINDOW_POPUP)

					# Place the image preview at the mouse location
					x,y = self.tab.__getMousePosition__()
					preview_window.move(x+20, y)
					
					# Resize it to fit the image
					preview_window.resize(int(width)+5, int(height)+5)

					# Add the preview and label to the window
					preview_box = gtk.VBox(gtk.FALSE)
					preview_window.add(preview_box)
					preview_box.pack_start(preview)
					preview_box.pack_start(preview_label)					

					# Set this to the window, similiar to what we did with the audio preview
					self.pic_preview = preview_window
					preview_window.show_all()
					
				# Here too, anything but an workingpicpreview = None					
				except:
					self.tab.setInfoBarText("Cannot Create Picture Preview for " + location, "ERROR")
					self.pic_preview = None
			else:
				self.pic_preview = None
		else:
			self.pic_preview = None

			# If this isn't a picture, then put a tool tip on it that shows file information
			if os.access(location, os.R_OK):
				if gtk.tooltips_data_get(eventbox) == None:
					try:
						if os.path.isdir(os.path.realpath(location)):
							file_size, file_number, permissions, owner = fileinfo.getStats(os.path.realpath(location))
							tip_text = str(owner) + "\n" + str(file_number) + " files\n" +str(file_size) + "\n" + str(permissions)
							self.tab.tool_tips.set_tip(eventbox, tip_text)
						else:
							file_size, permissions, owner = fileinfo.getStats(os.path.realpath(location))
							tip_text = str(owner) + "\n" + str(file_size) + "\n" + str(permissions)
							self.tab.tool_tips.set_tip(eventbox, tip_text)	
					except:
						self.tab.setInfoBarText("Can't Create Tool Tip For " + os.path.basename(location), "ERROR")


	def __mouseLeave__(self, eventbox, event):
		'''The function handles destroying popup information windows,
		resetting the variables and stopping audio previews'''

		# This variable let's the layout know if the mouse is over an icons
		self.over_icon = False

		# If a player is playing music then stop it
		if self.audio_preview_playing != False:
			os.system("killall " + self.audio_preview_playing)
			self.audio_preview_playing = False

		# If a audio tag is showing, destroy it
		if self.tag_preview != None:
			self.tag_preview.destroy()
			self.tag_preview = None

		# If a picture preview is showing, destroy it
		if self.pic_preview != None:
			self.pic_preview.destroy()
			self.pic_preview = None


	############################# FUNCTIONS ##############################

	def __createIcon__(self, location, mime_type):
		'''This function creates an icon and label and returns them.  
		There is another function which handles the actual image previews'''

		image_types = ["x-xpixmap","x-ms-bmp","bmp","gif", "jpeg", "png", "svg"] 

		image = gtk.Image()
		image.set_name(mime_type[0] +"/" + mime_type[1])
		image.show()

		# Filter out hidden files for now
		if (self.tab.show_hidden == True) or (self.tab.show_hidden == False and string.find(location, ".", 0, 1) == -1):

			pixbuf = self.tab.__getPixbuf__(location, None, mime_type)
			image.set_from_pixbuf(pixbuf)

			text = os.path.basename(location)
			text = string.replace(text, "&", "+")
			label = gtk.Label(text)
			label.modify_font(self.tab.font)
			label.set_selectable(True)
			label.modify_fg(gtk.STATE_NORMAL, self.tab.font_color)
			label.set_justify(gtk.JUSTIFY_CENTER)
			label.set_line_wrap(True)
			label.show()

			wrap = 25
			label.set_text(text)
			while label.get_layout().get_pixel_size()[0] >= (80 - ((self.tab.font.get_size()/1000)-11)*5):
				wrap -= 1
				text = textwrap.fill(os.path.basename(location), wrap)
				label.set_text(text)


			# Create the eventbox to capture events for the icon
			eventbox = gtk.EventBox()
			eventbox.add(image)
			eventbox.set_name(os.path.realpath(location))
			eventbox.modify_bg(gtk.STATE_NORMAL, self.tab.bg_color)		
			eventbox.show()

			# Set the events to be captured
			eventbox.connect("button-press-event", self.__iconClicked__)
			eventbox.connect("enter-notify-event", self.__mouseEnter__)
			eventbox.connect("leave-notify-event", self.__mouseLeave__)
			eventbox.connect("state-changed", self.__stateChanged__)

			# Initialize drag and drop
			eventbox.drag_source_set(gtk.gdk.BUTTON1_MASK, [("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)
			eventbox.drag_dest_set(gtk.DEST_DEFAULT_ALL, [("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)
			eventbox.connect("drag-data-get", self.__dataGet__)
			eventbox.connect("drag-data-received", self.__dataReceived__)
			eventbox.connect("drag-motion", self.__dragEnter__)
			eventbox.connect("drag-leave", self.__dragLeave__)

			try:
				eventbox.drag_source_set_icon_pixbuf(pixbuf)
			except:
				print "ERROR: Pixbuf error:", location 

			return eventbox, label

	def selectAll(self):
		for location in self.locations:
			self.icons[location].set_state(gtk.STATE_SELECTED)

	def _destroy_(self):
		'''This function destroys the layout widget'''
		self.main_layout.destroy()
		

	############################ NOTEBOOK CALLBACKS ###############################

	def __notebookResized__(self, main_notebook, requisition):
		'''This function checks for a resized layout and handles relocating the icons'''

		if self.__populating__ != True:
			if self.tab.scroll.allocation.width > self.new_layout_width + 120:
				self.main_layout.set_size_request(self.tab.scroll.allocation.width, requisition.height)
				self.__placeIcons__()
			elif self.tab.scroll.allocation.width  < self.new_layout_width:
				self.main_layout.set_size_request(self.tab.scroll.allocation.width, requisition.height)
				self.__placeIcons__()


def __returnClean__(text):
	'''This function returns Bash script command line friendly text'''	
					
	# Here are the bad characters
	badChars = [" ","\'","&","(",")"]
			
	# Here are the replacement characters
	goodChars=["\ ","\\'","\&","\(","\)"]
			
	# Replace the bad with the good
	for x in range(len(badChars)):
		text = string.replace(text, badChars[x], goodChars[x])		
		
	# Return the "cleaned", BASH friendly text
	return text
