(defpackage :screen-repl (:use :common-lisp :screen)
            (:export :start-repl))
(in-package :screen-repl)

(defvar *screen* (make-instance 'tty-screen))

(defvar *buffer*)

(defclass buffer ()
  ((lines :accessor buffer-lines :initform nil :initarg :lines)))

(defclass line ()
  ((logical-lines :accessor logical-lines :initform nil :initarg :logical-lines)))

(defclass logical-line ()
  ((characters :accessor logical-line-characters :initform nil :initarg :characters)
   (row-on-screen :accessor logical-line-screen-row :initform 0 :initarg :row-on-screen)))

(defclass input-line (logical-line)
  ((view-origin :accessor input-line-view-origin :initform 0 :initarg :view-origin)
   (prompt-characters :accessor input-line-prompt-characters :initform nil :initarg :prompt-characters)))

(defvar *cursor-position*)

(defvar *input-line* nil) ; where input is allowed

(defvar *screen-rows*)

(defvar *screen-width*)

(defvar *redisplay-p* t)

(defvar *repl-output-line*)

(defvar *repl-input-stream*)

(defvar *repl-output-stream*)

(defvar *child-input-stream*)

(defvar *child-output-stream*)

(defvar *child-io-stream*)

(defmacro with-one-redisplay (&body body)
  `(progn (let ((*redisplay-p* nil))
            ,@body)
    (finish-screen *screen*)))

(defun input-line-on-screen-width ()
  (- *screen-width* (length (input-line-prompt-characters *input-line*)) 1))

(defun start-repl ()
  (if *screen* (release-screen *screen*))
  (setf *screen* (make-instance 'tty-screen))
  (initialize-screen *screen*)
  (setf (key-hook *screen*) 'my-key-hook)
  (multiple-value-setq (*screen-rows* *screen-width*)
    (get-screen-size *screen*))
  (setf *buffer* (make-instance 'buffer :lines
                                (list
                                 (setf *repl-output-line*
                                       (make-instance 'line :logical-lines
                                                      (list (setf *input-line* (make-instance 'input-line))))))))
  (setf *cursor-position* 0)
  (redisplay-input-line)
  (let ((handler nil))
    (multiple-value-bind (r w) (sb-unix:unix-pipe)
      (setf *child-output-stream* (sb-sys:make-fd-stream w :output t :buffering :none))
      (setf *repl-input-stream* (sb-sys:make-fd-stream r :input t :buffering :none))
      (setf handler
            (sb-sys:add-fd-handler r :input (lambda (fd) (declare (ignore fd))
                                                    (funcall 'repl-input-handler *repl-input-stream*)))))
    (multiple-value-bind (r w) (sb-unix:unix-pipe)
      (setf *child-input-stream* (sb-sys:make-fd-stream r :input t :buffering :none))
      (setf *repl-output-stream* (sb-sys:make-fd-stream w :output t :buffering :none)))
    (setf *child-io-stream* (make-two-way-stream *child-input-stream*
                                                 *child-output-stream*))
    (let* ((in *child-input-stream*)
           (out *child-output-stream*)
           (io *child-io-stream*)
           (*standard-output* out)
           (*error-output* out)
           (*trace-output* out)
           (*debug-io* io)
           (*query-io* io)
           (*standard-input* in)
           (*terminal-io* io)
           (toplevel-catch (gensym)))
      (unwind-protect
           (catch 'quit-repl
             (loop (catch toplevel-catch
                     (restart-case
                         (progn
                           (funcall sb-int:*repl-prompt-fun* out)
                           (format t "~{~S~%~}"
                                   (multiple-value-list
                                    (eval
                                     (funcall sb-int:*repl-read-form-fun* in out)))))
                       (abort ()
                         :report "Abort this evaluation, returning to toplevel."
                         (throw toplevel-catch nil))
                       (sb-impl::toplevel ()
                         :report "Return to SB-SCREEN toplevel."
                         (throw toplevel-catch nil))))))
        (release-screen *screen*)
        (sb-sys:remove-fd-handler handler)
        (close *repl-output-stream*)
        (close *repl-input-stream*)
        (close *child-output-stream*)
        (close *child-input-stream*)
        (close *child-io-stream*)))))
  
(defun fix-on-screen-lines ()
  (let ((row (logical-line-screen-row *input-line*)))
    (when (> row (1- *screen-rows*))
      (loop for line in (buffer-lines *buffer*)
            do (loop for logical-line in (logical-lines line)
                     do (decf (logical-line-screen-row logical-line)
                              (- row (1- *screen-rows*)))))
      (redisplay-lines))))
  

(defun write-string-to-repl-output-no-newlines (string)
  (let ((logical-line (car (last (logical-lines *repl-output-line*)))))
    (etypecase logical-line
      (input-line (setf (input-line-prompt-characters logical-line)
                        (append (input-line-prompt-characters logical-line)
                                (coerce string 'list)))
                  (with-one-redisplay
                    (when (< (input-line-on-screen-width) 1)
                      (let* ((new-contents (input-line-prompt-characters logical-line))
                             (new-contents (subseq new-contents 0 (min *screen-width* (length new-contents))))
                             (new-prompt (subseq (input-line-prompt-characters logical-line)
                                                 (length new-contents))))
                        (setf (input-line-prompt-characters logical-line) new-prompt)
                        (setf (logical-lines *repl-output-line*)
                              (append (butlast (logical-lines *repl-output-line*))
                                      (list (make-instance 'logical-line :row-on-screen (logical-line-screen-row logical-line) :characters new-contents))
                                      (list logical-line)))
                        (incf (logical-line-screen-row logical-line))
                        (fix-on-screen-lines)
                        (redisplay-lines :starting-from (1- (screen-row logical-line)))))
                    (redisplay-input-line))))))

(defun terpri-repl-output ()
  (let ((new-logical-line (make-instance 'logical-line
                                         :row-on-screen
                                         (logical-line-screen-row *input-line*)
                                         :characters
                                         (input-line-prompt-characters *input-line*))))
    (setf (logical-lines *repl-output-line*)
          (append (remove *input-line* (logical-lines *repl-output-line*))
                  (list new-logical-line)))
    (let ((new-line (make-instance 'line :logical-lines (list *input-line*))))
      (setf (buffer-lines *buffer*)
            (nconc (buffer-lines *buffer*)
                   (list new-line)))
      (incf (logical-line-screen-row *input-line*))
      (setf (input-line-prompt-characters *input-line*) nil)
      (setf *repl-output-line* new-line)
      (with-one-redisplay
        (fix-on-screen-lines)
        (redisplay-lines :starting-from (1- (screen-row new-logical-line)))))))

(defun repl-input-handler (stream)
  (loop with pending-chars = nil
        for char = (read-char-no-hang stream nil nil)
        do (if (or (eql char #\newline) (not char))
               (progn
                 (write-string-to-repl-output-no-newlines (nreverse pending-chars))
                 (when (eql char #\newline)
                   (terpri-repl-output))
                 (setf pending-chars nil)
                 (if (not char) (return t)))
               (push char pending-chars))))

(defun ensure-cursor ()
  (if (> *cursor-position* (length (logical-line-characters *input-line*)))
      (setf *cursor-position* (length (logical-line-characters *input-line*))))
  (if (< *cursor-position* (input-line-view-origin *input-line*))
      (setf (input-line-view-origin *input-line*) *cursor-position*)
      (if (> *cursor-position* (+ (input-line-view-origin *input-line*)
                                  (input-line-on-screen-width)))
          (if (>= (- *cursor-position* (input-line-on-screen-width)) 0)
              (setf (input-line-view-origin *input-line*)
                    (- *cursor-position* (input-line-on-screen-width)))))))

(defun set-input-line-prompt (string)
  (setf (input-line-prompt-characters *input-line*)
        (coerce string 'list))
  (ensure-cursor))

(defvar *display-offset* 0)

(defun screen-row (ll)
  (+ (logical-line-screen-row ll) *display-offset*))

(defun redisplay-lines (&key (starting-from 0))
  (loop for line in (buffer-lines *buffer*)
       do (loop for logical-line in (logical-lines line)
               do (if (and
                       (>= (screen-row logical-line) starting-from)
                       (< (screen-row logical-line) *screen-rows*)
                       (not (eq logical-line *input-line*)))
                      (progn
                        (set-cursor *screen* (screen-row logical-line) 0)
                        (erase-from-cursor-to-eol *screen*)
                        (write-string-at-cursor *screen* (coerce (logical-line-characters logical-line) 'string))))))
  (with-one-redisplay
    (redisplay-input-line)))

(defun redisplay-input-line ()
  (when (<= 0 (screen-row *input-line*) (1- *screen-rows*))
    (ensure-cursor)
    (set-cursor *screen* (screen-row *input-line*) 0)
    (erase-from-cursor-to-eol *screen*)
    (set-color *screen* :brightcyan :black)
    (write-string-at-cursor *screen*
                            (coerce (input-line-prompt-characters *input-line*) 'string))
    (set-color *screen* :yellow :black)
    (write-string-at-cursor *screen*
                            (coerce
                             (subseq (logical-line-characters *input-line*)
                                     (input-line-view-origin *input-line*)
                                     (min
                                      (+ (input-line-view-origin *input-line*)
                                         (input-line-on-screen-width))
                                      (length (logical-line-characters *input-line*))))
                             'string))
    (set-color *screen* :brightred :black)
    (if (> (length (logical-line-characters *input-line*))
           (+ (input-line-view-origin *input-line*)
              (input-line-on-screen-width)))
        (write-string-at-cursor *screen* ">")
        (write-string-at-cursor *screen* " "))
    (set-to-default-color *screen*)
    (set-cursor *screen* (screen-row *input-line*)
                (+ (- *cursor-position*
                      (input-line-view-origin *input-line*))
                   (length (input-line-prompt-characters *input-line*))))
    (when *redisplay-p*
      (finish-screen *screen*))))

(defun insert-char-at-cursor (char)
  (setf (logical-line-characters *input-line*)
        (append
         (subseq (logical-line-characters *input-line*) 0 *cursor-position*)
         (list char)
         (subseq (logical-line-characters *input-line*) *cursor-position*)))
  (incf *cursor-position*)
  (if (>= (- *cursor-position*
             (input-line-view-origin *input-line*))
          (input-line-on-screen-width))
      (incf (input-line-view-origin *input-line*)))
  (redisplay-input-line))

(defun delete-char-before-cursor ()
  (when (> *cursor-position* 0)
    (setf (logical-line-characters *input-line*)
          (append
           (subseq (logical-line-characters *input-line*) 0 (1- *cursor-position*))
           (subseq (logical-line-characters *input-line*) *cursor-position*)))
    (with-one-redisplay
      (move-cursor-left))))

(defun delete-char-at-cursor ()
  (with-one-redisplay
    (when (< *cursor-position* (length (logical-line-characters *input-line*)))
      (move-cursor-right)
      (delete-char-before-cursor))))

(defun move-cursor-left (&key by to)
  (if (not by)
      (setf by 1))
  (if (not to)
      (setf to (- *cursor-position* by)))
  (if (>= to 0)
      (setf *cursor-position* to))
  (when (< *cursor-position*
           (input-line-view-origin *input-line*))
    (if (< *cursor-position* (input-line-view-origin *input-line*))
        (setf (input-line-view-origin *input-line*)
              (max
               (- *cursor-position* (floor (input-line-on-screen-width) 2)) 0))))
  (redisplay-input-line))

(defun move-cursor-right (&key by to)
  (if (not by)
      (setf by 1))
  (if (not to)
      (setf to (+ *cursor-position* by)))
  (setf by (- to *cursor-position*))
  (if (<= to (length (logical-line-characters *input-line*)))
      (setf *cursor-position* to))
  (when (>= (- *cursor-position*
               (input-line-view-origin *input-line*))
            (length (input-line-prompt-characters *input-line*)))
    (setf (input-line-view-origin *input-line*)
          (max 0 (1+ (- *cursor-position* (input-line-on-screen-width))))))
  (redisplay-input-line))

(defvar *kill-ring* nil)

(defun delete-input-region (&key (from *cursor-position*) (to (length (logical-line-characters *input-line*))))
  (let ((new-cursor-position (max from 0)))
    (push (subseq (logical-line-characters *input-line*) from to) *kill-ring*)
    (setf *kill-ring* (loop for i from 1 to 10
                           for j in *kill-ring* collect j))
    (setf (logical-line-characters *input-line*)
          (append (subseq (logical-line-characters *input-line*) 0 from)
                  (subseq (logical-line-characters *input-line*) to)))
    (if (> (input-line-view-origin *input-line*)
           (length (logical-line-characters *input-line*)))
        (setf (input-line-view-origin *input-line*)
              (1- from)))
    (setf *cursor-position* new-cursor-position)
    (redisplay-input-line)))

(defun insert-text (text pos)
  (setf (logical-line-characters *input-line*)
        (append
         (subseq (logical-line-characters *input-line*) 0 pos)
         text
         (subseq (logical-line-characters *input-line*) pos)))
  (if (< *cursor-position* (+ pos (length text)))
      (move-cursor-left :to (+ pos (length text)))
      (move-cursor-right :to (+ pos (length text))))
  (redisplay-input-line))

(defvar *last-yank-location* 0)

(defun yank-at-cursor ()
  (setf *last-yank-location* *cursor-position*)
  (insert-text (pop *kill-ring*) *cursor-position*))

(defun rotate-kill-ring ()
  (when (< *last-yank-location* (length (logical-line-characters *input-line*)))
    (delete-input-region :from *last-yank-location* :to *cursor-position*)
    (let ((top (pop *kill-ring*)))
      (setf *kill-ring* (nconc *kill-ring* (list top))))
    (yank-at-cursor)))

(defclass keybinding ()
  ((key :accessor key-of :initarg :key)))

(defparameter *key-hash* (make-hash-table :test #'equal))

(defun intern-keybinding (key)
  (multiple-value-bind (got found)
      (gethash key *key-hash*)
    (if found
        got
        (setf (gethash key *key-hash*) (make-instance 'keybinding :key key)))))

(defgeneric invoke-key (key))

(defmethod invoke-key (anything)
  (if (typep (key-of anything) 'character)
      (insert-char-at-cursor (key-of anything))))

(defmethod invoke-key ((anything (eql (intern-keybinding '(:control #\f)))))
  (move-cursor-right))

(defmethod invoke-key ((anything (eql (intern-keybinding '(:control #\b)))))
  (move-cursor-left))

(defmethod invoke-key ((anything (eql (intern-keybinding '(:control #\a)))))
  (move-cursor-left :to 0))

(defmethod invoke-key ((anything (eql (intern-keybinding '(:control #\e)))))
  (move-cursor-right :to (length (logical-line-characters *input-line*))))

(defmethod invoke-key ((anything (eql (intern-keybinding :rubout))))
  (delete-char-before-cursor))

(defmethod invoke-key ((anything (eql (intern-keybinding '(:control #\d)))))
  (delete-char-at-cursor))

(defun arrow-key-hook (decoded-key)
  (case decoded-key
    (#\A (invoke-key (intern-keybinding :up-arrow)))
    (#\B (invoke-key (intern-keybinding :down-arrow)))
    (#\D (invoke-key (intern-keybinding :left-arrow)))
    (#\C (invoke-key (intern-keybinding :right-arrow))))
  (setf (key-hook *screen*) 'my-key-hook))

(defun escape-key-hook (decoded-key)
  (case decoded-key
    (#\[ (setf (key-hook *screen*) 'arrow-key-hook))
    (t
     (setf (key-hook *screen*) 'my-key-hook)
     (invoke-key (intern-keybinding `(:meta ,decoded-key))))))

(defmethod invoke-key ((key (eql (intern-keybinding :escape))))
  (setf (key-hook *screen*) 'escape-key-hook))

(defmethod invoke-key ((key (eql (intern-keybinding :left-arrow))))
  (move-cursor-left))

(defmethod invoke-key ((key (eql (intern-keybinding :right-arrow))))
  (move-cursor-right))

(defvar *input-ring* nil)

(defmethod invoke-key ((key (eql (intern-keybinding :up-arrow))))
  (if *input-ring*
      (let ((new-text (pop *input-ring*))
            (old-text (logical-line-characters *input-line*)))
        (setf (logical-line-characters *input-line*) new-text)
        (setf *input-ring* (nconc *input-ring* (list old-text)))
        (ensure-cursor)
        (redisplay-input-line))))

(defmethod invoke-key ((key (eql (intern-keybinding :down-arrow))))
  (if *input-ring*
      (let ((new-text (prog1 (car (last *input-ring*))
                        (setf *input-ring* (nbutlast *input-ring*))))
            (old-text (logical-line-characters *input-line*)))
        (setf (logical-line-characters *input-line*) new-text)
        (push old-text *input-ring*)
        (ensure-cursor)
        (redisplay-input-line))))

(defmethod invoke-key ((key (eql (intern-keybinding '(:control #\k)))))
  (delete-input-region))

(defmethod invoke-key ((key (eql (intern-keybinding '(:control #\y)))))
  (yank-at-cursor))

(defmethod invoke-key ((key (eql (intern-keybinding '(:meta #\y)))))
  (rotate-kill-ring))

(defmethod invoke-key ((key (eql (intern-keybinding '(:control #\m)))))
  (let ((chars (logical-line-characters *input-line*)))
    (setf (logical-line-characters *input-line*) nil)
    (push chars *input-ring*)
    (with-one-redisplay
      (write-string-to-repl-output-no-newlines chars)
      (terpri-repl-output))
    (write-string (coerce chars 'string) *repl-output-stream*)
    (terpri *repl-output-stream*)))

(defmethod invoke-key ((key (eql (intern-keybinding '(:meta #\p)))))
  (incf *display-offset* (min 5
                              (- (screen-row (car
                                              (logical-lines
                                               (car (buffer-lines *buffer*))))))))
  (redisplay-lines))

(defmethod invoke-key ((key (eql (intern-keybinding '(:meta #\n)))))
  (setf *display-offset* (max 0 (- *display-offset* 5)))
  (redisplay-lines))

(defmethod invoke-key ((key (eql (intern-keybinding '(:control #\d)))))
  (throw 'quit-repl nil))

(defun my-key-hook (decoded-key)
  (invoke-key (intern-keybinding decoded-key)))

(defun repair-screen-fart ()
  (clear-screen *screen*)
  (redisplay-input-line))