(in-package :contextl)

(defclass standard-layer-object (special-object)
  ())

(defgeneric layer-name (layer)
  (:method ((layer symbol)) layer)
  (:method ((layer (eql (find-class 't)))) 't)
  (:method ((layer standard-layer-object)) (layer-name (class-of layer))))

(defclass standard-layer-class (special-class singleton-class)
  ((layer-name :initarg original-name
               :reader layer-name)))

(defmethod validate-superclass
           ((class standard-layer-class)
            (superclass standard-class))
  t)

(defmethod print-object ((object standard-layer-object) stream)
  (print-unreadable-object (object stream :type nil :identity t)
    (format stream "LAYER ~A" (layer-name object))))

(defmethod print-object ((object standard-layer-class) stream)
  (print-unreadable-object (object stream :type t :identity t)
    (princ (layer-name object) stream)))

(defmethod initialize-instance :around
  ((class standard-layer-class) &rest initargs &key direct-superclasses)
  (declare (dynamic-extent initargs))
  (if (loop for direct-superclass in direct-superclasses
            thereis (ignore-errors (subtypep direct-superclass 'standard-layer-object)))
    (call-next-method)
    (apply #'call-next-method
           class
           :direct-superclasses
           (append direct-superclasses
                   (list (find-class 'standard-layer-object)))
           initargs)))

(defmethod reinitialize-instance :around
  ((class standard-layer-class) &rest initargs &key (direct-superclasses '() direct-superclasses-p))
  (declare (dynamic-extent initargs))
  (if (or (not direct-superclasses-p)
          (loop for direct-superclass in direct-superclasses
                thereis (ignore-errors (subtypep direct-superclass 'standard-layer-object))))
    (call-next-method)
    (apply #'call-next-method
           class
           :direct-superclasses
           (append direct-superclasses
                   (list (find-class 'standard-layer-object)))
           initargs)))

(defclass layer-direct-slot-definition (singleton-direct-slot-definition
                                        special-direct-slot-definition)
  ())

(defmethod direct-slot-definition-class ((class standard-layer-class) &key)
  (find-class 'layer-direct-slot-definition))

(defmacro deflayer (name &optional superlayers &body options)
  (destructuring-bind (&optional slots &rest options) options
    `(defclass ,(defining-layer name) ,(mapcar #'defining-layer superlayers)
       ,(if slots slots '())
       ,@options
       ,@(unless (assoc :metaclass options)
           '((:metaclass standard-layer-class)))
       (original-name . ,name))))

(defun ensure-layer (layer-name
                     &rest initargs
                     &key (metaclass 'standard-layer-class)
                     &allow-other-keys)
  (declare (dynamic-extent initargs))
  (apply #'ensure-class
         (defining-layer layer-name)
         :metaclass metaclass
         'original-name layer-name
         initargs))

(defgeneric find-layer-class (layer &optional errorp environment)
  (:method ((layer (eql 't)) &optional errorp environment)
   (declare (ignore errorp environment))
   (load-time-value (find-class 't)))
  (:method ((layer (eql (find-class 't))) &optional errorp environment)
   (declare (ignore errorp environment))
   (load-time-value (find-class 't)))
  (:method ((layer symbol) &optional (errorp t) environment)
   (or (find-class (defining-layer layer) nil environment)
       (when errorp
         (error "There is no layer named ~S." layer))))
  (:method ((layer standard-layer-object) &optional errorp environment)
   (declare (ignore errorp environment))
   (class-of layer))
  (:method ((layer standard-layer-class) &optional errorp environment)
   (declare (ignore errorp environment))
   layer))

(defgeneric find-layer (layer &optional errorp environment)
  (:method ((layer (eql 't)) &optional errorp environment)
   (declare (ignore errorp environment))
   't)
  (:method ((layer (eql (find-class 't))) &optional errorp environment)
   (declare (ignore errorp environment))
   't)
  (:method ((layer symbol) &optional (errorp t) environment)
   (let ((layer-class (find-layer-class layer errorp environment)))
     (when layer-class
       #-lispworks (ensure-finalized layer-class)
       (class-prototype layer-class))))
  (:method ((layer standard-layer-object) &optional errorp environment)
   (declare (ignore errorp environment))
   layer)
  (:method ((layer standard-layer-class) &optional errorp environment)
   (declare (ignore errorp))
   #-lispworks (ensure-finalized layer)
   (class-prototype layer)))

(defgeneric layer-makunbound (layer)
  (:method ((layer symbol))
   (setf (find-class (defining-layer layer)) nil))
  (:method ((layer standard-layer-object))
   (setf (find-class (class-name (class-of layer))) nil))
  (:method ((layer standard-layer-class))
   (setf (find-class (class-name layer)) nil)))

(defclass root-specializer () ()
  (:metaclass standard-layer-class))
(ensure-finalized (find-class 'root-specializer))

(defstruct layer-context
  (prototype (class-prototype (find-class 'root-specializer))
             :type standard-object
             :read-only t)
  (specializer (find-class 'root-specializer)
               :type standard-layer-class
               :read-only t)
  (children/ensure-active () :type list)
  (children/ensure-inactive () :type list))

#-allegro
(declaim (type layer-context *root-context* *active-context*))
#+allegro
(eval-when (:load-toplevel :execute)
  (proclaim '(type layer-context *root-context* *active-context*)))

(defparameter *root-context* (make-layer-context))

(defparameter *active-context* *root-context*)

(declaim (inline current-layer-context))
(defun current-layer-context () *active-context*)

(defun layer-active-p (layer &optional (context *active-context*))
  (subtypep (layer-context-specializer context)
            (find-layer-class layer)))

(define-layered-function adjoin-layer-using-class (layer-class active-context)
  (:method ((layer-class (eql (find-class 't))) active-context)
   (values active-context t))
  (:method ((layer-class standard-layer-class) active-context)
   (let ((active-context-specializer (layer-context-specializer active-context)))
     (values
      (if (subtypep active-context-specializer layer-class)
        active-context
        (let ((new-specializer
	       (ensure-finalized
		(make-instance 'standard-layer-class
			       :direct-superclasses
			       (list layer-class active-context-specializer)))))
          (make-layer-context
           :prototype (class-prototype new-specializer)
           :specializer new-specializer)))
      t))))

(defun adjoin-layer (layer active-context)
  (multiple-value-bind
      (new-layer-context cacheablep)
      (adjoin-layer-using-class (find-layer-class layer) active-context)
    (if cacheablep
      (setf (getf (layer-context-children/ensure-active active-context) layer)
            new-layer-context)
      new-layer-context)))

(declaim (inline ensure-active-layer-context))

(defun ensure-active-layer-context (layer active-context)
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (or (getf (layer-context-children/ensure-active active-context) layer)
      (adjoin-layer layer active-context)))

(defun ensure-active-layer (layer-name)
  (setf *active-context*
        (locally
          (declare (optimize (speed 3) (debug 0) (safety 0)
                             (compilation-speed 0)))
          (ensure-active-layer-context layer-name *active-context*))))

(define-layered-function remove-layer-using-class (layer-class active-context)
  (:method ((layer-class (eql (find-class 't))) active-context)
   (declare (ignore active-context))
   (error "The layer T may never be removed."))
  (:method ((layer-class standard-layer-class) active-context)
   (let ((active-context-specializer (layer-context-specializer active-context)))
     (values
      (loop for context-specializer = active-context-specializer
            then (second (class-direct-superclasses context-specializer))
            for active-layers = (list (first (class-direct-superclasses context-specializer)))
            then (cons (first (class-direct-superclasses context-specializer)) active-layers)
            until (eq context-specializer (load-time-value (find-class 'root-specializer)))
            finally
            (return (loop for new-layer-context = *root-context*
                          then (if (subtypep active-layer layer-class)
                                 new-layer-context
                                 (ensure-active-layer-context 
				  (layer-name active-layer) 
				  new-layer-context))
                          for active-layer in (cdr active-layers)
                          finally (return new-layer-context))))
      t))))

(defun remove-layer (layer active-context)
  (multiple-value-bind
      (new-layer-context cacheablep)
      (remove-layer-using-class (find-layer-class layer) active-context)
    (if cacheablep
      (setf (getf (layer-context-children/ensure-inactive active-context) layer)
            new-layer-context)
      new-layer-context)))

(declaim (inline ensure-inactive-layer-context))

(defun ensure-inactive-layer-context (layer active-context)
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (or (getf (layer-context-children/ensure-inactive active-context) layer)
      (remove-layer layer active-context)))

(defun ensure-inactive-layer (layer-name)
  (setf *active-context*
        (locally
          (declare (optimize (speed 3) (debug 0) (safety 0)
                             (compilation-speed 0)))
          (ensure-inactive-layer-context layer-name *active-context*))))

(defmacro with-active-layer (layer-name &body body)
  `(let ((*active-context*
          (locally
            (declare (optimize (speed 3) (debug 0) (safety 0)
                               (compilation-speed 0)))
            (ensure-active-layer-context ',layer-name *active-context*))))
     ,@body))

(defmacro with-active-layers ((&rest layer-names) &body body)
  (if (every #'symbolp layer-names)
    (if layer-names
      `(with-active-layer ,(car layer-names)
         (with-active-layers ,(cdr layer-names)
           ,@body))
      `(progn ,@body))
    (loop for form = `(with-special-initargs 
			  ,(loop for layer-spec in layer-names
			      when (consp layer-spec)
			      collect `((find-layer ',(car layer-spec)) ,@(cdr layer-spec)))
                        ,@body)
          then `(with-active-layer ,layer-name ,form)
          for layer-spec in (reverse layer-names)
          for layer-name = (if (symbolp layer-spec)
                             layer-spec
                             (car layer-spec))
          finally (return form))))

(defmacro with-active-layers* ((&rest layer-names) &body body)
  (if (every #'symbolp layer-names)
    (if layer-names
      `(with-active-layer ,(car layer-names)
         (with-active-layers ,(cdr layer-names)
           ,@body))
      `(progn ,@body))
    (loop for form = `(with-special-initargs* 
			  ,(loop for layer-spec in layer-names
			      when (consp layer-spec)
			      collect `((find-layer ',(car layer-spec)) ,@(cdr layer-spec)))
                        ,@body)
          then `(with-active-layer ,layer-name ,form)
          for layer-spec in (reverse layer-names)
          for layer-name = (if (symbolp layer-spec)
                             layer-spec
                             (car layer-spec))
          finally (return form))))

(defmacro with-inactive-layer (layer-name &body body)
  `(let ((*active-context*
          (locally
            (declare (optimize (speed 3) (debug 0) (safety 0)
                               (compilation-speed 0)))
            (ensure-inactive-layer-context ',layer-name *active-context*))))
     ,@body))

(defmacro with-inactive-layers ((&rest layer-names) &body body)
  (if layer-names
    `(with-inactive-layer ,(car layer-names)
       (with-inactive-layers ,(cdr layer-names)
         ,@body))
    `(progn ,@body)))

(defun funcall-with-layer-context (layer-context function &rest args)
  (declare (dynamic-extent args))
  (let ((*active-context* layer-context))
    (apply function args)))

(defun apply-with-layer-context (layer-context function &rest args)
  (declare (dynamic-extent args))
  (let ((*active-context* layer-context))
    (apply #'apply function args)))
