#include "cp_types.h"
#include "cp_proto.h"

/* File for opening/managing/closing postscript output files.

This is all rather complicated since stuff can be added to the
postscript file in increments, pages an be defined, material
inserted, etc. Thus a 'trailer' file is created which is added 
when the file is finally completed and about to be closed. */

extern int sc_print_size;

int open_psfile(struct p_data *p,int flag,char *datastr,
		FILE **FP,char *filename,char *user_name,char *Message)
/* Open postscript file 'filename' for new data from pack p; 
may/may not be open/initialized already. If adding new material, 
position file after 'datastr'. Optional info includes 'user_name',
and Message (typically CPVersion). The 'flag' specifies: 
   0 = send to stdout
   1 = open the file (previous contents discarded!)
   2 = insert in file (open if necessary)
   3 = append to file (open if necessary) */
{
  int i,init_flag=1,reopen_flag=0;
  char buff[256];
  struct tm *t;
  time_t clicks=time(NULL);
  FILE *fp;

  fp=*FP;
  if (flag>1 && !fp && strlen(filename)<2) 
    {
      sprintf(msgbuf,"Need postscript file name");
      emsg();
      return 0;
    }

  if (flag==1) /* close file (contents lost) and reopen. */
    {
      if (fp) close_psfile(&fp);
      if (!(fp=*FP=fopen(filename,"w+"))) 
	{sprintf(msgbuf,"Can't open %s. ",filename);emsg();return 0;}
    }
  else if (flag) /* For inserting or appending new page. */
    {
      if (!fp) /* not currently open */
	{
	  if ((fp=*FP=fopen(filename,"r+"))) /* exists, assume initialized */
	    {
	      reopen_flag=1;
	      init_flag=0;
	    }
	  else if (!(fp=*FP=fopen(filename,"w+"))) /* new */
	    {sprintf(msgbuf,"Can't open %s. ",filename);emsg();return 0;}
	}
      else init_flag=0; /* already open, assume initialized */

      if (!init_flag) /* now open, assume initialized */
	{
	  if (flag==2) /* insert mode; position at string */
	    {
	      strncpy(buff,datastr,255);
	      stripsp(buff);
	      i=strlen(buff);
	      while(i>0 
		 && (buff[i]==' ' || buff[i]=='\t' 
		 || buff[i]=='\n' || buff[i]=='r'))
		buff[--i]='\0'; /* strip trailing white space */
	      if (strlen(buff)>1 && position_after(fp,buff))
		{
		  custom_trailer(fp,2); 
	           /* store insert trailer for later reattach. */
		  fprintf(fp,"\nsave %% insert\n");
		  if (reopen_flag) post_size_settings(fp,p,2);
		}
	      else /* default to new page  */
		{
		  sprintf(msgbuf,"Didn't find string %s in custom "
			  "print file; default to adding new page.",
			  datastr);
		  emsg();
		  fseek(fp,(long)0,SEEK_END); 
		  begin_page_info(fp,p,2); 
		}
	    }
	  else if (flag==3) /* append mode; start new page */
	    {
	      fseek(fp,(long)0,SEEK_END); 
	      begin_page_info(fp,p,2); 
	           /* fixup: ?? all later pages numbered as 2 for now */
	    }
	}
      else /* direct to stdout */
	fp=stdout;
    }
/* fp should now be open, initialized, and positioned */

  if (!fp) {sprintf(msgbuf,"Couldn't open %s. ",filename);emsg();return 0;} 

if (init_flag) /* need to initialize */
  {
/* -------- preamble: standard eps-type header info -------------- */
    fprintf(fp,"%%!PS-Adobe-2.0 EPSF-2.0\n%%%%Title: %s\n",p->file_name);
    t=localtime(&clicks); /* get time/date */
    strftime(buff,256,"%c",t);
    fprintf(fp,"%%%%Creator: CirclePack Version %s\n%%%%CreationDate: %s\n",
	    Message,buff);
    fprintf(fp,"%%%%For: %s\n%%%%Orientation: Portrait\n",user_name);
    post_size_settings(fp,p,1);
    fprintf(fp,"%%%%Pages: 1\n%%%%BeginSetup\n%%%%EndSetup\n%%%%"
	    "Magnification: 1.0000\n%%%%EndComments\n");
    fprintf(fp,"\n%% CirclePack dictionary ================\n");

/* -------- define CPdict ------------------------------------------ */
    fprintf(fp,"/CPdict 256 dict def\nCPdict begin");

    fprintf(fp,"\n%% --- Standard abbreviations\n");
    fprintf(fp,"\t/cp {closepath} bind def\n\t/ef {eofill} bind "
	    "def\n\t/gr {grestore} bind def\n\t/gs {gsave} bind "
	    "def\n\t/sa {save} bind def\n\t/rs {restore} bind "
	    "def\n\t/l {lineto} bind def\n\t/m {moveto} bind "
	    "def\n\t/rm {rmoveto} bind def\n");
    fprintf(fp,"\t/n {newpath} bind def\n\t/s {stroke} bind "
	    "def\n\t/sh {show} bind def\n\t/slc {setlinecap} bind "
	    "def\n\t/slj {setlinejoin} bind def\n\t/slw "
	    "{setlinewidth} bind def\n\t/srgb {setrgbcolor} bind "
	    "def\n\t/rot {rotate} bind def\n\t/sc {scale} bind def\n");
    fprintf(fp,"\t/sd {setdash} bind def\n\t/ff {findfont} bind "
	    "def\n\t/sf {setfont} bind def\n\t/scf {scalefont} bind "
	    "def\n\t/sw {stringwidth} bind def\n\t/tr {translate} "
	    "bind def\n");
/* -------- Unique to CirclePack abbreviations, etc. -------------- */
    fprintf(fp,"\n%% --- Special abbreviations\n");
    fprintf(fp,"   /sg {setgray} bind def\n   /a {arc} bind "
	    "def\n   /an {arcn} bind def\n");
    fprintf(fp,"   /c { 0 360 a s} bind def\t\t\t%% circle\n");
    fprintf(fp,"   /cc {0 360 a gs srgb s gr} bind def\t\t%% color circle\n");
    fprintf(fp,"   /d { 0 360 a gs sg fill gr s} bind def\t%% disc\n");
    fprintf(fp,"   /cd {0 360 a gs srgb fill gr s} bind def\t%% color disc\n");
    fprintf(fp,"   /mark {gs 0 360 a srgb fill gr} bind "
	    "def\t%%default mark symbol\n");
    fprintf(fp,"   /wht {1.0} bind def\t\t\t\t%% gray levels\n   "
	    "/gry {0.8} bind def\n  /drk {0.5} bind def\n   /blck "
	    "{0.0} bind def\n");
    fprintf(fp,"   /ourlinewidth {.004 mul setlinewidth} bind "
	    "def\n");
    fprintf(fp,"\nend\n%% end CirclePack dictionary "
	    "=================\n%%%%EndProlog\n");

/* -------------- begin first page ----------------------- */

    begin_page_info(fp,p,1);
  } 
         return 1;
} /* open_psfile */

int close_psfile(FILE **FP)
/* close a postscript file with the stored trailer */
{
  if (!(*FP)) return 0;
  custom_trailer(*FP,0); /* attach prepared trailer stuff */
  if (*FP!=stdout) 
    {
      fclose(*FP);
      *FP=NULL;
    }
  return 1;
} /* close_psfile */

int begin_page_info(FILE *fp,struct p_data *p,int n)
/* info on active pack to start ps page */
{
  fprintf(fp,"\n%%%%Page: %d %d\nCPdict begin\ngsave\n",n,n);
  fprintf(fp,"   72 72 sc %% inches\n   %f %f tr\n   1 slc   "
	  "1 slj\n   %f ourlinewidth\n", 4.25,5.5,ps_linewidth);
  post_size_settings(fp,p,6);
  custom_trailer(fp,1); /* store standard end trailer for later attachment */
  return 1;
} /* begin_page_info */

int custom_trailer(FILE *fp,int flag) 
/* store/recover final trailers for custom print file. Depends on 
mode. Always reset fp to right value before returning. */
{
  FILE *tmpfp;
  int c;
  long fspot;

  if (!fp) return 0;
  if (flag==0) /* attach trailer stuff onto end of fp */
    {
      if (!(tmpfp=fopen("/tmp/custom_tail.ps","r"))) return 0;
      while ((c=getc(tmpfp))!=EOF) putc(c,fp);
      fclose(tmpfp);
      fflush(fp);
      return 1;
    } 
  else if (flag==1) /* store trailer - new page mode. */
    {
      if (!(tmpfp=fopen("/tmp/custom_tail.ps","w"))) return 0;
      fprintf(tmpfp,"\ngrestore\nend\nshowpage\n"); /* page wrapup */
      fclose(tmpfp);
      return 1;
    }
  else if (flag==2) /* store trailer - insert mode. */
    {
      if (!(tmpfp=fopen("/tmp/custom_tail.ps","w"))) return 0;
      fprintf(tmpfp,"\nrestore %% insert\n");
      fspot=ftell(fp);
      while ((c=getc(fp))!=EOF) putc(c,tmpfp);
      fclose(tmpfp);
      fseek(fp,fspot,SEEK_SET);
      return 1;
    }
  return 1;
} /* custom_trailer */
		  
int position_after(FILE *fp,char *datastr)
/* position fp (open for read/write) after first occurance of datastr; 
return 1 = success, 0 = fail. Caution: changes file location. */
{
  int length=strlen(datastr),i;
  int got,ngot;
  long pos;

  if (length<1 || length>256) return 0; /* datastr not proper */
  rewind(fp);
  while ((got=getc(fp))!=EOF) 
    {
      if (got==(int)datastr[0])
	{
	  pos=ftell(fp);
	  i=1;
	  while ( (i<length) && (ngot=getc(fp))!=EOF 
		  && ngot==(int)(*(datastr+i)) ) i++;
	  if (i==length)
	    {
	      fseek(fp,(long)(pos+length-1),SEEK_SET);
	      return 1;
	    }
	  fseek(fp,pos,SEEK_SET); /* reposition */
	}
    }
  fflush(fp);
  return 0;
} /* position_after */

int position_before(FILE *fp,char *datastr)
/* position fp (open for read/write) just before first occurance of datastr; 
return 1 = success, 0 = fail. Caution: changes file location. */
{
  int length=strlen(datastr),i;
  int got,ngot;
  long pos;

  if (length<1 || length>256) return 0; /* datastr not proper */
  rewind(fp);
  while ((got=getc(fp))!=EOF) 
    {
      if (got==(int)datastr[0])
	{
	  pos=ftell(fp);
	  i=1;
	  while ( (i<length) && (ngot=getc(fp))!=EOF 
		  && ngot==(int)(*(datastr+i)) ) i++;
	  if (i==length && pos>2)
	    {
	      fseek(fp,(long)(pos-1),SEEK_SET);
	      return 1;
	    }
	  fseek(fp,pos,SEEK_SET); /* reposition */
	}
    }
  fflush(fp);
  return 0;
} /* position_before */

int post_size_settings(FILE *fp,struct p_data *p,int flag)
/* put screen-dependent size data in postscript file. 
   flag is bit coded: 1 = BoundingBox, 2 = 
1 = BoundingBox; 2 = size info; 4 = clip window. */
{
  double factor,size;

  if (!fp) return 0;
  size= (double) sc_print_size;

  if(flag & 1) /* BoundingBox */
    {
      fprintf(fp,"%%%%BoundingBox: %d %d %d %d\n",
	      (int)(72*(4.25-size/2))-10,(int)(72*(5.5-size/2))-10,
	      (int)(72*(4.25+size/2))+10,(int)(72*(5.5+size/2))+10);
    }
  if (flag==1) return 1;

  if (flag & 2) /* basic size info */
    {
      fprintf(fp,"%% ---------- pack size settings\n");
      factor= size/(p->screen->box.ry-p->screen->box.ly);
      /*  -------- define 'ourlinewidth' at 1 point */
      fprintf(fp,"     /ourlinewidth\n      { %f mul setlinewidth}  def\n",
	      1/(factor*72)); 
      /* -------- scaling for screen and desired output size */
      fprintf(fp,"     %f %f sc \n     %f %f tr\n     %f "
	      "ourlinewidth\n     0 sg\n",
	      factor,(1.01)*factor,-(p->screen->box.lx+p->screen->box.rx)/2.0,
	      -(p->screen->box.ly+p->screen->box.ry)/2.0,ps_linewidth);
      fprintf(fp,"/Times-Roman ff %f scf sf\nn\n%% ------------\n",
	  .15/factor); /* scale font */
    }
  if (flag & 4) /* clip window */
    {
/* ----- now scaled based on screens size and desired output size */
      fprintf(fp,"n\n%f %f m\n%f %f l\n%f %f l\n%f %f "
	      "l\ncp\n%%gs s gr\nclip\nn\n",
	      p->screen->box.lx,p->screen->box.ry,
	      p->screen->box.lx,p->screen->box.ly,
	      p->screen->box.rx,p->screen->box.ly,
	      p->screen->box.rx,p->screen->box.ry); /* clip box */
    }
  if (p->hes>0) /* draw sphere */
    fprintf(fp,"n 0 0 1 c\nn\n");
  return 1;
} /* post_size_settings */



