# +==========================================================================+
# || CipUX::RPC::Server                                                     ||
# ||                                                                        ||
# || CipUX RPC Server Class                                                 ||
# ||                                                                        ||
# || Copyright (C) 2007 - 2009 by Christian Kuelker                         ||
# ||                                                                        ||
# || License: GNU GPL - GNU general public license - version 2              ||
# ||          or (at your opinion) any later version.                       ||
# ||                                                                        ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RPC::Server;

use 5.008001;
use strict;
use warnings;
use utf8;

use Authen::Simple::PAM;
use Authen::Simple::Password;
use Carp;
use CipUX 3.4.0.4;
use CipUX::Task;
use CipUX::RBAC::Simple;
use Class::Std;
use Data::Dumper;
use Digest::MD5 qw(md5_hex);
use English qw( -no_match_vars);
use Frontier::Daemon;
use Frontier::RPC2;
use List::MoreUtils qw(any none);
use Log::Log4perl qw(get_logger :levels);
use Readonly;

use base qw(CipUX::RPC);
{             # BEGIN CLASS

    # CONSTRUCTOR
    # DESTRUCTOR
    # PRIVATE METHODS
    # * privacy cathegory 1
    #   - check_authentication
    #   - create_ticket
    #   - is_ticket_bad
    #   - calc_random_seed
    # * privacy cathegory 2
    #   - signal_handler
    #   - answer_requests
    #   - check_access_to_task
    #   - check_access_to_rpc_intern
    #   - check_access_to_cat_module
    #   - error
    #   - evaluate_access
    #   - update_task
    #   - update_cat_module
    # PUBLIC METHODS
    # - get_config
    # - rpc_list_functions
    # - rpc_start
    # XML-RPC METHODS
    # - ping
    # - version
    # - sum
    # - login
    # - logout
    # - session
    # - ttl
    # - rpc_task
    # - rpc_info
    # - rpc_intern

    use version; our $VERSION = qv('3.4.0.6');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $SCRIPT       => 'CipUX::RPC::Server';
    Readonly::Array my @PASSWD_CHARS =>
        ( 'A' .. 'Z', 'a' .. 'z', 0 .. 9, qw(! @ $ % &) );
    Readonly::Scalar my $TICKET_LENGTH => 128;
    Readonly::Scalar my $SEED          => calc_random_seed();
    Readonly::Scalar my $HEADER_HREF   => {
        'cipux_version'  => '3.4.0.0',
        'server_name'    => 'cipux_rpcd',
        'server_version' => '3.4.0.0',
        'rpc_version'    => '2.0',
        'server_key'     => $EMPTY_STRING,
        'server_cred'    => $EMPTY_STRING,
        'gmt_time'       => time,
    };
    Readonly::Scalar my $DEF_TTL            => 1200;       # in sec
    Readonly::Scalar my $A_SHORT_WHILE      => 3;          # in sec
    Readonly::Scalar my $INTERN_ADMIN_GROUP => 'admin';    # last resort

    Readonly::Array my @ADMIN_GROUP_CLIENT =>
        qw(cipux_rpc_test_client cipux_rpc_test_repetition cipuxpasswd);
    Readonly::Array my @RPC_INTERN_CMD    => qw(rpc_intern);
    Readonly::Array my @RPC_INTERN_SUBCMD => qw(flush
        cat_module_cache_size
        rpc_intern_cache_size
        task_cache_size
        user_task_access
        user_task_access_survey
        user_cat_module_access
        user_cat_module_access_survey
        user_rpc_intern_access
        user_rpc_intern_access_survey
    );

    # +======================================================================+
    # || OBJECT                                                             ||
    # +======================================================================+

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+
    my $cipux_task         = undef;
    my $cipux_task_list_ar = undef;
    my $rpc_cfg_hr         = undef;  # undef means: load it at the first time.
    my $rpc                = undef;
    my $rbac               = undef;
    my $time_to_die        = 0;      # shutdown server with SIG handler
    my %time               = ();

    # +======================================================================+
    # || CONSTRUCTOR                                                        ||
    # +======================================================================+
    sub BUILD {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident, $arg_r ) = @_;

        my $cache_dir
            = exists $arg_r->{cache_dir}
            ? $self->l( $arg_r->{cache_dir} )
            : $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | main
        $cipux_task         = CipUX::Task->new( { cache_dir => $cache_dir } );
        $cipux_task_list_ar = $cipux_task->list_task();
        $rpc                = CipUX::RPC->new( { cache_dir => $cache_dir } );
        $rbac               = CipUX::RBAC::Simple->new();

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || DESTRUCTOR                                                         ||
    # +======================================================================+
    sub DEMOLISH {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident ) = @_;

        # +------------------------------------------------------------------+
        # | main
        undef $cipux_task;
        undef $cipux_task_list_ar;
        undef $rpc_cfg_hr;
        undef $rpc;
        undef $rbac;
        undef $time_to_die;
        undef %time;

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || PRIVATE functions to the server (also not exported via XML-RPC)    ||
    # +======================================================================+

    # +----------------------------------------------------------------------+
    # |  PRIVACY CATEGORY 1 (have to keep them private)                      |
    # +----------------------------------------------------------------------+

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  check_authentication                                                :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub check_authentication : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        #$self->var_dump( { var_r => \$arg_r, name => 'arg_r' } );

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $password
            = exists $arg_r->{password}
            ? $self->l( $arg_r->{password} )
            : $self->perr('password');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $authenticated = 0;

        # check for empty password, return 0 (FALSE) if empty
        if ( $password eq $EMPTY_STRING ) {
            $logger->debug('Empty password');
            $logger->debug('END');
            return 0;
        }

        $logger->debug('using PAM to authenticate login');
        my $pam = Authen::Simple::PAM->new( service => 'password' );
        if ( $pam->authenticate( $login, $password ) ) {

            # successfully authentication return 1 (TRUE)
            $logger->debug("authentication sucessfully for login [$login]");
            $authenticated = 1;

        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $authenticated;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  create_ticket                                                    :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # This sub can be used in 2 manners:
    # (1):
    # my $testticket = $self->create_ticket({login=>$l,test=>1});
    # (0)
    # my $ticket = $self->create_ticket({login=>$l,test=>0});
    # my $ticket = $self->create_ticket({login=>$l});
    # This form also stores the valid untill time global for $login
    sub create_ticket : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login create_ticket');

        my $test
            = (     exists $arg_r->{test}
                and defined $arg_r->{test}
                and $arg_r->{test} ) ? 1 : 0;

        # +------------------------------------------------------------------+
        # | prepare
        $logger->debug("input parameter login: [$login]");
        $logger->debug("input parameter test: [$test]");
        $logger->debug("global parameter SEED: [$SEED]");

        my $ttl
            = ( $self->get_config('xml_rpc_ticket_ttl') )
            ? $self->get_config('xml_rpc_ticket_ttl')
            : $DEF_TTL;
        $logger->debug( 'ttl: ', $ttl );

        # main
        if ($test) {
            $logger->debug('This is just a test!');
        }
        else {
            $logger->debug('This is not a test!');
            $logger->debug( 'old time: ', $time{$login} );
            my $time = time;
            $logger->debug( 'actual time: ', $time{$login} );
            $time{$login} = $time + $ttl;
            $logger->debug( 'new time: ', $time{$login} );
        }

        my $ticket
            = md5_hex( join $EMPTY_STRING, $time{$login}, $SEED, $login );

        $logger->debug( 'created ticket: ', $ticket );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $ticket;

    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  is_ticket_bad                                                       :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub is_ticket_bad : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        #$self->var_dump( { var_r => \$arg_r, name => 'arg_r' } );
        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $answer_hr
            = exists $arg_r->{answer_hr}
            ? $self->h( $arg_r->{answer_hr} )
            : $self->perr('answer_hr');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( 'login: ',     $login );
        $logger->debug( 'ticket: ',    'not printed' );
        $logger->debug( 'answer_hr: ', 'not printed' );

        # +------------------------------------------------------------------+
        # | main
        my $negmsg = 'The ticket check failed! The ticked is not valid.';

        # 2 NO TIME (check if we have a time stamp, if not: noughty noughty)
        if ( not defined $time{$login} ) {

            $logger->debug( 'NO TIME login: ', $login );
            $logger->debug( 'time hash: ',
                { filter => \&Dumper, value => \%time } );
            $logger->debug('NO TIME problem 2 (reset cookies)');

            my $msg = 'No stored time can be found for the ticket.';
            $msg .= ' Probably you logged out before.';
            $msg .= ' Please log in to continue!';
            $answer_hr->{msg} = $negmsg . $msg;
            $answer_hr->{ticket}
                = 'test';    # reset cookie => print login dialog
            $answer_hr->{login} = 'test'; # reset cookie => print login dialog
            $answer_hr->{problem} = 2;        # NO TIME
            $answer_hr->{ltarget} = 'NULL';

            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return { result => 1, answer_hr => $answer_hr };

        }

        my $now = time;
        $logger->debug( 'the time is now: ',    $now );
        $logger->debug( 'ticket valid until: ', $time{$login} );

        # 3 TIME OUT (time out, game over!)
        if ( $time{$login} < $now ) {

            $logger->debug( 'TIME OUT login: ',       $login );
            $logger->debug( 'TIME OUT valid until: ', $time{$login} );
            $logger->debug( 'TIME OUT time now: :',   $now );
            $logger->debug('TIME OUT Ticket is bad? YES!!');
            $logger->debug('TIME OUT problem 3: time out');

            my $msg = ' It was a timeout, because the ticked time is too';
            $msg .= ' old. Please log in again!';
            $answer_hr->{msg}     = $negmsg . $msg;
            $answer_hr->{ticket}  = 'test';
            $answer_hr->{login}   = 'test';
            $answer_hr->{problem} = 3;                # time out
            $answer_hr->{ltarget} = 'NULL';

            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return { result => 1, answer_hr => $answer_hr };
        }

        # recreate ticket to be sure if given ticket is valid
        my $recreated
            = $self->create_ticket( { login => $login, test => 1 } );

        # 4 TIME LOGIN MISMATCH
        if ( $ticket ne $recreated ) {

            my $msg = 'The ticket is not valid. The time and login';
            $msg .= ' do not match. Do you try to use two connections?';
            $msg .= ' To continue, log in only once!';

            $logger->debug( 'TIME LOGIN MISM. login:       ', $login );
            $logger->debug( 'TIME LOGIN MISM. valid until: ', $time{$login} );
            $logger->debug( 'TIME LOGIN MISM. time now:    ', $now );
            $logger->debug( 'TIME LOGIN MISM. old ticket:  ', $ticket );
            $logger->debug( 'TIME LOGIN MISM. test ticket: ', $recreated );
            $logger->debug('Is the ticked bad? YES!! ');
            $logger->debug('problem 4: time login mismatch');

            $answer_hr->{msg}     = $negmsg . $msg;
            $answer_hr->{ticket}  = 'test';           # time login mismatch
            $answer_hr->{login}   = 'test';           # time login mismatch
            $answer_hr->{problem} = 4;                # time login mismatch
            $answer_hr->{ltarget} = 'NULL';

            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return { result => 1, answer_hr => $answer_hr };

        }

        # OK, time and ticket ans SEED match, must be ours!

        $answer_hr->{msg}     = $EMPTY_STRING;
        $answer_hr->{problem} = 0;               # no problem

        $logger->debug('Is the ticket bad? NO!! Ticket is not bad.');
        $logger->debug('problem 0: no problem');

        #$ticket = $self->create_ticket( { login => $login } );
        #$answer_hr->{ticket}  = $ticket;

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return { result => 0, answer_hr => $answer_hr };

    } ## end sub is_ticket_bad

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  calc_random_seed                                                    :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub calc_random_seed : PRIVATE {

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        # Calculating secret random seed for this session
        # "S ISp&FtR0z$EU!We8DvpUzC26D0RE1pVW8vSXp9at5RUwXk
        # WesmQvJY!w!LrLHdo^wB7f6lr7U9PGPTYhxTI!PhKjXhMmZZK
        # ckIi^Qbl&g^$Qir!9S5LIoo!J1bX*OHVw"

        srand;

        my @chars = @PASSWD_CHARS;

        my $seed = join q{},
            @chars[ map { rand @chars } ( 1 .. $TICKET_LENGTH ) ];

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $seed;

    } ## end sub calc_random_seed

    # +----------------------------------------------------------------------+
    # |  PRIVACY CATEGORY 2 (might change to public if needed)               |
    # +----------------------------------------------------------------------+

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  signal_handler                                                      :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub signal_handler : PRIVATE {

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        $time_to_die = 1;

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return;

    } ## end sub signal_handler

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  answer_requests                                                     :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub answer_requests : PRIVATE {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $port
            = exists $arg_r->{port}
            ? $self->l( $arg_r->{port} )
            : $self->perr('port');

        my $address
            = exists $arg_r->{address}
            ? $self->l( $arg_r->{address} )
            : $self->perr('address');

        my $reuse
            = exists $arg_r->{reuse}
            ? $self->l( $arg_r->{reuse} )
            : $self->perr('reuse');

        my $proto
            = exists $arg_r->{proto}
            ? $self->l( $arg_r->{proto} )
            : $self->perr('proto');

        my $meth_hr
            = exists $arg_r->{meth_hr}
            ? $self->h( $arg_r->{meth_hr} )
            : $self->perr('meth_hr');

        my $task_hr
            = exists $arg_r->{task_hr}
            ? $self->h( $arg_r->{task_hr} )
            : $self->perr('task_hr');

        # +------------------------------------------------------------------+
        # | main
        $logger->debug( 'port: ',    $port );
        $logger->debug( 'address: ', $address );
        $logger->debug( 'reuse: ',   $reuse );
        $logger->debug( 'proto: ',   $proto );
        $logger->debug( 'meth_hr: ', $meth_hr );
        $logger->debug( 'task_hr: ', $task_hr );

        my $z = 0;
        $logger->debug( 'try to bind to LocalAddr: ', $address );
        $logger->debug( 'using LocalPort: ',          $port );

        my $daemon = 0;
        while ( not $time_to_die ) {
            sleep $A_SHORT_WHILE;
            my $msg = 'Running - if this value counts up continiously,';
            $msg .= 'the server did not start correctly. Probably ';
            $msg .= 'the port is in use. If you see this message only ';
            $msg .= 'one time everything is OK.';

            $logger->debug( $msg, $z );
            $z++;

            # LocalPort => "https(8001)",
            # LocalAddr => "127.0.0.1:8001",
            $daemon = Frontier::Daemon->new(
                LocalPort => "https($port)",      # 8001
                LocalAddr => "$address:$port",    # localhost:8001
                ReuseAddr => $reuse,              # 1,
                Proto     => $proto,              # tcp
                methods   => $meth_hr             # href to sub refs
            );

            # not implemented for Debian sarge
            # ReusePort => 1,

            # On plain debian there is also no support compiled in:
            # "Your vendor has not defined Socket macro SO_REUSEPORT,
            # used at /usr/lib/perl/5.8/IO/Socket/INET.pm line 160"

            $logger->debug( 'daemon: ', $daemon );
        } ## end while ( not $time_to_die )

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 1;

    } ## end sub answer_requests

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  check_access_to_task                                                :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub check_access_to_task : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("login [$login]");
        $logger->debug("cmd [$cmd]");

        if ( $login eq $EMPTY_STRING ) {
            $logger->debug("NO ACCESS login is empty");
            return 0;
        }

        if ( $cmd eq 'cipux_task_sum' ) {
            return 1;
        }
        elsif ( $rbac->access_to_task( { task => $cmd, user => $login } ) ) {
            $logger->debug('ACCESS');
            return 1;
        }
        else {
            $logger->debug('NO ACCESS');
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 0;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  check_access_to_cat_module                                          :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub check_access_to_cat_module : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("login [$login]");
        $logger->debug("cmd [$cmd]");

        if ( $cmd eq 'index.cgi' ) {
            return 1;
        }
        elsif (
            $rbac->access_to_cat_module(
                { cat_module => $cmd, user => $login }
            )
            )
        {
            $logger->debug('ACCESS');
            return 1;
        }
        else {
            $logger->debug('NO ACCESS');
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 0;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  check_access_to_rpc_intern                                          :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub check_access_to_rpc_intern : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("login [$login]");
        $logger->debug("cmd [$cmd]");

        my $role
            = ( $self->get_config('xml_rpc_intern_admin_group') )
            ? $self->get_config('xml_rpc_intern_admin_group')
            : $INTERN_ADMIN_GROUP;

        $logger->debug("role [$role]");

        if ( none {m/^$cmd$/smx} @RPC_INTERN_CMD ) {
            $logger->debug("rpc_intern command not valid [$cmd]");
        }
        elsif ( ( any {m/^$cmd$/smx} @RPC_INTERN_CMD )
            and
            $rbac->access_to_rpc_intern( { role => $role, user => $login } ) )
        {
            $logger->debug('ACCESS');
            return 1;
        }
        else {
            $logger->debug('NO ACCESS');
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 0;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  error                                                               :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub error : PRIVATE {

        # +------------------------------------------------------------------+
        # | API layer 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $msg = "No access for [$login] to [$cmd]";

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => $cmd,
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $msg,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        $logger->debug( '> answer_hr',
            { filter => \&Dumper, value => $answer_hr } );

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  evaluate_access                                                 :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub evaluate_access : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $realm
            = exists $arg_r->{realm}
            ? $self->l( $arg_r->{realm} )
            : $self->perr('realm');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        my $check_access_disp_hr = {
            'task'       => \&check_access_to_task,
            'cat_module' => \&check_access_to_cat_module,
            'rpc_intern' => \&check_access_to_rpc_intern,
        };
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("login [$login]");
        $logger->debug("realm [$realm]");
        if ( not exists $check_access_disp_hr->{$realm} ) {
            croak "Realm [$realm] do not exists in dispatch table!\n";
        }

        # 'entity' => 'task',
        # 'to'     =>  'cipadmin',
        # 'to_ar' => [
        #            'cipux_task_list_user_accounts'
        #          ],
        # 'subcmd' => 'user_task_access',
        # 'rcpmode' => 'rpc_intern',
        # 'scope' => 'single'

        $logger->debug("login [$login]");

        my $cmdres_r = {};

        my $subcmd
            = ( exists $param_hr->{subcmd} )
            ? $param_hr->{subcmd}
            : return $cmdres_r;

        # we do not trust the "from" param.
        my $from
            = ( $subcmd eq 'task_access' or $subcmd eq 'task_access_survey' )
            ? $login
            : (    $subcmd eq 'user_task_access'
                or $subcmd eq 'user_task_access_survey'
                or $subcmd eq 'user_cat_module_access'
                or $subcmd eq 'user_cat_module_access_survey'
                or $subcmd eq 'user_rpc_intern_access'
                or $subcmd eq 'user_rpc_intern_access_survey'
                and exists $param_hr->{from} ) ? $param_hr->{from}
            : $login;

        $logger->debug("from [$from]");

        my $to_ar = exists $param_hr->{to_ar} ? $param_hr->{to_ar} : [];

        my %access = ();
        foreach my $to ( @{$to_ar} ) {
            $logger->debug("examine to [$to]");
            my $disp = $check_access_disp_hr->{$realm};
            my $r = $self->$disp( { login => $from, cmd => $to } );
            $logger->debug("access result [$to] for [$from]: [$r]");
            $access{$to} = $r;
        }

        $cmdres_r = { from => $from, access_hr => \%access, };

        # +------------------------------------------------------------------+
        # | API
        return $cmdres_r;

    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  update_task                                                         :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub update_task : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my %task     = ();
        my $return_r = $cipux_task->task(
            {
                script  => __PACKAGE__,
                task    => 'cipux_task_list_tasks',
                mode    => 'rpc',
                object  => undef,
                attr_hr => {},
            }
        );

        # $return_r = {
        #     'taskres_r' => {
        #         '<NAME>' => {
        #             'cn' => [
        #                 '<NAME>'
        #             ]
        #         }
        #     },
        #     'status' => 'OK',
        #     'type' => 'href'
        # };
        if ( exists $return_r->{status} and $return_r->{status} eq 'OK' ) {
            my @task = keys %{ $return_r->{taskres_r} };
            %task = map { $_ => 1 } @task;
        }
        else {
            confess 'Can not get tasks by cipux_task_list_tasks!';
        }

        foreach my $task ( sort @{$cipux_task_list_ar} ) {
            if ( $task{$task} ) {
                $logger->debug("task exists [$task]");
            }
            else {
                $logger->debug("task [$task] do not exist");

                my $return_r = $cipux_task->task(
                    {
                        script  => __PACKAGE__,
                        task    => 'cipux_task_register_task',
                        mode    => 'rpc',
                        object  => $task,
                        attr_hr => {},
                    }
                );
                if (
                    not( exists $return_r->{status}
                        and $return_r->{status} eq 'OK' )
                    )
                {
                    confess "failure cipux_task_register_task [$task]!";
                }

            }
        }

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    # :  update_cat                                                          :
    # +~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
    sub update_cat_module : PRIVATE {

        # +------------------------------------------------------------------
        # | API
        my ( $self, $arg_r ) = @_;

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my %cat      = ();
        my $return_r = $cipux_task->task(
            {
                script  => __PACKAGE__,
                task    => 'cipux_task_list_cat_modules',
                mode    => 'rpc',
                object  => undef,
                attr_hr => {},
            }
        );

        # $return_r = {
        #     'taskres_r' => {
        #         '<NAME>' => {
        #             'cn' => [
        #                 '<NAME>'
        #             ]
        #         }
        #     },
        #     'status' => 'OK',
        #     'type' => 'href'
        # };
        if ( exists $return_r->{status} and $return_r->{status} eq 'OK' ) {
            my @cat = keys %{ $return_r->{taskres_r} };
            %cat = map { $_ => 1 } @cat;
        }
        else {
            confess 'Can not get modules by cipux_task_list_cat_modules!';
        }

        foreach my $mod (@ADMIN_GROUP_CLIENT) {
            if ( $cat{$mod} ) {
                $logger->debug("task exists [$mod]");
            }
            else {
                $logger->debug("task [$mod] do not exist");

                # default 'admin' (could be other)
                my $role
                    = ( $self->get_config('xml_rpc_intern_admin_group') )
                    ? $self->get_config('xml_rpc_intern_admin_group')
                    : $INTERN_ADMIN_GROUP;

                my $attr_hr = {
                    cipuxName             => $mod,
                    cipuxTemplate         => 'none',
                    cipuxTemplateDir      => 'none',
                    cipuxEntity           => 'none',
                    cipuxModality         => 'none',
                    cipuxIsModuleArray    => 'FALSE',
                    cipuxAuthor           => 'same as CipUX',
                    cipuxLicense          => 'same as CipUX',
                    cipuxYear             => 'same as CipUX',
                    cipuxScript           => $mod,
                    cipuxIcon             => 'shell.png',
                    cipuxShortDescription => 'CAT on the command line',
                    cipuxIsEnabled        => 'yes',
                    cipuxMemberRid        => $role,
                };

                my $return_r = $cipux_task->task(
                    {
                        script  => __PACKAGE__,
                        task    => 'cipux_task_register_cat_module',
                        mode    => 'rpc',
                        object  => $mod,
                        attr_hr => {},
                    }
                );
                if (
                    not( exists $return_r->{status}
                        and $return_r->{status} eq 'OK' )
                    )
                {
                    confess "failure cipux_task_register_cat_module [$mod]!";
                }

            }
        }

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +======================================================================+
    # || PUBLIC METHODS                                                     ||
    # +======================================================================+

    # +----------------------------------------------------------------------+
    # |  get_config                                                          |
    # +----------------------------------------------------------------------+
    sub get_config {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $key ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("asked for configuration key [$key]");

        # +------------------------------------------------------------------+
        # | main

        if ( not defined $rpc_cfg_hr ) {
            $logger->debug('get_rpc_cfg');
            $rpc_cfg_hr = $rpc->get_rpc_cfg();
        }

        my $return = $rpc_cfg_hr->{$key};

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $return;

    } ## end sub get_config
     # +---------------------------------------------------------------------+
     # |  rpc_list_functions                                                 |
     # +---------------------------------------------------------------------+

    sub rpc_list_functions {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');    # all |

        # +------------------------------------------------------------------+
        # | main
        $logger->debug( 'scope: ', $scope );

        my @cmd = @{$cipux_task_list_ar};

        # add generic RPC commands:
        push @cmd, qw( ping version sum login logout session ttl);
        push @cmd, qw( task rpc_info rpc_intern );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return @cmd;

    }

    # +----------------------------------------------------------------------+
    # |  rpc_start                                                           |
    # +----------------------------------------------------------------------+
    sub rpc_start {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        # +------------------------------------------------------------------+
        # | main
        my @obj = qw(/var/cache/cipux/cipux-access
            /var/cache/cipux/cipux-object
            /var/cache/cipux/cipux-rbac
            /var/cache/cipux/cipux-rpc
            /var/cache/cipux/cipux-storage
            /var/cache/cipux/cipux-task);
        my $l = unlink @obj;

        $self->create_cache_dir_if_not_present();

        my $msg             = 'start server with address';
        my $xml_rpc_address = $self->get_config('xml_rpc_address');
        $logger->debug( $msg . q{: }, $xml_rpc_address );

        if (
            not(   $xml_rpc_address eq 'localhost'
                or $xml_rpc_address eq '127.0.0.1' )
            )
        {
            my $msg = 'xml_rpc_address other then localhost or 127.0.0.1 ';
            $msg .= 'are not supported. ';
            $msg .= 'Please choose a different value in configuration file. ';
            $msg .= 'The value which was given was';
            $self->exc( { msg => $msg, value => $xml_rpc_address } );
        }

        $msg = 'start server with port';
        my $xml_rpc_port = $self->get_config('xml_rpc_port');
        $logger->debug( $msg . q{: }, $xml_rpc_port );

        $msg = 'start server with ticket ttl';
        my $xml_rpc_ticket_ttl = $self->get_config('xml_rpc_ticket_ttl');
        $logger->debug( $msg . q{: }, $xml_rpc_ticket_ttl );

        if ( $xml_rpc_ticket_ttl <= 0 ) {
            my $msg = 'xml_rpc_ticket_ttl is less or equal then 0';
            $msg .= 'Please choose a different value in configuration file.';
            $self->exc( { msg => $msg, value => $xml_rpc_address } );
        }

        $msg = 'start server with use_pam';
        my $xml_rpc_use_pam = $self->get_config('xml_rpc_use_pam');
        $logger->debug( $msg . q{: }, $xml_rpc_use_pam );

        $msg = 'start server with pid_file';
        my $xml_rpc_pid_file = $self->get_config('xml_rpc_pid_file');
        $logger->debug( $msg . q{: }, $xml_rpc_pid_file );

        $msg = 'start server with proto';
        my $xml_rpc_proto = $self->get_config('xml_rpc_proto');
        $logger->debug( $msg . q{: }, $xml_rpc_proto );

        $msg = 'start server with reuse';
        my $xml_rpc_reuse = $self->get_config('xml_rpc_reuse');
        $logger->debug( $msg . q{: }, $xml_rpc_reuse );

        #  $msg = 'start server with TTL';
        #  my $xml_rpc_ticket_ttl = $self->get_config('xml_rpc_ticket_ttl');
        #  $logger->debug($msg.q{: }, $xml_rpc_ticket_ttl  ,"\n");

        # as everybody, who got killed, we have to sleep 3 seconds
        # to regenerate our self to be awake again ;-)
        sleep $A_SHORT_WHILE;

        # check all given tasks.
        $self->update_task();

        # check RPC own CAT modules.
        $self->update_cat_module();

        my @list = $self->rpc_list_functions( { scope => 'all' } );
        my $task_hr = {};
        foreach my $task (@list) {
            $task_hr->{$task} = 1;
        }

        # methode_hr
        my $meth_hr = {
            ping    => sub { $self->ping(@_); },
            version => sub { $self->version(@_); },
            sum     => sub { $self->sum(@_); },
            login   => sub { $self->login(@_); },
            logout  => sub { $self->logout(@_); },
            session => sub { $self->session(@_); },
            ttl     => sub { $self->ttl(@_); },
            task    => sub {

                #my $logger = get_logger(__PACKAGE__);
                #$logger->debug('BEGIN');
                #foreach my $x (@_) {
                #    $logger->debug("X [$x]");
                #    foreach my $h ( keys %{$x} ) {
                #        $logger->debug("h [$h] -> [$x->{$h}]");
                #    }
                #}
                #$logger->debug('END');
                if ( $self->check_access_to_task(@_) ) {
                    $self->rpc_task(@_);
                }
                else { $self->error(@_); }
            },
            rpc_info   => sub { $self->rpc_info(@_); },
            rpc_intern => sub {
                if ( $self->check_access_to_rpc_intern(@_) ) {
                    $self->rpc_intern(@_);
                }
                else { $self->error(@_); }
            },
        };

        $self->answer_requests(
            {
                address => $xml_rpc_address,
                port    => $xml_rpc_port,
                reuse   => $xml_rpc_reuse,
                proto   => $xml_rpc_proto,
                meth_hr => $meth_hr,
                task_hr => $task_hr,
            }
        );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 1;

    } ## end sub rpc_start

    # +======================================================================+
    # || XML-RPC METHODS (exported via XML-RPC server)                      ||
    # +======================================================================+

    # +----------------------------------------------------------------------+
    # | ping                                                                 |
    # +----------------------------------------------------------------------+
    sub ping : PRIVATE {

        # +------------------------------------------------------------------+
        # | API layer 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'ping',
            login     => $login,
            ticket    => $ticket,
            status    => 'TRUE',          # use this to test if available
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub ping

    # +----------------------------------------------------------------------+
    # |  version                                                             |
    # +----------------------------------------------------------------------+
    sub version : PRIVATE {

        # +------------------------------------------------------------------+
        # | API layer 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        my $cmdres_r = {
            cipux_version  => '3.4.0.0',
            server_version => '3.4.0.0',
            rpc_version    => '2.0',
        };

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'version',
            login     => $login,
            ticket    => $ticket,
            status    => 'TRUE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => $cmdres_r,
            ltarget   => 'NULL',
        };

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub version

    # +----------------------------------------------------------------------+
    # |  sum  (alias sum1)                                                   |
    # +----------------------------------------------------------------------+
    sub sum : PRIVATE {

        # +------------------------------------------------------------------+
        # | API level 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        my $summand1
            = exists $param_hr->{summand1}
            ? $self->l( $param_hr->{summand1} )
            : $self->perr('summand1 in param_hr not arg_r');

        my $summand2
            = exists $param_hr->{summand2}
            ? $self->l( $param_hr->{summand2} )
            : $self->perr('summand2 in param_hr not arg_r');

        # +------------------------------------------------------------------+
        # | main
        $logger->debug( 'arg_r: ', { filter => \&Dumper, value => $arg_r } );

        # main
        my $sum      = $summand1 + $summand2;
        my $cmdres_r = [$sum];

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'sum',
            login     => $login,
            ticket    => $ticket,
            status    => 'TRUE',
            msg       => $EMPTY_STRING,
            type      => 'aref',
            cmdres_r  => $cmdres_r,
            ltarget   => 'NULL',
        };

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub sum

    # +----------------------------------------------------------------------+
    # |  login                                                               |
    # +----------------------------------------------------------------------+
    sub login : PRIVATE {

        # +------------------------------------------------------------------+
        # | API layer 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        my $password
            = exists $param_hr->{password}
            ? $self->l( $param_hr->{password} )
            : $self->perr('password');

        # +------------------------------------------------------------------+
        # | main

        # +------------------------------------------------------------------+
        # | debug
        my $msg = 'we got login';
        $logger->debug( $msg . q{: }, $login );
        $msg = 'given password was defined - no print here -';
        $logger->debug($msg);

        $msg = 'start use login with ticket ttl';
        my $xml_rpc_ticket_ttl = $self->get_config('xml_rpc_ticket_ttl');
        $logger->debug( $msg . q{: }, $xml_rpc_ticket_ttl );

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'login',
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        if ( $password eq $EMPTY_STRING ) {
            my $msg = 'empty password given.';
            $logger->debug($msg);

            $logger->debug('END');

           # +---------------------------------------------------------------+
           # | API
            return $answer_hr;
        }

        $msg = 'password was given for login';
        $logger->debug( $msg . q{: }, $login );

        # check if login is authenticated
        my $is_authenticated = $self->check_authentication(
            { login => $login, password => $password } );

        if ($is_authenticated) {
            $logger->debug('password is valid for login');
            $logger->debug("login is_authenticated:  $login");

            #$logger->debug("old login ticket: [$ticket]");
            $ticket = $self->create_ticket( { login => $login } );

            #$logger->debug("new login ticket: [$ticket]");
            $logger->debug('got ticket. not printed here.');

            # differnt answer
            $answer_hr->{status}   = 'TRUE';
            $answer_hr->{type}     = 'href';
            $answer_hr->{cmdres_r} = {
                login  => $login,
                ticket => $ticket,
                ttl    => $xml_rpc_ticket_ttl,
            };
            $answer_hr->{ltarget} = 'NULL';

            $logger->debug('END');

          # +----------------------------------------------------------------+
          # | API
            return $answer_hr;
        }
        else {
            my $msg = 'WARN password is valid NOT for login login ';
            $logger->debug( $msg . q{: }, $login );

            $logger->debug('END');

          # +----------------------------------------------------------------+
          # | API
            return $answer_hr;
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub login

    # +----------------------------------------------------------------------+
    # |  logout                                                              |
    # +----------------------------------------------------------------------+
    sub logout : PRIVATE {

        # +------------------------------------------------------------------+
        # | API level 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'logout',
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        # check the ticket
        my $bad_hr = $self->is_ticket_bad(
            {
                login     => $login,
                ticket    => $ticket,
                time      => $time{$login},
                answer_hr => $answer_hr
            }
        );

        # +------------------------------------------------------------------+
        # | API
        if ( $bad_hr->{result} ) {
            $logger->debug('bad result');
            $logger->debug('END');
            return $bad_hr->{answer_hr};
        }

        # main
        undef $time{$login};

        # answer
        $answer_hr->{status}   = 'TRUE';
        $answer_hr->{type}     = 'href';
        $answer_hr->{cmdres_r} = {};

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub logout

    # +----------------------------------------------------------------------+
    # |  session                                                             |
    # +----------------------------------------------------------------------+
    sub session : PRIVATE {

        # +------------------------------------------------------------------+
        # | API level 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'session',
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        # check the ticket
        my $bad_hr = $self->is_ticket_bad(
            {
                login     => $login,
                ticket    => $ticket,
                'time'    => $time{$login},
                answer_hr => $answer_hr
            }
        );

        if ( $bad_hr->{result} ) {
            $logger->debug('bad result');
            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return $bad_hr->{answer_hr};
        }

        # main (renew_ticket)
        $ticket = $self->create_ticket( { login => $login } );

        # answer
        $answer_hr->{ticket}   = $ticket;
        $answer_hr->{status}   = 'TRUE';
        $answer_hr->{type}     = 'href';
        $answer_hr->{cmdres_r} = {};

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub session

    # +----------------------------------------------------------------------+
    # |  ttl                                                                 |
    # +----------------------------------------------------------------------+
    sub ttl : PRIVATE {

        # +------------------------------------------------------------------+
        # | API level 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'ttl',
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        # check the ticket
        $logger->debug( 'stored time is: ', $time{$login} );
        my $bad_hr = $self->is_ticket_bad(
            {
                login     => $login,
                ticket    => $ticket,
                time      => $time{$login},
                answer_hr => $answer_hr
            }
        );
        $logger->debug( 'stored time is: ', $time{$login} );

        if ( $bad_hr->{result} ) {
            $logger->debug('bad result');
            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return $bad_hr->{answer_hr};
        }
        $logger->debug( 'stored time is: ', $time{$login} );

        # renew_ticket
        #$ticket = $self->create_ticket( { login => $login } );

        # main
        my $ttl = $self->get_config('xml_rpc_ticket_ttl');

        # answer
        $answer_hr->{login}    = $login;
        $answer_hr->{ticket}   = $ticket;
        $answer_hr->{status}   = 'TRUE';
        $answer_hr->{type}     = 'href';
        $answer_hr->{cmdres_r} = { ttl => $ttl };

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub ttl

    # +----------------------------------------------------------------------+
    # |  rpc_task (uses CipUX::Task)                                         |
    # +----------------------------------------------------------------------+
    sub rpc_task : PRIVATE {

        # +------------------------------------------------------------------+
        # | API layer 1
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # API layer 2
        my $object = $self->l( $param_hr->{object} ) || $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | main
        $logger->debug("cmd [$cmd]");

        # collect valid tasks
        my @task_list = $self->rpc_list_functions( { scope => 'all' } );
        my %valid_task = map { $_ => 1 } @task_list;
        $logger->debug( 'param_hr,: ',
            { filter => \&Dumper, value => $param_hr } );

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => $cmd,
            login     => $login,
            ticket    => $ticket,
            status    => 'FALSE',
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => {},
            ltarget   => 'NULL',
        };

        # check if the task is valid, if not return FALSE
        if ( not $valid_task{$cmd} ) {
            my $msg = "Not a valid task $cmd.";
            $answer_hr->{msg} = $cmd;

            $logger->debug($cmd);
            $logger->debug('END');

            # +--------------------------------------------------------------+
            # | API
            return $answer_hr;
        }

        # you have to be logged in only for non test tasks
        if ( not $cmd eq 'cipux_task_sum' ) {

            # check the ticket
            my $bad_hr = $self->is_ticket_bad(
                {
                    login     => $login,
                    ticket    => $ticket,
                    'time'    => $time{$login},
                    answer_hr => $answer_hr
                }
            );

            if ( $bad_hr->{result} ) {
                $logger->debug('bad result');
                $logger->debug('END');

                # +----------------------------------------------------------+
                # | API
                return $bad_hr->{answer_hr};
            }

            $logger->debug("create new ticket for $login");

            # renew_ticket
            $ticket = $self->create_ticket( { login => $login } );

        }

        if ( $cmd eq 'cipux_task_change_own_password_interactive' ) {

            my $password
                = ( exists $param_hr->{userPassword} )
                ? $param_hr->{userPassword}
                : ( exists $param_hr->{value} ) ? $param_hr->{value}
                :                                 $self->random_password();

            $param_hr = {
                object       => $login,
                userPassword => $password,
            };

        }
        else {

            # do work
            foreach my $attr ( sort keys %{$param_hr} ) {
                if ( ref $param_hr->{$attr} ne 'ARRAY' ) {
                    $param_hr->{$attr} = [ $param_hr->{$attr} ];
                }
            }

        }

        my $return_r = $cipux_task->task(
            {
                script  => $SCRIPT,
                task    => $cmd,
                mode    => 'rpc',
                object  => $object,
                attr_hr => $param_hr,
            }
        );

        # main
        $answer_hr->{login}  = $login;
        $answer_hr->{ticket} = $ticket;
        if ( $return_r->{status} eq 'OK' ) {
            $answer_hr->{status}   = 'TRUE';
            $answer_hr->{type}     = ref $return_r->{taskres_r};
            $answer_hr->{cmdres_r} = $return_r->{taskres_r};
            $answer_hr->{ltarget}  = $return_r->{ltarget};
        }

        if ( ref $answer_hr eq 'HASH' ) {
            $logger->debug('got a answer_hr hash.');
            $logger->debug( 'storage answer_hr ',
                { filter => \&Dumper, value => $answer_hr } );
        }
        else {
            $logger->debug('got something else (not a hash) as answer_hr');
        }
        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    } ## end sub rpc_task

    # +----------------------------------------------------------------------+
    # |  rpc_info                                                            |
    # +----------------------------------------------------------------------+
    sub rpc_info : PRIVATE {

        #        $arg_r => {
        #           'param_hr' => {
        #                        'source' => 'cipadmin',
        #                        'destination' => 'index.cgi',
        #                        'rbac' => 'access',
        #                        'ticket' => 'test',
        #                        'login' => 'cipadmin'
        #                      },
        #          'cmd' => 'cipux_rbac',
        #                    'header_hr' => {
        #                         'cipux_version' => '3.4.0.0',
        #                         'client_key' => '',
        #                         'gmt_time' => '1222543779',
        #                         'client_cred' => '',
        #                         'client_name' => 'cipux_cat_web',
        #                         'client_version' => '3.4.0.0',
        #                         'rpc_version' => '2.0'
        #                       },
        #          'ticket' => 'dummy',
        #          'login' => 'dummy'
        #   }

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);

        $logger->debug('BEGIN');
        $logger->debug("> login:     $login");
        $logger->debug('> ticket:    (not printed');
        $logger->debug('> header_hr: (not printed)');
        $logger->debug( '> param_hr',
            { filter => \&Dumper, value => $param_hr } );

        # check_access_task()
        # check_access_task_survey()

        # valid_cat_module_for_user(user, cat_module_list)
        # evaluate_access(cat_module_list) # nicht?  (user, cat_module_list)
        # check_access_to_task(task)              # nicht? (user,task)
        # check_access_to_cat_module(cat_module)  #(user,cat_module)
        # check_access_to_rpc_intern()            # (user)

        my $subcmd
            = ( exists $param_hr->{subcmd} )
            ? $param_hr->{subcmd}
            : $EMPTY_STRING;
        $logger->debug("subcmd [$subcmd]");

        my $cmdres_r = {};

        # rpc_info   | rpc_intern
        my $rpcmode
            = ( exists $param_hr->{rpcmode} )
            ? $param_hr->{rpcmode}
            : $EMPTY_STRING;
        $logger->debug("rpcmode [$rpcmode]");

        # task   | rpc_intern | cat_module
        my $entity
            = ( exists $param_hr->{entity} )
            ? $param_hr->{entity}
            : $EMPTY_STRING;
        $logger->debug("entity [$entity]");

        # single | manifold
        my $scope
            = ( exists $param_hr->{scope} )
            ? $param_hr->{scope}
            : $EMPTY_STRING;
        $logger->debug("scope [$scope]");

        my $status = 'FALSE';

        # -------------------- check_access_to_task ------------------
        if ( $subcmd eq 'task_access' or $subcmd eq 'task_access_survey' ) {
            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'task',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';
        }

        # -------------------- check_access_to_cat_module ------------
        if (   $subcmd eq 'cat_module_access'
            or $subcmd eq 'cat_module_access_survey' )
        {
            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'cat_module',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';
        }

        # -------------------- check_access_to_rpc_intern ------------
        if (   $subcmd eq 'rpc_intern_access'
            or $subcmd eq 'rpc_intern_access_survey' )
        {
            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'rpc_intern',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';
        }

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'rpc_info',
            login     => $login,
            ticket    => $ticket,
            status    => $status,
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => $cmdres_r,
            ltarget   => 'NULL',
        };

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;

    }

    # +----------------------------------------------------------------------+
    # |  rpc_intern                                                          |
    # +----------------------------------------------------------------------+
    sub rpc_intern : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $header_hr
            = exists $arg_r->{header_hr}
            ? $self->h( $arg_r->{header_hr} )
            : $self->perr('header_hr');

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);

        $logger->debug('BEGIN');
        $logger->debug("> login:     $login");
        $logger->debug('> ticket:    (not printed');
        $logger->debug('> header_hr: (not printed)');
        $logger->debug( '> param_hr',
            { filter => \&Dumper, value => $param_hr } );

        my $status = 'FALSE';

        my $subcmd
            = ( exists $param_hr->{subcmd} )
            ? $self->l( $param_hr->{subcmd} )
            : $EMPTY_STRING;

        $logger->debug("subcmd [$subcmd]");

        my $cmdres_r = {};

        if ( $subcmd eq 'flush' ) {
            $cmdres_r = { refresh => 1, };
            $status = 'TRUE';
            $logger->debug('flush cache triggered');

            #$self->rbac_graph_cache( { force => 1 } );
            $rbac->flush();
            $status = 'TRUE';

        }
        if ( $subcmd eq 'cat_module_cache_size' ) {
            $logger->debug('cat_module_cache_size triggered');
            my $size = $rbac->cat_module_cache_size();
            $logger->debug("cat_module cache size [$size]");
            $cmdres_r = { cat_module_cache_size => $size, };
            $status = 'TRUE';
        }
        if ( $subcmd eq 'rpc_intern_cache_size' ) {
            $logger->debug('rpc_intern_cache_size triggered');
            my $size = $rbac->rpc_intern_cache_size();
            $logger->debug("rpc_intern cache size [$size]");
            $cmdres_r = { rpc_intern_cache_size => $size, };
            $status = 'TRUE';
        }
        if ( $subcmd eq 'task_cache_size' ) {
            $logger->debug('task_cache_size triggered');
            my $size = $rbac->task_cache_size();
            $logger->debug("task cache size [$size]");
            $cmdres_r = { task_cache_size => $size, };
            $status = 'TRUE';
        }

        # ----------------- evaluate_access for task ---------------
        if (   $subcmd eq 'user_task_access'
            or $subcmd eq 'user_task_access_survey' )
        {

            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'task',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';

        }

        # ----------------- evaluate_access for cat_module ---------
        if (   $subcmd eq 'user_cat_module_access'
            or $subcmd eq 'user_cat_module_access_survey' )
        {

            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'cat_module',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';

        }

        # ----------------- evaluate_access for rpc_intern ---------
        if (   $subcmd eq 'user_rpc_intern_access'
            or $subcmd eq 'user_rpc_intern_access_survey' )
        {

            $cmdres_r = $self->evaluate_access(
                {
                    realm    => 'rpc_intern',
                    login    => $login,
                    param_hr => $param_hr
                }
            );
            $status = 'TRUE';

        }

        # answer
        my $answer_hr = {
            header_hr => $HEADER_HREF,
            cmd       => 'rpc_intern',
            login     => $login,
            ticket    => $ticket,
            status    => $status,
            msg       => $EMPTY_STRING,
            type      => 'href',
            cmdres_r  => $cmdres_r,
            ltarget   => 'NULL',
        };

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $answer_hr;
    }

}    # END INSIDE-OUT CLASS

1;

__END__

=pod

=head1 NAME

CipUX::RPC::Server - RPC server class for CipUX

=head1 VERSION

version 3.4.0.6

=head1 SYNOPSIS

  use CipUX::RPC::Server;

=head1 DESCRIPTION

Provides the functions for CipUX RPC server.

=head1 ABSTRACT

The CipUX rpc server is a generic abstract class, which can be used by other
classes or scripts.

=head1 SUBROUTINES/METHODS

The following functions are implemented or supported by CipUX::RPC::Server.

=head2 BUILD

Constructor, see new.

=head2 DEMOLISH

Destructor.

=head2 new

Constructor

B<Syntax:>

  my $cipux_rpc = CipUX::RPC::Server->new();


=head2 check_authentication

Check weather the login has access or not.

=head2 create_ticket

Create a ticket.

=head2 is_ticket_bad

Return 1 if a ticket is OK otherwise 0;

=head2 calc_random_seed

Calculate a long random number. This is used in the ticket.

=head2 signal_handler

Install signal_handler alias time to die.

B<Syntax:>

 $server->signal_handler({});

=head2 answer_requests

Answer Requests.

B<Syntax:>

 $server->answer_requests({
     port=>8000,
     address=>'localhost',
     reuse=>0,
     proto=>'tcp',
     meth_hr=>TODO
     task_hr=>TODO
 });

=head2 check_access_to_task

Check the access for login to a task.

=head2 check_access_to_rpc_intern

Check the access for login to the rpc_intern section.

=head2 check_access_to_cat_module

Check the access for login to a CAT module.

=head2 error

Construct an error message.

=head2 evaluate_access

Evaluate the access to login or a given user depending on the sub command
(subcmd) of rpc_info or rpc_intern for one or more realms. Known realms are:
task, cat_module, rpc_intern.

=head2 update_task

Check and update tasks entries.

=head2 update_cat_module

Check and update CAT modules entries.

=head2 get_config

Return the value for a given configuration variable.

=head2 rpc_list_functions

List the rpc functions.

=head2 rpc_start

Start the RPC server.

B<Syntax:>

 $server->rpc_start({});


=head2 ping

The function 'ping' is for testing the connection. It requires not to log in
and no arguments. It returns 'OK'.

=head2 version

Return the CipUX version.

=head2 sum

The function 'sum' is for testing the connection. It requires not to log in and
2 arguments. It returns the sum of the arguments as a hash reference with a
single line.

=head2 login

Perform a login.

=head2 logout

Perform a logout.

=head2 session

Check the ticket and if it is valid update and return a new ticket.

=head2 ttl

Return the Time To Live. Default 1200 seconds.

=head2 rpc_task

Execute a CipUX::Task.

=head2 rpc_info

Execute a rpc_info sub-command.

=head3 task_access

Needs parameter: TASK

=head3 task_access_survey

Needs parameter: TASK [TASK] ...

=head3 cat_module_access

Needs parameter: MODULE

=head3 cat_module_access_survey

Needs parameter: MODULE [MODULE] ...

=head3 rpc_intern_access

Needs parameter: COMMAND

=head3 rpc_intern_access_survey

Needs parameter: COMMAND [COMMAND] ...

=head2 rpc_intern

Execute a rpc_intern sub-command.

=head3 ttl

Prints current TTL in seconds

=head3 cat_module_cache_size

Prints current cat module cache size

=head3 rpc_intern_cache_size

Prints current rpc intern cache size

=head3 task_cache_size

Prints current task_cache_size of cache

=head3 user_task_access

Needs parameter: USER TASK

=head3 user_task_access_survey

Needs parameter: $USER TASK [TASK] ...

=head3 user_cat_module_access

Needs parameter: USER MODULE

=head3 user_cat_module_access_survey

Needs parameter: USER MODULE [MODULE] ...

=head3 user_rpc_intern_access

Needs parameter: USER COMMAND

=head3 user_rpc_intern_access_survey

Needs parameter: USER COMMAND [COMMAND] ...

=head3 flush_cache

Flush RPC server RBAC cache


=head1 Public XML-RPC functions.

All the following CipUX::Task methods are public. Public means that they could
be executed remotely. Public means not that everybody can do this remotely.

There are two kinds of public functions:

(1) Functions without authorization

 - login
 - ping
 - sum

(2) Every other function is available only after using 'login' function, with a
uid as first parameter and a valid ticket as second parameter.

Examples (pseudo code):

 - (reference to user list) = cipux_task_list_users( uid, ticket );
 - (true|false) = logout( uid, ticket );
 - (new ticket|false) = session( uid, ticket );

If the uid do not match, or the uid has not the authorization to use the
function, or the group of the uid has not the authorization to use the
function, or the ticket is expired, or the ticket is not valid the request will
not be fulfilled.

In other words: if the uid match and has the right and the role also has the
right and the ticket is valid and is not expired, the request will be executed.

To see real examples have a look at CipUX::RPC::Client client.

=head1 DIAGNOSTICS

TODO

=head1 CONFIGURATION AND ENVIRONMENT

Need no environment variables. But do need a configuration file. For example
cipux-rpc.ini

=head1 DEPENDENCIES

 Authen::Simple::PAM
 Authen::Simple::Password
 Carp
 CipUX::Task
 CipUX::RBAC::Simple
 Class::Std
 Data::Dumper
 Digest::MD5
 English
 Frontier::Daemon
 Frontier::RPC2
 List::MoreUtils
 Log::Log4perl
 POSIX
 Readonly

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX webpage and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2007 - 2009 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut

