/*
 * check.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2007-05-12 17:42:00 +0000 (Sa, 12 Mai 2007) $
 * $LastChangedRevision: 1413 $
 */

#include <config.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "check.h"
#include "execute.h"
#include "frontend.h"
#include "suite.h"

static int check_md5 (const char *target, const char *md5sum, const char *message)
{
  int ret;
  FILE *in;
  char buf[1024];

  log_message (LOG_MESSAGE_INFO_DOWNLOAD_VALIDATE, message);

  snprintf (buf, sizeof (buf), "/usr/bin/md5sum %s\n", target);

  in = popen (buf, "r");
  fgets (buf, sizeof (buf), in);

  ret = pclose (in);
  if (ret)
    return 1;

  if (!strncmp (buf, md5sum, 32))
    return 0;
  return 1;
}

int check_deb (const char *target, di_package *p, const char *message)
{
  return check_md5 (target, p->md5sum, message);
}

int check_packages (const char *target, const char *ext, di_release *rel)
{
  char buf_name[64];
  char buf_file[128];
  di_release_file *item;
  di_rstring key;

  snprintf (buf_name, sizeof (buf_name), "Packages%s", ext);
  snprintf (buf_file, sizeof (buf_file), "main/binary-%s/Packages%s", arch, ext);
  key.string = (char *) buf_file;
  key.size = strlen (buf_file);
  item = di_hash_table_lookup (rel->md5sum, &key);
  if (!item)
    log_text (DI_LOG_LEVEL_ERROR, "Can't find checksum for Packages file");
  return check_md5 (target, item->sum, buf_name);
}

struct check_release
{
  int goodsigs;
  int badsigs;
};

static int check_release_error_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data __attribute__ ((unused)))
{ 
  log_text (DI_LOG_LEVEL_OUTPUT, "%s", buf);
  return 0;
}

const char GNUPGPREFIX[] = "[GNUPG:]";
const char GNUPGBADSIG[] = "[GNUPG:] BADSIG";
const char GNUPGGOODSIG[] = "[GNUPG:] GOODSIG";

static int check_release_status_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data)
{ 
  struct check_release *data = user_data;

  if (strncmp (buf, GNUPGGOODSIG, sizeof GNUPGGOODSIG - 1) == 0)
  {
    const char *b = buf + sizeof GNUPGGOODSIG + 16 + 1;
    data->goodsigs++;
    log_text (DI_LOG_LEVEL_INFO, "Good signature from \"%s\"", b);
  }
  else if (strncmp (buf, GNUPGBADSIG, sizeof GNUPGBADSIG - 1) == 0)
  {
    const char *b = buf + sizeof GNUPGBADSIG + 16 + 1;
    data->badsigs++;
    log_text (DI_LOG_LEVEL_WARNING, "BAD signature from \"%s\"", b);
  }
  else if (strncmp (buf, GNUPGPREFIX, sizeof GNUPGPREFIX - 1) != 0)
    log_text (DI_LOG_LEVEL_OUTPUT, "%s", buf);
  return 0;
}

int check_release (const char *file, const char *sig)
{ 
  char command[4096];
  struct check_release data = { 0, 0 };
  
  log_message (LOG_MESSAGE_INFO_DOWNLOAD_VALIDATE, "Release");

  snprintf (command, sizeof command, "gpgv --status-fd 1 --keyring %s %s %s", keyring, sig, file);
  
  execute_full (command, check_release_status_io_handler, check_release_error_io_handler, &data, NULL, NULL);

  return !data.goodsigs || data.badsigs;
}

