/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/



#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "AudioDeviceLinux.h"
#include "AudioMixerControlLinux.h"

CAudioDeviceLinux::CAudioDeviceLinux(int card_number)
{
   char *snd_card_name = 0, *snd_card_longname = 0;

   m_CardNumber = card_number;
   m_NodeName.sprintf("%d", card_number);
   m_pSndPcm = 0;
   m_pHWParams = 0;
   m_pSWParams = 0;
   m_pControl = 0;

   m_Validated = (snd_card_get_name(card_number, &snd_card_name) == 0);
   m_ShortName = snd_card_name;
   snd_card_get_longname(card_number, &snd_card_longname);
   m_LongName = snd_card_longname;

   qDebug("CAudioDeviceLinux::CAudioDeviceLinux(%d) = '%s', '%s'", card_number, snd_card_name, snd_card_longname);
}

CAudioDeviceLinux::~CAudioDeviceLinux()
{
   delete m_pControl;
}

// private

int CAudioDeviceLinux::GetHWParameters()
{
//   SoundAttributes this_attr;
   int ret;

   if (m_pSndPcm == 0 || m_pHWParams == 0)
     return -ENODEV;

   ret = snd_pcm_hw_params_get_buffer_size_min(m_pHWParams, &m_MinBufferSize);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_get_buffer_size_min() failed (%s)", snd_strerror(ret));
   }
   ret = snd_pcm_hw_params_get_buffer_size_max(m_pHWParams, &m_MaxBufferSize);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_get_buffer_size_max() failed (%s)", snd_strerror(ret));
   }
   ret = snd_pcm_hw_params_get_buffer_size(m_pHWParams, &m_CurrentBufferSize);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_get_buffer_size() failed (%s)", snd_strerror(ret));
   }
   qDebug("HW Parameters: buffer settings: %lu .. %lu .. %lu", m_MinBufferSize, m_MaxBufferSize, m_CurrentBufferSize);

   return 0;
}

int CAudioDeviceLinux::SetHWParameters()
{
   int ret;
   unsigned int channels;
   unsigned int rrate;

   if (m_pSndPcm == 0 || m_pHWParams == 0)
     return -ENOMEM;

   qDebug(">> CAudioDeviceLinux::SetHWParameters()");
   ret = snd_pcm_hw_params_any(m_pSndPcm, m_pHWParams); // load hardware parameters
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_any() failed (%s)", snd_strerror(ret));
     return ret;
   }

   ret = snd_pcm_hw_params_set_access(m_pSndPcm, m_pHWParams, SND_PCM_ACCESS_RW_INTERLEAVED);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_set_access(RW_INTERLEAVED) failed (%d)", ret);
     return ret;
   }

   // set speed, bits, channels, etc.
   channels = m_CurrentSoundAttr.Channels;
   ret = snd_pcm_hw_params_set_channels(m_pSndPcm, m_pHWParams, channels);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_set_channels(%d) failed (%d)", channels, ret);
     return ret;
   }

   switch(m_CurrentSoundAttr.SampleFormat)
   {
     case SoundAttributes::Signed8:    m_Format = SND_PCM_FORMAT_S8;       break;
     case SoundAttributes::Signed16:   m_Format = SND_PCM_FORMAT_S16_LE;   break;
     case SoundAttributes::Signed24:   m_Format = SND_PCM_FORMAT_S24_LE;   break;
     case SoundAttributes::Signed32:   m_Format = SND_PCM_FORMAT_S32_LE;   break;
     case SoundAttributes::Unsigned8:  m_Format = SND_PCM_FORMAT_U8;       break;
     case SoundAttributes::Unsigned16: m_Format = SND_PCM_FORMAT_U16_LE;   break;
     case SoundAttributes::Unsigned24: m_Format = SND_PCM_FORMAT_U24_LE;   break;
     case SoundAttributes::Unsigned32: m_Format = SND_PCM_FORMAT_U32_LE;   break;
     case SoundAttributes::Float:      m_Format = SND_PCM_FORMAT_FLOAT_LE; break;
     default:                          m_Format = SND_PCM_FORMAT_UNKNOWN;  break;
   }
   ret = snd_pcm_hw_params_set_format(m_pSndPcm, m_pHWParams, m_Format);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_set_format(%d) failed (%d)", (int)m_CurrentSoundAttr.SampleFormat, ret);
     return ret;
   }

   rrate = m_CurrentSoundAttr.SampleRate;
   ret = snd_pcm_hw_params_set_rate_near(m_pSndPcm, m_pHWParams, &rrate, 0);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_set_rate_near(%ud) failed (%d)", m_CurrentSoundAttr.SampleRate, ret);
     return ret;
   }
   if (m_CurrentSoundAttr.SampleRate != rrate)
   {
     qWarning("Set rate (%ud) does not match requested rate (%ud)", rrate, m_CurrentSoundAttr.SampleRate);
     // return?
   }

#if 1
   unsigned int buffer_time = 500000;

   ret = snd_pcm_hw_params_set_buffer_time_near(m_pSndPcm, m_pHWParams, &buffer_time, 0);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_set_hw_buffer_time_near() failed (%s)", snd_strerror(ret));
     return ret;
   }
#endif
   ret = snd_pcm_hw_params(m_pSndPcm, m_pHWParams);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params() failed (%d)", ret);
     return ret;
   }
   qDebug("<< CAudioDeviceLinux::SetHWParameters()");
   return 0;
}

void CAudioDeviceLinux::DumpHWParameters()
{
   unsigned int rate = 0, channels = 0;
   int ret = 0;
   snd_pcm_uframes_t buffer_size = 0;

   if (m_pSndPcm == 0)
     return;
   qDebug("AudioDeviceLinux: HW parameters dump of '%s'", m_ShortName.ascii());

   //qDebug("Sample resolution              : %d bits", snd_pcm_hw_params_get_sbits(&hw_params));
   snd_pcm_hw_params_get_rate_min(m_pHWParams, &rate, 0);
   qDebug("  Min. sample rate               : %u", rate);
   snd_pcm_hw_params_get_rate(m_pHWParams, &rate, 0);
   qDebug("  Current sample rate            : %u", rate);
   snd_pcm_hw_params_get_rate_max(m_pHWParams, &rate, 0);
   qDebug("  Max. sample rate               : %u", rate);
   snd_pcm_hw_params_get_buffer_size(m_pHWParams, &buffer_size);
   qDebug("  Buffer size                    : %lu", buffer_size);

   snd_pcm_hw_params_get_channels_min(m_pHWParams, &channels);
   qDebug("  Minimum channels               : %u", channels);
   snd_pcm_hw_params_get_channels_max(m_pHWParams, &channels);
   qDebug("  Maximum channels               : %u", channels);

#if 0
   int f;
   for (f = 0; f <= SND_PCM_FORMAT_LAST; f++)
   {
      if (snd_pcm_hw_params_test_format(m_pSndPcm, m_pHWParams, (snd_pcm_format_t)f))
        qDebug("Format %d supported.", (int)f);
   }
#endif
}

int CAudioDeviceLinux::SetSWParamaters()
{
   int ret;

   if (m_pSWParams == 0)
     return -ENOMEM;

   qDebug(">> CAudioDeviceLinux::SetSWParameters()");
   ret = snd_pcm_sw_params_current(m_pSndPcm, m_pSWParams); // Get the current software parameters
   if (ret < 0)
   {
     qWarning("snd_pcm_sw_params_current() failed (%d)", ret);
     return ret;
   }

   qDebug("<< CAudioDeviceLinux::SetSWParameters()");
   return 0;
}

// private slots

void CAudioDeviceLinux::ResetStream()
{
   int ret;

   if (m_pSndPcm == 0)
     return;
   qDebug("<> CAudioDeviceLinux::ResetStream()");
   m_ResetBusy.lock();
   snd_pcm_drop(m_pSndPcm);
   snd_pcm_drain(m_pSndPcm);
   ret = snd_pcm_prepare(m_pSndPcm);
   m_SamplesWritten = m_SamplesRead = 0;
   if (ret < 0)
     qDebug("CAudioDeviceLinux::ResetStream: snd_pcm_prepare failed.");
   m_ResetBusy.unlock();
}



// protected

void CAudioDeviceLinux::run()
{
   int ret;
   unsigned char *local_buffer = 0;//[32768];
   snd_pcm_uframes_t WantFrames;
   int FrameSize;

   qDebug(">> CAudioDeviceLinux::run()...");
   DumpHWParameters();

   m_SamplesWritten = m_SamplesRead = 0;
   WantFrames = 1024; //2048;
   FrameSize = m_CurrentSoundAttr.BytesPerSample();
   local_buffer = new unsigned char[WantFrames * FrameSize];

   if (GetCaptureCount() > 0)
   {
     CRingBufferWriter writer(&m_CaptureBuffer);

     qDebug("CAudioDeviceLinux::run() Starting capture.");
     while (!m_StopCapture)
     {
        ret = snd_pcm_readi(m_pSndPcm, local_buffer, WantFrames);
        if (ret < 0)
        {
          qWarning("CAudioDeviceLinux::run() snd_pcm_readi() failed (%d)", ret);
          sleep(2);
        }
        else {
          writer.WriteToBuffer(local_buffer, ret * FrameSize);
#if 0
        qDebug("Read %d bytes: 0x%02x 0x%02x 0x%02x 0x%02x", ret, local_buffer[0], local_buffer[1], local_buffer[2], local_buffer[3]);
#endif
        }
     }
     // Wait until our buffer is drained
     qDebug("CAudioDeviceLinux::run() Capture halted. Draining...");
     while (writer.SpaceUsed() > 0)
     {
       msleep(100);
     }
     qDebug("CAudioDeviceLinux::run() Capture buffer drained.");
     //m_CaptureFinished.wakeAll();
   }

   if (GetPlaybackCount() > 0)
   {
     CRingBufferReader reader(&m_PlaybackBuffer);
     int got;

     qDebug("CAudioDeviceLinux::run() Starting playback.");
     connect(&reader, SIGNAL(BufferFlushed()), this, SLOT(ResetStream()));
     while (!m_StopPlayback)
     {
        got = reader.ReadFromBufferTail(local_buffer, FrameSize * WantFrames) / FrameSize;

        if (got == 0)
        {
          msleep(50);
          continue;
        }
        m_ResetBusy.lock(); // just in case...
        ret = snd_pcm_writei(m_pSndPcm, local_buffer, got);
        m_ResetBusy.unlock();
        if (ret < 0)
        {
          qWarning("CAudioDeviceLinux::run() snd_pcm_writei() failed (%d)", ret);
          sleep(2);
        }
        else
        {
          m_SamplesWritten += ret;
        }
     }
     //m_PlaybackFinished.wakeAll();
   }

   delete local_buffer;
   qDebug("<< CAudioDeviceLinux::run()");
}

/**
  \brief Initialize device; open device, store handle

*/
bool CAudioDeviceLinux::Init()
{
   int ret;
   QString hw_name;

   qDebug(">> CAudioDeviceLinux::Init()");
   if (m_pSndPcm != 0)
   {
     qWarning("CAudioDeviceLinux::Init(): SndPcm != 0 !");
     return false;
   }

   hw_name.sprintf("hw:%d", m_CardNumber);
   ret = -1;
   if (GetMode() == Capture)
     ret = snd_pcm_open(&m_pSndPcm, hw_name.ascii(), SND_PCM_STREAM_CAPTURE, 0);
   if (GetMode() == Playback)
     ret = snd_pcm_open(&m_pSndPcm, hw_name.ascii(), SND_PCM_STREAM_PLAYBACK, 0);
   if (ret < 0)
   {
     qWarning("CAudioDeviceLinux: snd_pcm_open failed (%d).", ret);
     return false;
   }

   snd_pcm_hw_params_malloc(&m_pHWParams);
   if (m_pHWParams == 0)
     qWarning("CAudioDeviceLinux: Failed to allocate HW Params.");

   // Reset HW params to get full range of parameters
   ret = snd_pcm_hw_params_any(m_pSndPcm, m_pHWParams);
   if (ret < 0)
   {
     qWarning("snd_pcm_hw_params_any() failed (%s)", snd_strerror(ret));
     return false;
   }
   DumpHWParameters();


   snd_pcm_sw_params_malloc(&m_pSWParams);
   if (m_pSWParams == 0)
     qWarning("CAudioDeviceLinux: Failed to allocate SW Params.");

   m_pControl = new CAudioControlLinux(hw_name);

   qDebug("<< CAudioDeviceLinux::Init()");
   return true;
}


void CAudioDeviceLinux::Exit()
{
   qDebug(">> CAudioDeviceLinux::Exit()");
   if (m_pSndPcm != 0)
   {
     if (snd_pcm_state(m_pSndPcm) >= SND_PCM_STATE_RUNNING)
       snd_pcm_drop(m_pSndPcm);
     snd_pcm_close(m_pSndPcm);
   }
   m_pSndPcm = 0;

   if (m_pHWParams != 0)
     snd_pcm_hw_params_free(m_pHWParams);
   m_pHWParams = 0;
   if (m_pSWParams != 0)
     snd_pcm_sw_params_free(m_pSWParams);
   m_pSWParams = 0;

   delete m_pControl;
   m_pControl = 0;

   qDebug("<< CAudioDeviceLinux::Exit()");
}



int CAudioDeviceLinux::StartCapture()
{
   int ret;

   if (m_pSndPcm == 0)
   {
     qWarning("CAudioDeviceLinux::StartCapture() no device.");
     return -ENODEV;
   }

   qDebug(">> CAudioDeviceLinux::StartCapture()");
   ret = SetHWParameters();
   if (ret < 0)
   {
     return ret;
   }
   GetHWParameters();

   ret = SetSWParamaters();
   ret = snd_pcm_prepare(m_pSndPcm);
   if (ret < 0)
   {
     qWarning("snd_pcm_prepare() failed.");
     return ret;
   }

   m_StopCapture = false;
   start();

   qDebug("<< CAudioDeviceLinux::StartCapture()");
   return 0;
}

void CAudioDeviceLinux::StopCapture()
{
   qDebug(">> CAudioDeviceLinux::StopCapture()");
   if (running()) {
     qDebug("Waiting for CAudioDeviceLinux capture thread to stop...");
     m_StopCapture = true;
     //m_CaptureFinished.wait();
     while (running()) {};
     qDebug("CAudioDeviceLinux capture thread stopped.");
   }
   if (m_pSndPcm != 0 && snd_pcm_state(m_pSndPcm) >= SND_PCM_STATE_SETUP)
   {
     snd_pcm_drop(m_pSndPcm);
   }
   qDebug("<< CAudioDeviceLinux::StopCapture()");
}


int CAudioDeviceLinux::StartPlayback()
{
   int ret;

   if (m_pSndPcm == 0)
     return -ENODEV;

   qDebug(">> CAudioDeviceLinux::StartPlayback()");
   ret = SetHWParameters();
   if (ret < 0)
   {
     qDebug("SetHWParameters failed.");
     return ret;
   }
   GetHWParameters();

   ret = SetSWParamaters();
   ret = snd_pcm_prepare(m_pSndPcm);
   if (ret < 0)
   {
     qWarning("snd_pcm_prepare() failed.");
     return ret;
   }

   m_StopPlayback = false;
   start();

   qDebug("<< CAudioDeviceLinux::StartPlayback()");
   return 0;
}

void CAudioDeviceLinux::StopPlayback()
{
   qDebug(">> CAudioDeviceLinux::StopPlayback()");
   if (running()) {
     qDebug("Waiting for CAudioDeviceLinux playback thread to stop...");
     m_StopPlayback = true;
     //m_PlaybackFinished.wait();
     while (running()) {};
     qDebug("CAudioDeviceLinux playback thread stopped.");
   }
   snd_pcm_drop(m_pSndPcm);
   qDebug("<< CAudioDeviceLinux::StopPlayback()");
}


// public

// calculate playback pointer based on total samples written and what's left in the buffer
unsigned long CAudioDeviceLinux::GetPlaybackPointer() const
{
   snd_pcm_sframes_t DevBufferLength;

   if (m_pSndPcm == 0)
     return 0;

   DevBufferLength = snd_pcm_avail_update(m_pSndPcm);
   if (DevBufferLength < 0)
   {
     qWarning("snd_pcm_avail_update returned %ld", DevBufferLength);
     return 0;
   }
//   qDebug("snd_pcm_avail = %ld", DevBufferLength);
   return m_SamplesWritten - m_CurrentBufferSize + DevBufferLength;
}


void CAudioDeviceLinux::SetBufferLength(unsigned int len, unsigned int chunk_length)
{
   snd_pcm_uframes_t NewBufferSize;

   if (m_pSndPcm == 0)
     return;
   qDebug("CAudioDeviceLinux::SetBufferLength(%d, %d)", len, chunk_length);

   GetHWParameters();

   NewBufferSize = len / m_CurrentSoundAttr.BytesPerSample();
   if (snd_pcm_hw_params_test_buffer_size(m_pSndPcm, m_pHWParams, NewBufferSize) < 0)
   {
     qWarning("CAudioDeviceLinux: New buffer size rejected.");
     return;
   }

}

void CAudioDeviceLinux::SetBufferTime(unsigned int ms, unsigned int chunk_length)
{

}

void CAudioDeviceLinux::GetMixerSettings(QDomNode &dom_node) const
{
   qDebug(">> CAudioDeviceLinux::GetMixerSettings");
   if (m_pControl != 0)
   {
///qDebug("xml before = %s", dom_node.ownerDocument().toString().ascii());
     m_pControl->GetConfiguration(dom_node);
///qDebug("xml after = %s", dom_node.ownerDocument().toString().ascii());
   }
   qDebug("<< CAudioDeviceLinux::GetMixerSettings");
}

void CAudioDeviceLinux::SetMixerSettings(const QDomNode &dom_node) const
{
   qDebug(">> CAudioDeviceLinux::SetMixerSettings");
   if (m_pControl != 0)
   {
     m_pControl->SetConfiguration(dom_node);
   }
   qDebug("<< CAudioDeviceLinux::SetMixerSettings");
}


// public slots

void CAudioDeviceLinux::ShowMixerControls()
{
   if (m_pControl != 0)
     m_pControl->ShowControls();
}
